/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * RLV12 + RL02
 *
 * Arguments:
 *	ctrl csr_base vector irq sync
 *	  disk-no file 
 * 	end
 *
 * sync is the interval between msync()s in milliseconds
 */

# include "proc.h"

# define D(X)


# define RLS	4			/* disks per controller */
# define RLSIZE	(256 * 512 * 2 * 40)	/* rl2 in bytes */

/*
 * rl02 drive status bits
 */
enum {
	Load_state	= 00,
	Spin_up		= 01,
	Brush_cycle	= 02,
	Load_heads	= 03,
	Seek_track	= 04,
	Lock_on		= 05,
	Unload_heads	= 06,
	Spin_down	= 07,
	Brush_home	= 010,
	Heads_out	= 020,
	Cover_open	= 040,
	Head_select	= 0100,
	Drive_rl02	= 0200,
	Select_error	= 0400,
	Volume_check	= 01000,
	WGate_error	= 02000,
	Spin_error	= 04000,
	Seek_time_out	= 010000,
	Write_lock	= 020000,
	Current_error	= 040000,
	Write_error	= 0100000,
};

typedef struct RL RL;

struct RL {
	unsigned csr_base;
	ushort	vector;
	ushort	irq_level;
	int	dls;		/* real number of disks		*/
	int	sync_rate;

	int	command;	/* CSR 1:3 			*/
	int	ien;		/* CSR 6 			*/
	int	crdy;		/* CSR 7			*/
	int	sel;		/* CSR 8:9 			*/
	int	err;		/* CSR 10:15			*/
	int	ba;		/* BA 0:15, CSR 4:5, BAE 0:5	*/

	int	da;		/* multi-interpretable DA	*/
	int	buffer[256];	/* internal buffer		*/
	int	wc;		/* mpr during r/w		*/
	int	*mpr;		/* mpr pointer			*/

	int	pp[RLS];	/* present position (cylinder)	*/
	int	ph[RLS];	/* present head			*/
	int	wl[RLS];	/* write locked			*/
	int	st[RLS];	/* drive status			*/
	char   *mm[RLS];	/* mapped files			*/
	char   *fn[RLS];	/* file name 			*/
};

static int sector;

void	rl_ctrl_create(IODev *, int, char **);
void	rl_dev_create(IODev *, int, char **);
void	rl_ctrl_complete(IODev *);

void	rl_reset(IODev *);
ushort	rl_fetch(IODev *, unsigned);
void	rl_store(IODev *, unsigned, int, ushort);
ushort	rl_vector(IODev *);
void	rl_dma(IODev *);
void	rl_async(IODev *);
void	rl_info(IODev *);
void	rl_command(IODev *, int, char *[]);


static void dofunc(IODev *);
static void rlsync(void *);
static void unload(RL *, int);
static void load(RL *, int, char *, int);


IOOps rl_ops = {
	rl_ctrl_create,		/* ctrl_create		*/
	rl_dev_create,		/* dev_create		*/
	rl_ctrl_complete,	/* ctrl_complete	*/
	rl_reset,		/* reset		*/
	rl_fetch,		/* fetch		*/
	rl_store,		/* store		*/
	rl_vector,		/* vector		*/
	rl_dma,			/* dma			*/
	0,			/* async 		*/
	rl_info,		/* info			*/
	rl_command,		/* command 		*/
};

enum {
	RL_CSR		= 000,
	RL_BA		= 002,
	RL_DA		= 004,
	RL_MP		= 006,
	RL_BAE		= 010,
};

/*
 * initialise controler
 * args:
 *	CSR-base
 *	vector
 *	interrupt request level
 */
void
rl_ctrl_create(IODev *dev, int argc, char **argv)
{
	RL	*d;
	int	i;

	if(argc != 4)
		conf_panic("rl: missing args in controller configuration");

	dev->data = d = xalloc(sizeof(RL), "rl_ctrl_create");

	d->csr_base = parse_csr(argv[0], "rl");
	d->vector = parse_vec(argv[1], "rl");
	d->irq_level = parse_irq(argv[2], "rl");
	d->sync_rate = strtol(argv[3], 0, 10);

	proc.iopage[IOP(d->csr_base + RL_CSR)] = dev;
	proc.iopage[IOP(d->csr_base + RL_BA )] = dev;
	proc.iopage[IOP(d->csr_base + RL_DA )] = dev;
	proc.iopage[IOP(d->csr_base + RL_MP )] = dev;
	proc.iopage[IOP(d->csr_base + RL_BAE)] = dev;

	for(i = 0; i < RLS; i++) {
		d->pp[i] = 0;
		d->ph[i] = 0;
		d->wl[i] = 0;
		d->st[i] = Load_state | Drive_rl02 | Brush_home;
		d->mm[i] = 0;
		d->fn[i] = 0;
	}
	d->dls = 4;
}


/*
 * create drive
 * args:
 *	drive-no
 *	file
 */
void
rl_dev_create(IODev *dev, int argc, char **argv)
{
	RL *d = (RL *)dev->data;
	int	i;

	if(argc != 2)
		conf_panic("rl: bad args int device description");

	i = (int)strtol(argv[0], 0, 0);
	if(i > 4)
		conf_panic("rl: controller supports up to 4 disks only");

	if(d->mm[i])
		unload(d, i);
	load(d, i, argv[1], 1);
}


/*
 * load new disk in drive i
 * should we simulate the spin up, brushes ... ?
 * set write lock, if file is read-only
 */
static void
load(RL *d, int i, char *fn, int isconf)
{
	typedef void (*pfunc)(char *, ...);
	struct stat statb;
	int	fd;
	pfunc ef = isconf ? (pfunc)conf_panic : (pfunc)printf;

	d->wl[i] = (access(fn, W_OK) != 0);
	if((fd = open(fn, d->wl[i] ? O_RDONLY : (O_RDWR | O_CREAT), 0666)) < 0) {
		(*ef)("rl%d: can't open %s: %s", i, fn, strerror(errno));
		return;
	}

	if(fstat(fd, &statb)) {
		(*ef)("rl%d: can't stat %s: %s", i, fn, strerror(errno));
		return;
	}

	if(statb.st_size < RLSIZE) {
		if(d->wl[i]) {
			(*ef)("rl%d: can't expand %s to required size\n", i, fn);
			return;
		}
		lseek(fd, RLSIZE - 1, 0);
		write(fd, "\0", 1);
	} else if(statb.st_size > RLSIZE) {
		if(d->wl[i]) {
			(*ef)("rl%d: can't truncate %s to required size\n", i, fn);
			return;
		}
		ftruncate(fd, RLSIZE);
	}

	d->mm[i] = mmap(0, RLSIZE, PROT_READ | (d->wl[i] ? 0 : PROT_WRITE), MAP_FILE | MAP_SHARED, fd, 0);
	if((long)d->mm[i] == -1) {
		(*ef)("rl%d: can't mmap %s: %s", i, fn, strerror(errno));
		return;
	}

	close(fd);

	d->st[i] = Volume_check | Brush_home | Lock_on | Heads_out | Drive_rl02;

	d->fn[i] = xalloc((strlen(fn) + 1) * sizeof(char), "rl::load");
	strcpy(d->fn[i], fn);

	d->pp[i] = 0;
	d->ph[i] = 0;
}


/*
 * unload disk from drive i
 */
static void
unload(RL *d, int i)
{
	if(!d->mm[i])
		return;
	d->wl[i] = 0;
	d->st[i] = Load_state | Drive_rl02 | Brush_home;
	d->pp[i] = 0;
	d->ph[i] = 0;
	free(d->fn[i]);
	d->fn[i] = 0;
	munmap(d->mm[i], RLSIZE);
	d->mm[i] = 0;
}


void
rl_ctrl_complete(IODev *dev)
{
	RL *d = (RL *)dev->data;

	if(d->dls > 0 && d->sync_rate > 0)
		register_timer(d->sync_rate, rlsync, d);
}


static void
rlsync(void *v)
{
	RL *d = v;
	int i;

	for(i = 0; i < d->dls; i++)
		if(d->mm[i])
			MSYNC(d->mm[i], RLSIZE);
}


void
rl_reset(IODev *dev)
{
	RL *d = (RL *)dev->data;
	int i;

	d->command	= 0;
	d->ien		= 0;
	d->sel		= 0;
	d->err		= 0;
	d->ba		= 0;
	d->crdy 	= 0200;

	d->da		= 0;
	d->wc 		= 0;
	bzero((caddr_t)d->buffer, sizeof(d->buffer));
	d->mpr 		= &d->wc;
}

/*
 * fetch io-register
 */
ushort
rl_fetch(IODev *dev, unsigned a)
{
	RL *d = (RL *)dev->data;
	ushort v;

	switch(a - d->csr_base) {

	case RL_CSR:
		v = ((d->mm[d->sel]) ? 1 : 0)	/* DRDY			*/
		  | d->command			/* FUNC			*/
		  | ((d->ba >> 12) & 060)	/* BA 16:17		*/
		  | d->ien			/* IEN			*/
		  | d->crdy			/* CRDY			*/
		  | (d->sel << 8)		/* DSEL			*/
		  | d->err;			/* ERROR		*/
		break;

	case RL_BA:
		v = d->ba;
		break;

	case RL_DA:
		v = d->da;
		break;

	case RL_MP:
		v = *d->mpr;
		switch(d->command) {

		case 010:	/* read header */
			if(d->mpr < &d->buffer[2])
				d->mpr++;
			break;

		case 04:	/* get status */
			break;
		}
		break;

	case RL_BAE:
		v = d->ba >> 16;
		break;

	default:
		Warning("rl_fetch(%o)", a);
		Trap4(020);
	}

	return v;
}

/*
 * store in io-regs
 */
void
rl_store(IODev *dev, unsigned a, int mode, ushort v)
{
	RL *d = (RL *)dev->data;

	switch(a - d->csr_base) {

	case RL_CSR:
		if(!(mode & M_Low))
			d->sel = (v >> 8) & 3;
		if(!(mode & M_High)) {
			d->ba = (d->ba & ~0600000) | ((v & 060) << 12);
			if(!(v & 0200)) {	/* GO */
				d->crdy = 0;
				d->err = 0;
			}
			if(!d->ien && (v & 0100) && d->crdy)
				IRQ(dev, d->irq_level);
			if(!(d->ien = v & 0100))
				dev->reqpri = 0;
			d->command = v & 016;

			if(!d->crdy) {
				d->err &= 040000;
				if(d->sel >= d->dls)
					d->err |= 02000;
				if(!d->err)
					dofunc(dev);
				if(d->err & 0076000) {
					d->err |= 0100000;
					d->crdy = 0200;
				}
				if(d->ien && d->crdy)
					IRQ(dev, d->irq_level);
			}
		}
		break;

	case RL_BA:		/* has 22 bit! */
		v &= ~1;
		if(!(mode & M_High))
			d->ba = (d->ba & ~0377) | (v & 0377);
		if(!(mode & M_Low))
			d->ba = (d->ba & ~0177400) | (v & 0177400);
		break;

	case RL_DA:
		if(!(mode & M_High))
			SLB(d->da, v);
		if(!(mode & M_Low))
			SHB(d->da, v);
		break;

	case RL_MP:
		d->mpr = &d->wc;
		v |= 0160000;
		if(!(mode & M_High))
			SLB(*d->mpr, v);
		if(!(mode & M_Low))
			SHB(*d->mpr, v);
		break;

	case RL_BAE:
		if(mode & M_High)
			break;
		v &= 077;
		d->ba = (d->ba & 0177777) | (v << 16);
		break;

	default:
		Warning("rl_store(%o)", a);
		Trap4(020);
	}

D(
	if(d->err) {
		printf("rl: error %o\n", d->err);
		proc.halt++;
	}
)
}

/*
 * interrupt ack.
 */
ushort
rl_vector(IODev *dev)
{
	RL 	*d = (RL *)dev->data;

	dev->reqpri = 0;

	return d->vector;
}

/*
 * dma
 * copy operations will be byte aligned and with an even byte count
 * in any case
 * currently assumes that the disk is not unloaded after initiating
 * the command and before doing dma
 */
void
rl_dma(IODev *dev)
{
	RL 	*d = (RL *)dev->data;
	int	cyl = d->pp[d->sel];
	int	hs = d->ph[d->sel];
	int	sec = (d->command == 016) ? (sector % 40) : (d->da & 077);
	int 	track = 2 * cyl + hs;
	int	off = 256 * (track * 40 + sec);
	char	*add = (char *)proc.memops->addrphys(proc.mmg, d->ba);
	char	*end = add - ((int)(short)d->wc << 1);
	char	*endmem = (char *)proc.memops->addrphys(proc.mmg, proc.physmem);
	unsigned bytes = end - add;
	unsigned words;

	if(end > endmem) {
		bytes -= end - endmem;
		d->err |= 020000;
	}

	words = bytes >> 1;

	switch(d->command) {

	case 002:	/* write check */
		break;

	case 012:	/* write data */
		if(words)
			CopyW(d->mm[d->sel] + off, add, words);
		d->ba += bytes;
		break;

	case 014:	/* read data */
		if(words)
			CopyW(add, d->mm[d->sel] + off, words);
		d->ba += bytes;
		break;

	case 016:	/* read data without check */
		if(words)
			CopyW(add, d->mm[d->sel] + off, words);
		d->ba += bytes;
		break;
	}

	d->crdy = 0200;
	dev->reqpri = 0;
	if(d->ien)
		IRQ(dev, d->irq_level);
}

/*
 * doing
 */
static void
dofunc(IODev *dev)
{
	RL *d = (RL *)dev->data;

	D(printf("dofunc: %o %o\n", d->command, d->da));

	if(d->st[d->sel] & Volume_check) {
		d->err |= 040000;
		if(d->command != 04) {
			d->crdy = 0200;
			return;
		}
	}

	if(d->command != 000 && d->command != 04 && !d->mm[d->sel]) {
		d->err |= 02000;
		return;
	}

	switch(d->command) {

	case 000:	/* maintenance (nop) */
		d->crdy = 0200;
		d->err &= 040000;	/* drive error */
		break;

	case 04:	/* get status */
		if(d->da & 010) {
			if(d->mm[d->sel])
				d->st[d->sel] = Brush_home | Lock_on | Heads_out | Drive_rl02;
			else
				d->st[d->sel] = Load_state | Drive_rl02 | Brush_home;
		}
		d->buffer[0] = d->st[d->sel] 
			     | (d->wl[d->sel] << 13)
			     | (d->ph[d->sel] << 6);
		d->mpr = d->buffer;
		d->crdy = 0200;
		break;

	case 06: {	/* seek */
		int diff = d->da >> 7;
		int np = d->pp[d->sel] + ((d->da & 04) ? (diff) : (-diff));

		if(np < 0) {
			d->pp[d->sel] = 0;	/* XXX how about errors ? */
			d->ph[d->sel] = 0;
		} else if(np >= 512) {
			d->pp[d->sel] = 511;	/* XXX how about errors ? */
			d->ph[d->sel] = 0;
		} else {
			d->pp[d->sel] = np;
			d->ph[d->sel] = (d->da & 020) ? 1 : 0;
		} 
		d->crdy = 0200;
		}
		break;

	case 010:	/* read header */
		d->buffer[0] = (d->pp[d->sel] << 7) 
			     | (d->ph[d->sel] << 6)
			     | (sector++ % 40);
		d->buffer[1] = 0;
		d->buffer[2] = 0123456;
		d->mpr = d->buffer;
		d->crdy = 0200;
		break;

	case 012:	/* write data */
		if(d->wl[d->sel]) {
			d->err |= 040000;
			break;
		}
		/* durchfall */

	case 014:	/* read data */
	case 02:	/* write check */
D(printf("r/w: %3ho [%d,%d,%d] ", d->command, (d->da >> 7) & 0777, (d->da >> 6) & 1, d->da & 077); fflush(stdout));
		if(((d->da >> 7) & 0777) != d->pp[d->sel] ||
     		   ((d->da >> 6) & 0001) != d->ph[d->sel] ||
		   ((d->da >> 0) & 0077) >= 40) {
			d->err |= 012000;
D(printf("r/w error: %3ho [%d,%d,%d] ", d->command, (d->da >> 7) & 0777, (d->da >> 6) & 1, d->da & 077); fflush(stdout));
		} else
			IRQ(dev, DMAPRI);
		break;

	case 016:	/* read data without header check */
		IRQ(dev, DMAPRI);
		break;
	}
}

void
rl_info(IODev *dev)
{
	RL *d = (RL *)dev->data;
	int i;

	printf("RLV12 Controller\n");
	printf("CSR at %08o, vector %03o at level %d, %d RL02 disks\n", d->csr_base, 
			d->vector, d->irq_level, d->dls);

	printf("UNIT\tPP\tPH\tWL\tST\tFile\n");

	for(i = 0; i < d->dls; i++)
		printf("%d\t%d\t%d\t%d\t%06o\t%s\n", i, 
			d->pp[i], d->ph[i], d->wl[i], d->st[i], d->fn[i] ? d->fn[i] : "");
}


static char *cmds[] = {
	"load",
	"unload",
	"wlock",
	"wunlock",
	"?",
};

static char help[] = 
"rl: commands are:\n"
"\tload {drive} {file}\n"
"\tunload {drive}\n"
"\twlock {drive}\n"
"\twunlock {drive}\n";

/*
 * execute a monitor command
 * I don't like to include too much error checking into the code
 * so we prevent the user from changing anything if a dma request is pending.
 * If we did not do this someone could just unload a disk that is about to get
 * dma granted and the dma routine had to check if the disk is still there.
 */
void	
rl_command(IODev *dev, int argc, char *argv[])
{
	int cc, f = -1, drive;
	char *end;
	RL *d = (RL *)dev->data;

	if(dev->reqpri == DMAPRI) {
		printf("rl: DMA request pending - try again\n");
		return;
	}

	for(cc = 0; cc < ArraySize(cmds); cc++)
		if(!strncmp(cmds[cc], argv[0], strlen(argv[0]))) {
			if(f >= 0) {
				printf("rl: ambiguous command '%s'\n", argv[0]);
				return;
			}
			f = cc;
		}
	if(f < 0) {
		printf("rl: unknown command '%s'\n", argv[0]);
		return;
	}
	if(f == 4) {
		printf("%s", help);
		return;
	}

	if(argc == 1) {
		printf("rl: missing drive number\n");
		return;
	}

	drive = strtol(argv[1], &end, 0);
	if(*end || drive < 0 || drive >= RLS) {
		printf("rl: bad drive number\n");
		return;
	}

	switch(f) {

	case 0:		/* load */
		if(argc < 3)
			printf("rl: need filename for load\n");
		else if(d->mm[drive])
			printf("rl%d: already loaded and spinning\n", drive);
		else  {
			load(d, drive, argv[2], 0);
			if(d->mm[drive])
				printf("rl%d: loaded\n", drive);
		}
		break;

	case 1:		/* unload */
		if(!d->mm[drive])
			printf("rl%d: no disk loaded\n", drive);
		else {
			unload(d, drive);
			if(!d->mm[drive])
				printf("rl%d: unloaded\n", drive);
		}
		break;

	case 2:		/* wlock */
		if(!d->mm[drive])
			printf("rl%d: not loaded\n", drive);
		else if(d->wl[drive])
			printf("rl%d: already write locked\n", drive);
		else if(mprotect(d->mm[drive], RLSIZE, PROT_READ))
			printf("rl%d: %s\n", drive, strerror(errno));
		else {
			d->wl[drive] = 1;
			printf("rl%d write locked\n", drive);
		}
		break;

	case 3:		/* wunlock */
		if(!d->mm[drive])
			printf("rl%d: not loaded\n", drive);
		else if(!d->wl[drive])
			printf("rl%d: not write locked\n", drive);
		else if(access(d->fn[drive], W_OK))
			printf("rl%d: file is not writeable: %s\n", drive, d->fn[drive]);
		else if(mprotect(d->mm[drive], RLSIZE, PROT_READ | PROT_WRITE))
			printf("rl%d: %s\n", drive, strerror(errno));
		else {
			d->wl[drive] = 0;
			printf("rl%d: write enabled\n", drive);
		}
		break;
	}
}
