/* p11 - pdp11 emulator; Copyright (C) 1994 Hartmut Brandt, Joerg Micheel 
 * see the file LICENSE for further information */

/*
 * generate .s or .c 'microcode dispatch table' from instruction list
 */
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <unistd.h>
# include <errno.h>
# include "../system.h"

# define ArraySize(A)	(sizeof(A)/sizeof(A[0]))
# define BAD_SWITCH	fprintf(stderr, "bad switch in '%s', %d\n", __FILE__, __LINE__); exit(1);

# define MAXNAMES 100000
int	getmfields(char *, char **, int);
void	fill(int, char *);
void	trans(int, int, char **);
void	appfunc(char *);
char	*strerror(int);

int 	code;
int	ccc;
FILE	*tmp;
char	*names[MAXNAMES];
int	nnames;
int	coo = -1;
int	profiler_output;
char	*ul;

char	*cookies[] = {
	"i386-as",
	"sparc-elf-as",
	"sparc-sunos-as",
	"c"
};
# define COO_i386as	0
# define COO_self_as	1
# define COO_sun_as	2
# define COO_c		3

void	do0(int, char **, int);
void	do1(int, char **, int);
void	doQ(int, char **, int);
void	doA(int, char **, int);
void	doS(int, char **, int);
void	doJ(int, char **, int);
void	doK(int, char **, int);
void	doD(int, char **, int);
void	doR(int, char **, int);
void	doF(int, char **, int);
void	doG(int, char **, int);
void	doH(int, char **, int);
void	doI(int, char **, int);
void	doB(int, char **, int);
void	define(char *);
void	do_profiler_output();
void	getul();
int	ncmp(const void *, const void *);

main(int argc, char *argv[])
{
	char	buf[1000];
	char	*fields[20];
	int	base, f, i;
	char	*end;
	int	opt;

	while((opt = getopt(argc, argv, "-mp")) != EOF)
		switch(opt) {

		case 'm':
			profiler_output = 1;
			break;
		case 'p':
			profiler_output = 2;
			break;
	}
	argc -= optind;
	argv += optind;

	if(argc != 2) {
		fprintf(stderr, "Usage: file cookie\n");
		return 1;
	}

	if(strcmp(argv[0], "-") && !freopen(argv[0], "r", stdin)) {
		fprintf(stderr, "%s: %s\n", argv[0], strerror(errno));
		return 1;
	}

	for(i = 0; i < ArraySize(cookies); i++)
		if(!strcmp(argv[1], cookies[i])) {
			coo = i;
			break;
		}
	if(coo < 0) {
		fprintf(stderr, "unknown cookie '%s'\n", argv[1]);
		return 1;
	}

	getul();

	if(!(tmp = fopen("geni.tmp", "w+"))) {
		fprintf(stderr, "cannot create tempfile\n");
		exit(1);
	}

	unlink("geni.tmp");
	while(gets(buf)) {
		f = getmfields(buf, fields, 20);
		if(f == 0 || fields[0][0] == '#')
			continue;
		if(f < 3) {
			fprintf(stderr, "missing fields: %s\n", buf);
			exit(1);
		}
		base = strtol(fields[0], &end, 8);
		if(*end) {
			fprintf(stderr, "syntax: %s\n", buf);
			exit(1);
		}
		if(base < code) {
			fprintf(stderr, "cannot go backward to %o\n", base);
			exit(1);
		}

		fill(base, "ILL");

		switch(fields[1][0]) {
		case '0':	do0(base, fields+2, f-2); break;
		case '1':	do1(base, fields+2, f-2); break;
		case 'G':	doG(base, fields+2, f-2); break;
		case 'Q':	doQ(base, fields+2, f-2); break;
		case 'A':	doA(base, fields+2, f-2); break;
		case 'S':	doS(base, fields+2, f-2); break;
		case 'J':	doJ(base, fields+2, f-2); break;
		case 'K':	doK(base, fields+2, f-2); break;
		case 'D':	doD(base, fields+2, f-2); break;
		case 'R':	doR(base, fields+2, f-2); break;
		case 'F':	doF(base, fields+2, f-2); break;
		case 'H':	doH(base, fields+2, f-2); break;
		case 'I':	doI(base, fields+2, f-2); break;
		case 'B':	doB(base, fields+2, f-2); break;
		default: fprintf(stderr, "bad type key: %s\n", buf); exit(1);
		}
	}
	fill(0200000, "ILL");
	define("ILL");
	define("FILL");
	define("ret");

	if(profiler_output) {
		qsort(names, nnames, sizeof(names[0]), ncmp);
		for(i = 0; i < nnames; i++)
			printf("# define %s%s %d\n", ul, names[i], i);
	}

	switch(coo) {
	
	case COO_i386as:
		printf("\t.file\t\"%s\"\n", argv[0]);
		printf("\t.globl\t_instab\n");
		printf(".text\n");
		printf("\t.align\t2\n");
		printf("_instab:\n");
		break;

	case COO_self_as:
		printf("\t.global\tinstab\n");
		printf(".section\t\".text\"\n");
		printf("\t.align\t4\n");
		printf("\t.type\tinstab,#object\n");
		printf("\t.size\tinstab,%lu\n", (1L << 16) * 4 * 4);
		printf("instab:\n");
		break;

	case COO_sun_as:
		printf("\t.global\t_instab\n");
		printf(".text\n");
		printf("\t.align\t2\n");
		printf("_instab:\n");
		break;

	case COO_c:
		if(profiler_output) {
			printf("int instab[0200000][4] = {\n");
		} else {
			for(i = 0; i < nnames; i++)
				printf("void %s(void);\n", names[i]);
			printf("void (*const instab[0200000][4])(void) = {\n");
		}
		break;

	default:
		BAD_SWITCH;
	}

	rewind(tmp);
	while(fgets(buf, sizeof(buf), tmp))
		if( fputs(buf, stdout) == EOF ) {
			fprintf(stderr,"geni: fputs(stdout) failed: %s\n",
				strerror(errno) );
			exit(1);
		}

	if(coo == COO_c)
		printf("};\n");

	if(profiler_output == 2)
		do_profiler_output();

	return 0;
}

void
start(void)
{
	switch(coo) {
	case COO_i386as:
	case COO_sun_as:
		fprintf(tmp, "\t.long\t");
		break;

	case COO_self_as:
		fprintf(tmp, "\t.word\t");
		break;

	case COO_c:
		fprintf(tmp, "/* %06o */ { ", code);
		break;

	default:
		BAD_SWITCH;
	}
	ccc = 0;
}


void
theend(void)
{
	while(ccc < 4)
		appfunc("ret");

	switch(coo) {
	case COO_i386as:
		fprintf(tmp, "\t# %06o\n", code);
		break;
	case COO_sun_as:
		fprintf(tmp, "\t! %06o\n", code);
		break;

	case COO_self_as:
		fprintf(tmp, "\n");
		break;

	case COO_c:
		fprintf(tmp, " },\n");
		break;

	default:
		BAD_SWITCH;
	}
	code++;
}

void	
fill(int to, char *w)
{
	while(code < to) {
		start();
		appfunc(w);
		theend();
	}
}

/*
 * one code only
 */
void	do0(int base, char **fields, int f)
{
	start();
	appfunc(fields[0]);
	define(fields[0]);
	theend();
}

/*
 * dito, but do translation
 */
void	do1(int base, char **fields, int f)
{
	start();
	trans(0, f, fields);
	theend();
}

/* 
 * JMP - register address is illegal
 */
void	doG(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * RTS/SPL last 3 bits are arg
 */
void	doQ(int base, char **fields, int f)
{
	while(code < base + 010) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * SCC/CCC last 4 bits are arg
 */
void	doA(int base, char **fields, int f)
{
	while(code < base + 020) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * single operand
 */
void	doS(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * single operand but no mode 0
 */
void	doB(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * BRs, TRAP, EMT lower byte is arg
 */
void	doJ(int base, char **fields, int f)
{
	while(code < base + 0400) {
		start();
		appfunc(fields[0]);
		define(fields[0]);
		theend();
	}
}

/*
 * JSR 
 */
void	doK(int base, char **fields, int f)
{
	while(code < base + 01000) {
		start();
		if((code & 070) == 0)
			appfunc("ILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * double ops
 */
void	doD(int base, char **fields, int f)
{
	while(code < base + 010000) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * double register and source/dest op
 */
void	doR(int base, char **fields, int f)
{
	while(code < base + 01000) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/* 
 * FPP with single float destination
 * mode 0 with register 6 and 7 is illegal
 */
void	doF(int base, char **fields, int f)
{
	while(code < base + 0100) {
		start();
		if((code & 077) == 006 || (code & 077) == 007)
			appfunc("FILL");
		else
			trans(code, f, fields);
		theend();
	}
}

/*
 * FPP - one operand is integer source the other FPP reg.
 */
void	doH(int base, char **fields, int f)
{
	while(code < base + 0400) {
		start();
		trans(code, f, fields);
		theend();
	}
}

/*
 * FPP - two float ops
 * mode 06 and 07 on source is illegal
 */
void	doI(int base, char **fields, int f)
{
	while(code < base + 0400) {
		start();
		if((code & 077) == 006 || (code & 077) == 007)
			appfunc("FILL");
		else
			trans(code, f, fields);
		theend();
	}
}

void	trans(int v, int f, char **fields)
{
	int i;
	char	nbuf[100];

	for(i = 1; i < f; i++) {
		if(fields[i][0] == '%') {
			appfunc(fields[0]);
			define(fields[0]);
		} else switch(fields[i][1]) {
		case '0':	/* DD */
			sprintf(nbuf, "%c%02o", fields[i][0], v & 077);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		case '1':	/* SS */
			sprintf(nbuf, "%c%02o", fields[i][0], (v >> 6) & 077);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		case '2':	/* FPP */
			sprintf(nbuf, "FPP_%c", fields[i][0]);
			if( profiler_output != 2 )
				appfunc(nbuf);
			define(nbuf);
			break;
		default:
			fprintf(stderr, "bad translation key: %c\n", fields[i][1]);
			exit(1);
		}
	}
}

void
getul()
{
	switch(coo) {

	case COO_i386as:
	case COO_sun_as:
		ul = "_";
		break;

	case COO_self_as:
	case COO_c:
		ul = "";
		break;

	default:
		BAD_SWITCH;
	}
}

void
appfunc(char *s)
{
	fprintf(tmp, "%s%s%s", ul, s, (ccc++ < 3) ? ", " : "");
}

void
define(char *n)
{
	int i;

	if(coo != COO_c && !profiler_output)
		return;
	for(i = 0; i < nnames; i++)
		if(!strcmp(names[i], n))
			return;

	if(nnames == MAXNAMES) {
		fprintf(stderr, "recompile me with more MAXNAMES!\n");
		exit(1);
	}
	if(!(n = strdup(n))) {
		fprintf(stderr, "out of mem\n");
		exit(1);
	}
	names[nnames++] = n;
}

void
do_profiler_output()
{
	FILE	*fp;
	int	i;

	if((fp = fopen("cproftab.c", "w")) == NULL) {
		fprintf(stderr, "cproftab.c: %s", strerror(errno));
		exit(1);
	}

	fprintf(fp, "char *inames[] = {\n");
	for(i = 0; i < nnames; i++)
		fprintf(fp, "\t\"%s\",	/* %3d */\n", names[i], i);
	fprintf(fp, "};\n\n");

	fprintf(fp, "# define NNAMES %d\n", nnames);

	fclose(fp);
}

int
ncmp(const void *p1, const void *p2)
{
	return strcmp(*(char **)p1, *(char **)p2);
}
