/* aout.c - parse and load the contents of a UNIX a.out file, for
 * several flavours of PDP-11 UNIX
 */
#include "defines.h"
#define EIGHT_K		 8192

/* UNIX magic numbers for the a.out header */
#define ANY_NORMAL	0407	/* normal: V5,V6,V7,2.11BSD */
#define ANY_ROTEXT	0410	/* read-only text: V5,V6,V7,2.11BSD */
#define ANY_SPLITID	0411	/* seperated I&D: V5,V6,V7,2.11BSD */
#define ANY_OVERLAY	0405	/* overlay: V5,V6,V7,2.11BSD */
#define BSD_OVERLAY	0430	/* 2.11BSD overlay, non-separate */
#define BSD_ROVERLAY	0431	/* 2.11BSD overlay, separate */

/* a.out header for all UNIX flavours */
struct exec {
    uint16_t a_magic;		/* magic number */
    uint16_t a_text;		/* size of text segment */
    uint16_t a_data;		/* size of initialised data */
    uint16_t a_bss;		/* size of initialised bss */
    uint16_t a_syms;		/* size of symbol table */
    uint16_t a_entry;		/* entry point */
    uint16_t a_unused;		/* unused */
    uint16_t a_flag;		/* relocation info stripped */
				/* 16 bytes up to here */

				/* 2.11BSD overlay files have the following */
#define NOVL	15
     int16_t max_ovl;		/* maximum overlay size */
    uint16_t ov_siz[NOVL];	/* size of the i'th overlay */
				/* Note that if the file isn't a 2.11BSD */
				/* overlay, we have to rewind to undo */
				/* the read of this section */
};

/* Because V5, V6, V7 and 2.11BSD share several magic numbers
 * in their a.out headers, we must distinguish them so as to
 * set up the correct emulated environment. This is done by
 * observing the differences in their crt0.s code: they all
 * differ at position 021
 */
#define a_magic2	ov_siz[0]
#define V6_M2		0010600
#define V7_M2		0016600
#define BSD_M2		0162706


/* Array of 64K for data and instruction space */
static uint8_t darray[PDP_MEM_SIZE], iarray[PDP_MEM_SIZE];

/* For programs without an environmwnt, we set the environment statically.
 * Eventually there will be code to get some environment variables
 */
static char *Envp[3] = {
	"PATH=/bin:/usr/bin:/usr/sbin:/usr/ucb:/usr/games:/usr/local/bin:.",
	"HOME=/",
	"TERM=vt100"
};
static int Envc = 3;


/* Load the named PDP-11 executable file into the emulator's memory.
 * Returns 0 if ok, -1 if error
 */
int
load_a_out(const char *file)
{
    FILE *zin;
    struct exec e;
    uint8_t *ibase, *dbase, *bbase;
    int size, i;
    int is_bsd=0;

    signal(SIGBUS, bus_error);	/* Catch all bus errors here */

				/* Open the file & read the header */
    zin = fopen(file, "r");
    if (zin == NULL) return (-1);
    if (fread(&e, sizeof(e), 1, zin) != 1) {
	fclose(zin); return (-1);
    }
				/* Set up the memory areas according to */
				/* the magic numbers */
    switch (e.a_magic) {
    case ANY_NORMAL:
	/* fprintf(stderr, "Normal PDP binary\n"); */
	ibase = ispace = dspace = darray;
	dbase = &(ispace[e.a_text]);
	bbase = &(ispace[e.a_text + e.a_data]);
	dwrite_base= e.a_text;
	if (e.a_magic2==BSD_M2) {
#ifndef EMU211
	    printf("Apout not compiled to support 2.11BSD binaries\n");
	    exit(1);
#endif
	    is_bsd=1;
	}
	fseek(zin, 16, SEEK_SET);
	break;
    case ANY_ROTEXT:
	/* fprintf(stderr, "Read-only text PDP binary\n"); */

				/* Round up text area to next 8K boundary */
	if (e.a_text % EIGHT_K) {
	    size = EIGHT_K * (1 + e.a_text / EIGHT_K);
	} else {
	    size = e.a_text;
	}
	ibase = ispace = dspace = darray;
	dbase = &(ispace[size]);
	bbase = &(ispace[size + e.a_data]);
	dwrite_base= size;
	if (e.a_magic2==BSD_M2) {
#ifndef EMU211
	    printf("Apout not compiled to support 2.11BSD binaries\n");
	    exit(1);
#endif
	    is_bsd=1;
	}
	fseek(zin, 16, SEEK_SET);
	break;
    case ANY_SPLITID:
	/* fprintf(stderr, "Apout - split I&D PDP binary\n"); */
	ibase = ispace = iarray;
	dbase = dspace = darray;
	bbase = &(dspace[e.a_data]);
	if (e.a_magic2==BSD_M2) {
#ifndef EMU211
	    printf("Apout not compiled to support 2.11BSD binaries\n");
	    exit(1);
#endif
	    is_bsd=1; dwrite_base=0;
	} else { dwrite_base=2; }	/* Try to stop null refs */
	fseek(zin, 16, SEEK_SET);
	break;
    case ANY_OVERLAY:
       fprintf(stderr,"Apout currently does not support the 0%o a.out format\n",
							ANY_OVERLAY);
	fclose(zin); return(-1);
    case BSD_OVERLAY:
#ifndef EMU211
	printf("Apout not compiled to support 2.11BSD binaries\n"); exit(1);
#endif
	fprintf(stderr,"Apout currently does not support the BSD 0%o %s",
				BSD_OVERLAY, "overlay a.out format\n");
	fclose(zin); return(-1);
    case BSD_ROVERLAY:
#ifndef EMU211
	printf("Apout not compiled to support 2.11BSD binaries\n"); exit(1);
#endif
	fprintf(stderr,"Apout currently does not support the BSD 0%o %s",
				BSD_ROVERLAY, "overlay a.out format\n");
	fclose(zin); return(-1);
    default:
	fprintf(stderr, "Apout - unknown a.out format\n");
	fclose(zin); return(-1);
    }

    /* Initialise the instruction table for our environment */
#ifdef EMU211
    if (is_bsd)
	{ for (i=548; i<552; i++) itab[i]= bsdtrap; }
    else
#endif
	{ for (i=548; i<552; i++) itab[i]= v7trap; }

#if 1
    memset(darray, 0, PDP_MEM_SIZE);	/* Clear all memory */
    if (ispace != dspace) memset(iarray, 0, PDP_MEM_SIZE);
#endif

    /* Now load the text into ibase */
    for (size = e.a_text; size;) {
	i = fread(ibase, 1, size, zin);
	if (i == -1) {
	    fclose(zin);
	    return (i);
	}
	size -= i;
	ibase += i;
    }

    /* Now load the data into dbase */
    if (dbase)
	for (size = e.a_data; size;) {
	    i = fread(dbase, 1, size, zin);
	    if (i == -1) {
		fclose(zin);
		return (i);
	    }
	    size -= i;
	    dbase += i;
	}

    /* Now clear the bss */
    if (bbase && e.a_bss)
	memset(bbase, 0, e.a_bss);

    regs[PC] = e.a_entry;

    fclose(zin);
    return (0);
}

/*
 * C runtime startoff.	When an a.out is loaded by the kernel, the kernel
 * sets up the stack as follows:
 *
 *	_________________________________
 *	| (NULL)			| 0177776: top of memory
 *	|-------------------------------|
 *	|				|
 *	| environment strings		|
 *	|				|
 *	|-------------------------------|
 *	|				|
 *	| argument strings		|
 *	|				|
 *	|-------------------------------|
 *	| envv[envc] (NULL)		| end of environment vector tag, a 0
 *	|-------------------------------|
 *	| envv[envc-1]			| pointer to last environment string
 *	|-------------------------------|
 *	| ...				|
 *	|-------------------------------|
 *	| envv[0]			| pointer to first environment string
 *	|-------------------------------|
 *	| argv[argc] (NULL)		| end of argument vector tag, a 0
 *	|-------------------------------|
 *	| argv[argc-1]			| pointer to last argument string
 *	|-------------------------------|
 *	| ...				|
 *	|-------------------------------|
 *	| argv[0]			| pointer to first argument string
 *	|-------------------------------|
 * sp-> | argc				| number of arguments
 *	---------------------------------
 *
 * Crt0 simply moves the argc down two places in the stack, calculates the
 * the addresses of argv[0] and envv[0], putting those values into the two
 * spaces opened up to set the stack up as main expects to see it.
 * If v6exec is set, create a 5th/6th Edition environment. Otherwise,
 * create a 7th Edition or 2.11BSD environment.
 */
void
set_arg_env(int argc, char **argv, int envc, char **envp, int v6exec)
{
    int i, posn, len;
    int eposn[MAX_ARGS];
    int aposn[MAX_ARGS];

			/* Set default environment if there is none */
    if (envp==NULL) { envc=Envc; envp=Envp; }

#ifdef DEBUG
    /* Set up the program's name -- used for debugging */
    if (progname) free(progname);
    progname = strdup(argv[0]);

    if (trap_debug) {
	fprintf(dbg_file, "    In set_arg_env, argc is %d\n", argc);
	for (i=0;i<argc;i++)
	    fprintf(dbg_file, "	argv[%d] is %s\n", i, argv[i]);
	for (i=0;i<envc;i++)
	    fprintf(dbg_file, "	envp[%d] is %s\n", i, envp[i]);
    }
#endif

    if (argc > MAX_ARGS) argc = MAX_ARGS;
    if (envc > MAX_ARGS) envc = MAX_ARGS;

    /* Now build the arguments and pointers on the stack; see the Sixth */
    /* and Seventh Edition manuals for a description of the layout */
	
    posn = PDP_MEM_SIZE - 2;
    sl_word(posn, 0);		/* Put a NULL on top of stack */

    if (!v6exec)
	for (i = envc - 1; i != -1; i--) {	/* For each env string */
	    len = strlen(envp[i]) + 1;		/* get its length */
	    posn -= len;
	    memcpy(&dspace[posn], envp[i], len);
	    eposn[i] = posn;
	}

    for (i = argc - 1; i != -1; i--) {	/* For each arg string */
	len = strlen(argv[i]) + 1;	/* get its length */
	posn -= len;
	memcpy(&dspace[posn], argv[i], len);
	aposn[i] = posn;
    }
    posn -= 2;
    sl_word(posn, 0);			/* Put a NULL at end of env array */

    if (!v6exec) {			/* For each env string */
	for (i = envc - 1; i != -1; i--) {
	    posn -= 2;
	    sl_word(posn, eposn[i]);	/* put a pointer to the string */
	}
	posn -= 2;
    }
				/* Put a NULL or -1 before arg pointers */
    if (v6exec) sl_word(posn, -1)
    else sl_word(posn, 0);

    for (i = argc - 1; i != -1; i--) {	/* For each arg string */
	posn -= 2;
	sl_word(posn, aposn[i]);	/* put a pointer to the string */
    }
    posn -= 2;
    sl_word(posn, argc);		/* Save the count of args */
    regs[SP] = posn;			/* and initialise the SP */
}
