#include "defines.h"

#define EIGHT_K		 8192

/* Seventh Edition UNIX magic numbers */
#define V7_NORMAL       0407    /* normal */
#define V7_ROTEXT       0410    /* read-only text */
#define V7_SPLITID      0411    /* seperated I&D */
#define V7_OVERLAY      0405    /* overlay */

/* a.out header */
struct exec {
    uint16_t a_magic;		/* magic number */
    uint16_t a_text;		/* size of text segment */
    uint16_t a_data;		/* size of initialised data */
    uint16_t a_bss;		/* size of initialised bss */
    uint16_t a_syms;		/* size of symbol table */
    uint16_t a_entry;		/* entry point */
    uint16_t a_unused;		/* unused */
    uint16_t a_flag;		/* relocation info stripped */
};

static uint8_t darray[PDP_MEM_SIZE], iarray[PDP_MEM_SIZE];

/* Returns 0 if ok, -1 if error */
int
load_v7_a_out(const char *file)
{
    FILE *zin;
    struct exec e;
    uint8_t *ibase, *dbase, *bbase;
    int size, i;

				/* Open the file & read the header */
    zin = fopen(file, "r");
    if (zin == NULL)
	return (-1);
    if (fread(&e, sizeof(e), 1, zin) != 1) {
	fclose(zin);
	return (-1);
    }
				/* Set up the memory areas according to */
				/* the magic numbers */
    switch (e.a_magic) {
    case V7_NORMAL:
	/* fprintf(stderr, "Normal PDP binary\n"); */
	ibase = ispace = dspace = darray;
	dbase = &(ispace[e.a_text]);
	bbase = &(ispace[e.a_text + e.a_data]);
	dwrite_base= e.a_text;
	break;
    case V7_ROTEXT:
	/* fprintf(stderr, "Read-only text PDP binary\n"); */

				/* Round up text area to next 8K boundary */
	if (e.a_text % EIGHT_K) {
	    size = EIGHT_K * (1 + e.a_text / EIGHT_K);
	} else {
	    size = e.a_text;
	}
	ibase = ispace = dspace = darray;
	dbase = &(ispace[size]);
	bbase = &(ispace[size + e.a_data]);
	dwrite_base= size;
	break;
    case V7_SPLITID:
	/* fprintf(stderr, "Apout - split I&D PDP binary\n"); */
	ibase = ispace = iarray;
	dbase = dspace = darray;
	bbase = &(dspace[e.a_data]);
	dwrite_base=2;
	break;
    case V7_OVERLAY:
	fprintf(stderr, "Apout - PDP overlay binary\n");
	return(-1);
    default:
	fprintf(stderr, "Apout - unknown a.out format\n");
	return(-1);
    }

    /* Initialise the instruction table for our environment */
    for (i=548; i<552; i++) itab[i]= v7trap;

    /* Now load the text into ibase */
    for (size = e.a_text; size;) {
	i = fread(ibase, 1, size, zin);
	if (i == -1) {
	    fclose(zin);
	    return (i);
	}
	size -= i;
	ibase += i;
    }

    /* Now load the data into dbase */
    if (dbase)
	for (size = e.a_data; size;) {
	    i = fread(dbase, 1, size, zin);
	    if (i == -1) {
		fclose(zin);
		return (i);
	    }
	    size -= i;
	    dbase += i;
	}

    /* Now clear the bss */
    if (bbase && e.a_bss)
	memset(bbase, 0, e.a_bss);

    regs[PC] = e.a_entry;

    fclose(zin);
    return (0);
}
