#include <param.h>
#include <systm.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <reg.h>
#include <file.h>
#include <fcntl.h>
#include <inode.h>
#include <errno.h>

extern struct user u;

/*
 * read system call
 */
read()
{
	rdwr(FREAD);
}

/*
 * write system call
 */
write()
{
	rdwr(FWRITE);
}

/*
 * common code for read and write calls:
 * check permissions, set base, count, and offset,
 * and switch out to readi, writei, or pipe code.
 */
rdwr(mode)
register mode;
{
	register struct file *fp;
	register struct inode *ip;
	register struct a
	{
		int	fdes;
		char	*cbuf;
		unsigned count;
	} *uap;
	int type;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if(fp == NULL)
		return;
	if((fp->f_flag&mode) == 0)
	{
		u.u_error = EBADF;
		return;
	}
	u.u_base = (caddr_t)uap->cbuf;
	u.u_count = uap->count;
	u.u_segflg = USERD;
	u.u_fmode = fp->f_flag;
	ip = fp->f_inode;
	type = ip->i_mode & IFMT;
	if (type == IFIFO)
	{
		if(mode == FREAD)
			readp(fp);
		else
			writep(fp);
	}
	else
	{
		u.u_offset = fp->f_offset;
		if (type == IFDIR || type == IFREG || type == IFLOK || type == IFALK)
			plock(ip);
		if(mode == FREAD)
			readi(ip);
		else
			writei(ip);
		if (type == IFDIR || type == IFREG || type == IFLOK || type == IFALK)
			prele(ip);
		fp->f_offset += uap->count-u.u_count;
	}
	u.u_r.r_r0 = uap->count-u.u_count;
}

/*
 * open system call
 */
open()
{
	register struct a
	{
		int	crtmode;
		char	*fname;
		int	mode;
	} *uap;

	uap = (struct a *)u.u_ap;
	/*
	 * The mode passed to open is 0 (read only), 1 (write only),
	 * or 2 (read/write). The internal flag bits are 1 (open for reading),
	 * and 2 (open for writing). Thus by incrementing the passed mode,
	 * the correct internal flag is achieved (yuk!). Other higher-order
	 * bits are unaffected. Also, for compatibility with level 6,
	 * the optional create-mode argument is passed in r0 - hence
	 * (due to the internals of trap) u_dirp must be corrected.
	 */
	u.u_dirp = uap->fname;
	copen(++uap->mode, uap->crtmode);
}

/*
 * creat system call
 */
creat()
{
	register struct a
	{
		char	*fname;
		int	fmode;
	} *uap;

	uap = (struct a *)u.u_ap;
	copen(FWRITE|O_CREAT|O_TRUNC, uap->fmode);
}

/*
 * common code for open and creat.
 * Check permissions, allocate an open file structure,
 * and call the device open routine if any.
 */
copen(mode, arg)
register mode;
{
	register struct inode *ip;
	register struct file *fp;
	int i;

	if (u.u_uid)
		mode &= ~O_FREE;
	if ((mode & (FREAD|FWRITE)) == 0)	/* must have at least one */
	{
		u.u_error = EINVAL;
		return;
	}
	u.u_fmode = mode;
	if (mode & O_CREAT)
	{
		if ((ip = namei(uchar, 1)) == NULL)
		{
			if (u.u_error)
				return;
			if ((ip = maknode(arg & 07777 & ~ISVTX)) == NULL)
				return;
			mode &= ~O_TRUNC;
		}
		else
		{
			if ((mode & O_EXCL) && (ip->i_mode & IFMT) != IFCHR)
			{
				u.u_error = EEXIST;
				iput(ip);
				return;
			}
			mode &= ~O_CREAT;
		}
	}
	else if ((ip = namei(uchar, 0)) == NULL)
		return;
	if ((mode & O_CREAT) == 0)	/* file already present */
	{
		if (mode & FREAD)
			access(ip, IREAD);
		if (mode & (FWRITE|O_TRUNC))
		{
			access(ip, IWRITE);
			if ((ip->i_mode & IFMT) == IFDIR)
				u.u_error = EISDIR;
		}
	}
	if (u.u_error || (fp = falloc()) == NULL)
	{
		iput(ip);
		return;
	}
	if ((mode & O_TRUNC) && (ip->i_mode & IFMT) == IFREG)
		itrunc(ip);
	prele(ip);
	fp->f_flag = mode & FMASK;
	fp->f_inode = ip;
	if ((ip->i_mode & IFMT) == IFIFO)
		fp->f_flag |= O_FREE;
	i = u.u_r.r_r0;
	if (save(u.u_qsav))	/* catch half-opens here rather than in trap */
	{
		if (u.u_error == 0)
			u.u_error = EINTR;
		u.u_ofile[i] = NULL;
		closef(fp);
	}
	else
	{
		openi(ip, mode);
		if (mode & O_APPEND)
			fp->f_offset = ip->i_size;
		if (u.u_error == 0)
		{
			if ((ip->i_mode & IFMT) != IFALK)
				return;
			if (lock(ip, (mode & FWRITE) ? L_WLOCK : L_RLOCK) == 0)
				return;
		}
		u.u_ofile[i] = NULL;
		fp->f_count--;
		iput(ip);
	}
}

/*
 * close system call
 */
close()
{
	register struct file *fp;
	register struct a
	{
		int	fdes;
	} *uap;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if(fp == NULL)
		return;
	u.u_ofile[uap->fdes] = NULL;
	closef(fp);
}

/*
 * seek system call
 */
seek()
{
	register struct file *fp;
	register struct a
	{
		int	fdes;
		off_t	off;
		int	sbase;
	} *uap;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if(fp == NULL)
		return;
	if ((fp->f_inode->i_mode & IFMT) == IFIFO)
	{
		u.u_error = ESPIPE;
		return;
	}
	if(uap->sbase == 1)
		uap->off += fp->f_offset;
	else if(uap->sbase == 2)
		uap->off += fp->f_inode->i_size;
	fp->f_offset = uap->off;
	u.u_r.r_off = uap->off;
}

/*
 * link system call
 */
link()
{
	register struct inode *ip, *xp;
	register struct a
	{
		char	*target;
		char	*linkname;
	} *uap;

	uap = (struct a *)u.u_ap;
	ip = namei(uchar, 0);
	if(ip == NULL)
		return;
	if((ip->i_mode&IFMT)==IFDIR && !suser())
		goto out;
	/*
	 * Unlock to avoid possibly hanging the namei.
	 * Sadly, this means races. (Suppose someone
	 * deletes the file in the meantime?)
	 * Nor can it be locked again later
	 * because then there will be deadly
	 * embraces.
	 */
	prele(ip);
	u.u_dirp = (caddr_t)uap->linkname;
	xp = namei(uchar, 1);
	if(xp != NULL)
	{
		u.u_error = EEXIST;
		iput(xp);
		goto out;
	}
	if (u.u_error)
		goto out;
	if(u.u_pdir->i_dev != ip->i_dev)
	{
		iput(u.u_pdir);
		u.u_error = EXDEV;
		goto out;
	}
	wdir(ip);
	if (u.u_error==0)
	{
		ip->i_nlink++;
		ip->i_flag |= ICHG;
	}

out:
	iput(ip);
}

/*
 * mknod system call
 */
mknod()
{
	register struct inode *ip;
	register struct a
	{
		char	*fname;
		int	fmode;
		int	dev;
	} *uap;

	uap = (struct a *)u.u_ap;
	if(suser())
	{
		ip = namei(uchar, 1);
		if(ip != NULL)
		{
			u.u_error = EEXIST;
			goto out;
		}
	}
	if(u.u_error)
		return;
	u.u_fmode = 0;		/* no O_FREE mknod's yet */
	ip = maknode(uap->fmode);
	if (ip == NULL)
		return;
	setdevno(ip, uap->dev);

out:
	iput(ip);
}

/*
 * access system call
 */
saccess()
{
	register svuid, svgid;
	register struct inode *ip;
	register struct a
	{
		char	*fname;
		int	fmode;
	} *uap;

	uap = (struct a *)u.u_ap;
	svuid = u.u_uid;
	svgid = u.u_gid;
	u.u_uid = u.u_ruid;
	u.u_gid = u.u_rgid;
	ip = namei(uchar, 0);
	if (ip != NULL)
	{
		if (uap->fmode&(IREAD>>6))
			access(ip, IREAD);
		if (uap->fmode&(IWRITE>>6))
			access(ip, IWRITE);
		if (uap->fmode&(IEXEC>>6))
			access(ip, IEXEC);
		iput(ip);
	}
	u.u_uid = svuid;
	u.u_gid = svgid;
}
