/*
 *	Indirect driver for multiplexed tty lines
 *
 *	Each physical line can support up to 16 virtual lines.
 *
 *	The protocol blocks data as follows:-
 *		SOH | PORT_ID | DATA_COUNT | ~DATA_COUNT | ... DATA ...
 *	Flow control is enabled by receipt of a flow control block as follows:-
 *		SOH | PORT_ID | 0 | SIZE	[0<SIZE<127]
 *
 *	March 1980		Piers Lauder
 *				Dept of Comp Sci
 *				Sydney University
 *
 *	Note: delays are disallowed.
 *
 *	Modified to run under level 7 - Kevin Hill (UNSW) (Feb 81)
 */

#include <param.h>
#include <tty.h>
#include <sgtty.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <conf.h>
#include <file.h>
#include <mx.h>
#include <errno.h>
#include <fcntl.h>

extern struct user u;

/*
 *	Names from tty.h reused by mx.c
 *	t_mxent is for the real lines (t_cchar only used when connecting).
 *	t_mxline is for the virtual lines (t_addr only used in kl driver).
 */
#define	t_mxent		t_cchar		/* used by mxinput to find mx slot */
#define	t_mxline	t_addr		/* line number back reference */

/*
 *	Line structures/counts
 */
struct mx mx[];			/* the physical lines */
short	mx_nlines;		/* number of physical lines */
short	mx_cnt;			/* number of virtual lines */
struct	tty	mx_tty[];	/* one per virtual line */
struct	mxflc	mxflc[];	/* one per virtual line */

/*
 *	Other parameters
 */
#define	MAXFIDLE	5		/* secs. before retry on idle line */
#define	MXSCANRATE	10		/* scan ports for hung output every 10 ticks */
#define	BLOCK_HDR_SIZ	4		/* bytes in block header */
#define	MAX_BLK_SIZ	(2*CBSIZE)	/* max recommended block size */
#define	FLOW_RATE	(4*CBSIZE)	/* flow control rate enabled */
#define	SOH		0376		/* "start-of-header" */
#define	SPLMX		spl6		/* priority to lock out any possible device interrupts */
/* #define LOST_BYTES_DEFAULT		/* send lost bytes to port 0 */

/*
 *	mx_state (must be in order given - see mxinput)
 */
#define	MX_IDLE		0		/* must be 0		*/
#define	MX_SOH		MX_IDLE		/* expecting SOH	*/
#define	MX_PORT		1		/*    "    port_id	*/
#define	MX_SIZE0	2		/*    "    data size	*/
#define	MX_SIZE1	3		/*    "    second size	*/
#define	MX_FLOW		4		/*    "    flow control	*/
#define	MX_WOPEN	5		/* open or close active already */

/*
 *	Miscellany
 */
char	mxscanning;		/* TRUE when "mxscan" timing out */
struct timout	*timeout();
char	mxinflag;		/* TRUE when mxinput calling ttyinput within a data block */
				/* This is used to avoid a degradation in overhead on "connected" ports */

short	mxspeedmap[16] =	/* map "t_ospeed" to byte rate */
{
	0, 5, 8, 10, 15, 15, 20, 30,
	60, 120, 180, 240, 480, 960, 1920, 200
};



/*
 *	Open -- call device open and set up structures
 */
mxopen(dev, flag)
{
	register struct mx	*mxp;
	register struct tty	*tp;
	register struct mxflc	*mxf;
	short			port;
	short			line;
	short			mxstart();

	port = minor(dev);
	if (port >= mx_cnt)
	{
		u.u_error = ENXIO;
		return;
	}
	tp = &mx_tty[port];
	if (tp->t_state & XCLUDE)
	{
		u.u_error = ENXIO;
		spl0();
		return;
	}
	mxf = &mxflc[port];
	line = 0;

	mxp = &mx[0];
	while (port >= mxp->mx_nports)
	{
		port -= mxp->mx_nports;
		line++;
		mxp++;
	}
	/*
	 * "port" is now the port number for the line given by "line"
	 * "tp" is pointing to its tty structure
	 */

	SPLMX();
	if (flag & O_EXCL)
		tp->t_state |= XCLUDE;

	/*
	 * Is someone else currently opening/closing a port on this line?
	 */
	while (mxp->mx_state == MX_WOPEN)
		sleep((caddr_t)mxp, TTOPRI);

	if (mxp->mx_open == 0)	/* this is the first for this line */
	{
		struct tty *dtp;	/* tty structure of the actual line */
		struct sgttyb lp;
		int i;			/* big enough to contain an address */

		spl0();
		mxp->mx_dbase = minor(dev) - port;
		if ((dtp = cdevsw[major(mxp->mx_dev)].d_ttys) == 0)
		{
			u.u_error = ENXIO;
			return;
		}
		dtp = &dtp[minor(mxp->mx_dev)];	/* assumed contiguous */

		/*
		 * Make sure the physical line is idle. If so,
		 * grab it for exclusive use to prevent foul-ups.
		 */
		if (dtp->t_state & (ISOPEN|WOPEN))
		{
			u.u_error = ENXIO;
			return;
		}
		i = (int)u.u_ttyp;	/* address of controlling tty */
		mxp->mx_state = MX_WOPEN;	/* lock out others if sleep */
		if (save(u.u_qsav))
		{
			u.u_error = EINTR;
			dtp->t_state &= ~WOPEN;
			mxp->mx_state = MX_IDLE;
			wakeup((caddr_t)mxp);
			return;
		}
		(*cdevsw[major(mxp->mx_dev)].d_open)(mxp->mx_dev, O_EXCL);
		if (u.u_error)
		{
			mxp->mx_state = MX_IDLE;
			wakeup((caddr_t)mxp);
			return;
		}
		u.u_ttyp = (struct tty *)i;	/* restore controlling tty */
		/*
		 * set real line parameters
		 */
		i = mxp->mx_speeds;
		/*
		 * An unprivileged process may not set its terminal speed
		 * higher than that given it by default (at this site).
		 * As this is not applicable in this case, the following
		 * line forces the speed change.
		 */
		dtp->t_ispeed = i; dtp->t_ospeed = i;
		lp.sg_ispeed = i; lp.sg_ospeed = i;
		lp.sg_kill = 0; lp.sg_erase = 0;
		lp.sg_flags = RAW;
		u.u_segflg = KERND;	/* copy in from kernel space */
		u.u_count = sizeof lp;
		(*cdevsw[major(mxp->mx_dev)].d_ioctl)(mxp->mx_dev, TIOCSETP, &lp, O_EXCL);

		SPLMX();
		dtp->t_rtype = TMULTIPLEX;
		dtp->t_mxent = line;
		dtp->t_pgrp = NULL;
		mxp->mx_rtty = dtp;
		mxp->mx_count = 0;
		mxp->mx_state = MX_IDLE;
		wakeup((caddr_t)mxp);
	}

	/*
	 * "tp" is still pointing to the tty structure for this port
	 */
	if ((tp->t_state & ISOPEN) == 0)
	{
		tp->t_ispeed = tp->t_ospeed = mxp->mx_speeds;
		tp->t_oproc = mxstart;
		tp->t_flags = (ODDP | EVENP | RAW);
		tp->t_state |= CARR_ON;
		tp->t_mxline = (caddr_t)line;
		mxf->fc_lrsiz = 0;
		mxf->fc_mxtflc = 0;
	}
	ttyopen(dev, tp);	/* sets ISOPEN amongst other things */
	mxp->mx_open |= 1 << port;
	if (!mxscanning)
		mxscan();
	mxflow(tp);
	spl0();
}

/*
 *	Read -- set tty structure and call "ttread"
 */
mxread(dev)
{
	register struct tty *tp;
	register struct mxflc *mxf;

	tp = &mx_tty[minor(dev)];
	mxf = &mxflc[minor(dev)];
	ttread(tp);
	if ((mxf->fc_mxrflc + tp->t_rawq.c_cc) < (FLOW_RATE/2))
	{
		SPLMX();
		mxflow(tp);	/* "read ahead" */
		spl0();
	}
}

/*
 *	Write -- set tty structure and call "ttwrite",
 *		 then ensure all output transferred to real line
 */
mxwrite(dev)
{
	register struct tty *tp;

	tp = &mx_tty[minor(dev)];
	ttwrite(tp);
	while (tp->t_outq.c_cc)
	{
		SPLMX();
		tp->t_state |= ASLEEP;
		sleep((caddr_t)&tp->t_outq, TTOPRI);
		mxstart(tp);
		spl0();
	}
}

/*
 *	Ioctl -- set tty structure and call "ttioccomm"
 */
mxioctl(dev, cmd, arg, mode)
{
	register struct tty *tp;

	tp = &mx_tty[minor(dev)];
	/*
	 * delays can not be allowed at this stage (too messy!)
	 */
	if (ttioccomm(cmd, tp, arg, dev))
		tp->t_flags &= ~(VTDELAY|CRDELAY|TBDELAY|NLDELAY);
}

/*
 *	Close -- call device close if last port
 */
mxclose(dev)
{
	register struct tty *tp;
	register struct mx *mxp;

	tp = &mx_tty[minor(dev)];
	flushtty(tp);	/* mxwrite has guaranteed there is no output waiting */
	tp->t_state = CARR_ON;
	mxp = &mx[(short)(tp->t_mxline)];
	mxp->mx_open &= ~(1 << (minor(dev) - mxp->mx_dbase));
	if (mxp->mx_open)	/* still some open ports? */
		return;
	mxp->mx_state = MX_WOPEN;	/* lock out opens in case sleep */
	(*cdevsw[major(mxp->mx_dev)].d_close)(mxp->mx_dev, O_EXCL);
	tp = mxp->mx_rtty;
	tp->t_rtype = 0;
	mxp->mx_state = MX_IDLE;
	wakeup((caddr_t)mxp);
}

/*
 *	Start -- called from "ttwrite" & "mxwrite" & "mxscan" & interrupts
 *	(for echo) at spltty -- if device has enough space, and the port output
 *	rate is low enough, transfer outq's, adding protocol.
 */
short
mxstart(tp)
register struct tty *tp;
{
	struct tty *dtp;
	register struct mxflc *mxf;
	register count;
	register struct clist *cp;
	register struct clist *dcp;
	short line;
	extern struct cblock *cfreelist;

	if (mxinflag || (tp->t_state & TTSTOP))
		return;
	dtp = mx[(short)(tp->t_mxline)].mx_rtty;	/* real line tty structure */
	cp = &tp->t_outq;
	dcp = &dtp->t_outq;
	line = (short)(tp - mx_tty);
	mxf = &mxflc[line];
	if ((count = TTHIWAT - dcp->c_cc - BLOCK_HDR_SIZ) > cp->c_cc)
		count = cp->c_cc;
	if (count > MAX_BLK_SIZ)
		count = MAX_BLK_SIZ;
	if (mxf->fc_lrsiz && (count > mxf->fc_mxtflc))
		count = mxf->fc_mxtflc;
	if (count > 0 && mxf->fc_rate < mxspeedmap[tp->t_ospeed])
	{
		spl6();			/* keep clock out */
		if (cfreelist != NULL)	/* one extra block may be needed */
		{
			putc(SOH, dcp);
			putc(line - mx[(short)(tp->t_mxline)].mx_dbase, dcp);
			putc(count, dcp);
			putc(~count, dcp);
			spl5();
			mxf->fc_rate += count;
			if (mxf->fc_lrsiz)
				mxf->fc_mxtflc -= count;
			do
				putc(getc(cp), dcp);
			while (--count);
			ttstart(dtp);
		}
		else
			spl5();
	}
}


/*
 *	Flow - generate flow control block for port
 *		N.B. flow enabled is not cumulative
 */
mxflow(tp)
register struct tty *tp;
{
	register struct tty *dtp;
	register struct clist *dcp;
	short line;
	extern struct cblock *cfreelist;

	dtp = mx[(short)(tp->t_mxline)].mx_rtty;
	dcp = &dtp->t_outq;
	line = (short)(tp - mx_tty);
	if ((TTHIWAT - dcp->c_cc) < BLOCK_HDR_SIZ)
		return;
	spl6();
	if (cfreelist != NULL)
	{
		putc(SOH, dcp);
		putc(line - mx[(short)(tp->t_mxline)].mx_dbase, dcp);
		putc(0, dcp);
		putc(FLOW_RATE, dcp);
		spl5();
		ttstart(dtp);
		mxflc[line].fc_mxrflc = FLOW_RATE;
	}
	else
		spl5();
}


/*
 *	Scan ports for hung output and restart it
 */
mxscan()
{
	register struct tty *tp;
	register struct mxflc *mxf;
	short line;
	short rescan;
	static short rate_scan;

	rescan = 0;
	if ((rate_scan += MXSCANRATE) >= HZ)
	{
		/*
		 * Scan all virtual lines once per second
		 */
		rate_scan -= HZ;
		tp = mx_tty;
		mxf = mxflc;
		for (line = 0; line < mx_cnt; line++)
		{
			if (tp->t_state & ISOPEN)
			{
				if ((mxf->fc_rate -= mxspeedmap[tp->t_ospeed]) < 0)
					mxf->fc_rate = 0;
				if (mxf->fc_idle > MAXFIDLE)
				{
					mxf->fc_mxrflc = 0;
					mxf->fc_idle = 0;
				}
				else
					mxf->fc_idle++;
			}
			tp++;
			mxf++;
		}
	}
	for (line = 0; line < mx_nlines; line++)
	{
		register struct mx *mxp;
		short port;

		mxp = &mx[line];
		if (mxp->mx_state == MX_WOPEN)
			continue;
		port = mxp->mx_oport;
		tp = &mx_tty[mxp->mx_dbase + port];
		mxf = &mxflc[mxp->mx_dbase + port];
		do
		{
			if (++port >= mxp->mx_nports)
			{
				tp = &mx_tty[mxp->mx_dbase];
				mxf = &mxflc[mxp->mx_dbase];
				port = 0;
			}
			else
			{
				tp++;
				mxf++;
			}
			if (mxp->mx_open & (1 << port))	/* line open */
			{
				rescan++;	/* reprime when finished */
				/*
				 *	Output?
				 */
				if (tp->t_state & ASLEEP)
				{
					if (mxp->mx_rtty->t_outq.c_cc <= TTLOWAT)
					{
						tp->t_state &= ~ASLEEP;
						wakeup((caddr_t)&tp->t_outq);
						/*
						 * only wakeup one port this line
						 */
						break;
					}
				}
				else if (tp->t_rtype && mxf->fc_rate == 0 && tp->t_outq.c_cc)
				{
					/*
					 * this line redirected
					 */
					mxstart(tp);
					break;
				}
				/*
				 *	Input?
				 */
				if (tp->t_rtype == NULL)
				{
					if (mxf->fc_mxrflc == 0 &&
					   (tp->t_rawq.c_cc < FLOW_RATE ||
					    ((tp->t_flags&(RAW|CBREAK)) == 0 &&
					     tp->t_delct == 0)))
						mxflow(tp);
				}
				else if (mxf->fc_mxrflc < FLOW_RATE &&
				    (mxf->fc_mxrflc + tp->t_redirect->t_outq.c_cc) < TTHIWAT)
					mxflow(tp);
			}
		}
		while (port != mxp->mx_oport);
		mxp->mx_oport = port;
	}
	if (rescan)
		mxscanning = timeout(mxscan, 0, MXSCANRATE) != NULL;
	else
		mxscanning = 0;
}


/*
 *	Input -- the protocol handler for received bytes
 *	-- called from "ttyinput" at interrupt time
 */
mxinput(c, dtp)
register unsigned short c;		/* really a char */
register struct tty *dtp;
{
	register struct mx *mxp;

	mxp = &mx[dtp->t_mxent];
	if (mxp->mx_count)
	{
		if (--mxp->mx_count)
			/*
			 * Inform any indirect call to mxstart that there is
			 * more to come; unfortunately this generates
			 * "store and forward" behaviour.
			 */
			mxinflag++;
		ttyinput(c, &mx_tty[mxp->mx_ctty]);
		mxinflag = 0;
		return;
	}
	c &= 0377;
	switch (mxp->mx_state)
	{
	case MX_SOH:
		if (c == SOH)
		{
			mxp->mx_state++;
			return;
		}
		break;

	case MX_PORT:
		if (c < mxp->mx_nports)
		{
			mxp->mx_ctty = mxp->mx_dbase + c;
			mxp->mx_state++;
			return;
		}
		break;

	case MX_SIZE0:
		if (mxp->mx_cnt0 = c)
			mxp->mx_state++;
		else
			mxp->mx_state = MX_FLOW;
		return;

	case MX_SIZE1:
		c = ~c & 0377;
		if (c == mxp->mx_cnt0)
		{
			mxinset(mxp);
			mxp->mx_state = MX_SOH;
			return;
		}
		break;

	case MX_FLOW:
		if (c != 0 && c <= 127)		/* so no sign-extension */
		{
			mxinflow(mxp->mx_ctty, c);
			mxp->mx_state = MX_SOH;
			return;
		}
	}
#ifdef	LOST_BYTES_DEFAULT
	ttyinput(c, &mx_tty[mxp->mx_dbase]);
#endif	LOST_BYTES_DEFAULT
	mxp->mx_xbytes++;		/* log the stray char */
	mxp->mx_state = MX_SOH;
}


/*
 *	Set up input from block
 */
mxinset(mxp)
register struct mx *mxp;
{
	register struct tty *tp;
	register struct mxflc *mxf;
	register struct tty *rtp;

	tp = &mx_tty[mxp->mx_ctty];
	mxf = &mxflc[mxp->mx_ctty];
	if ((rtp = tp->t_redirect) == NULL)
	{
		if ((tp->t_rawq.c_cc + mxf->fc_mxrflc) < FLOW_RATE ||
		    ((tp->t_flags & (RAW|CBREAK)) == 0 && tp->t_delct == 0))
			mxflow(tp);
		else if ((mxf->fc_mxrflc -= mxp->mx_cnt0) < 0)
			mxf->fc_mxrflc = 0;
	}
	else if ((rtp->t_outq.c_cc + mxf->fc_mxrflc) < FLOW_RATE)
		mxflow(tp);
	else if ((mxf->fc_mxrflc -= mxp->mx_cnt0) < 0)
		mxf->fc_mxrflc = 0;
	mxp->mx_count = mxp->mx_cnt0;
	mxf->fc_idle = 0;
}


/*
 *	Enable transmit flow
 */
mxinflow(line, c)
short line;
char c;
{
	register struct tty *tp;
	register struct mxflc *mxf;

	tp = &mx_tty[line];
	mxf = &mxflc[line];
	mxf->fc_lrsiz = c;
	mxf->fc_mxtflc = c;
	if (tp->t_outq.c_cc)
		mxstart(tp);
}
