/*
 *   KL/DL-11 driver
 */
#include <param.h>
#include <conf.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <tty.h>
#include <sgtty.h>
#include <systm.h>
#include <errno.h>

extern struct user u;

/* base address */
#define	KLADDR	((struct device *)0177560)	/* console */
#define	KLBASE	((struct device *)0176500)	/* kl and dl11-a */
#define	DLBASE	((struct device *)0175610)	/* dl-e */
#define	NKL11	1
#define	NDL11	0
#define	DONE	0200
#define	IENABLE	0100
#define DSRDY	02
#define	RDRENB	01
#define	DLDELAY	4	/* Extra delay for DL's (double buff) */

#define	NL1	000400
#define	NL2	001000
#define	CR2	020000
#define	FF1	040000
#define	TAB1	002000

struct	tty kl_tty[NKL11+NDL11];
int	klstart();
int	ttrstrt();
char	partab[];

struct device
{
	int	rcsr;
	int	rbuf;
	int	tcsr;
	int	tbuf;
};

klopen(dev, flag)
dev_t dev;
{
	register struct device *addr;
	register struct tty *tp;
	register d;

	d = minor(dev);
	if(d >= NKL11+NDL11)
	{
		u.u_error = ENXIO;
		return;
	}
	tp = &kl_tty[d];
	/*
	 * set up minor 0 to address KLADDR
	 * set up minor 1 thru NKL11-1 to address from KLBASE
	 * set up minor NKL11 on to address from DLBASE
	 */
	if(d == 0)
		addr = KLADDR;
	else if(d < NKL11)
		addr = KLBASE + (d-1);
	else
		addr = DLBASE + (d-NKL11);
	tp->t_addr = (caddr_t)addr;
	tp->t_oproc = klstart;
	if ((tp->t_state&ISOPEN) == 0)
	{
		tp->t_state = ISOPEN|CARR_ON;
		tp->t_flags = EVENP|ECHO|XTABS|CRMOD;
		tp->t_erase = CERASE;
		tp->t_kill = CKILL;
		tp->t_ispeed = B300;	/* default speed 300 baud */
		tp->t_ospeed = B300;
	}
	addr->rcsr |= IENABLE|DSRDY|RDRENB;
	addr->tcsr |= IENABLE;
	ttyopen(dev, tp);
}

klclose(dev, flag)
dev_t dev;
int flag;
{
	register struct tty *tp;

	tp = &kl_tty[minor(dev)];
	ttyclose(tp);
}

klread(dev)
dev_t dev;
{
	ttread(&kl_tty[minor(dev)]);
}

klwrite(dev)
dev_t dev;
{
	ttwrite(&kl_tty[minor(dev)]);
}

klxint(dev)
dev_t dev;
{
	register struct tty *tp;

	tp = &kl_tty[minor(dev)];
	ttstart(tp);
	if (tp->t_state&ASLEEP && tp->t_outq.c_cc<=TTLOWAT)
		wakeup((caddr_t)&tp->t_outq);
}

klrint(dev)
dev_t dev;
{
	register int c;
	register struct device *addr;
	register struct tty *tp;

	tp = &kl_tty[minor(dev)];
	addr = (struct device *)tp->t_addr;
	c = addr->rbuf;
	addr->rcsr |= RDRENB;
	ttyinput(c, tp);
}

klioctl(dev, cmd, addr, flag)
caddr_t addr;
dev_t dev;
{
	if (ttioccom(cmd, &kl_tty[minor(dev)], addr, dev)==0)
		u.u_error = ENOTTY;
}

klstart(tp)
register struct tty *tp;
{
	register c;
	register struct device *addr;

	addr = (struct device *)tp->t_addr;
	if((addr->tcsr&DONE) == 0)
		return;
	if ((c=getc(&tp->t_outq)) >= 0)
	{
		if (tp->t_flags&RAW)
			addr->tbuf = c;
		else if (c<=0177)
			addr->tbuf = c | (partab[c]&0200);
		else
		{
			timeout(ttrstrt, (caddr_t)tp, (c&0177) + DLDELAY);
			tp->t_state |= TIMEOUT;
		}
	}
}

/*
 * Print a character on console.
 * Attempts to save and restore device
 * status.
 */
putchar(c)
register c;
{
	register s, timo;

	/*
	 *  If last char was a break or null, don't print
	*/
	if ((KLADDR->rbuf&0177) == 0)
		return;
	timo = 30000;
	/*
	 * Try waiting for the console tty to come ready,
	 * otherwise give up after a reasonable time.
	 */
	while((KLADDR->tcsr&0200) == 0)
		if(--timo == 0)
			break;
	if(c == 0)
		return;
	s = KLADDR->tcsr;
	KLADDR->tcsr = 0;
	KLADDR->tbuf = c;
	if(c == '\n')
	{
		putchar('\r');
		putchar(0177);
		putchar(0177);
	}
	putchar(0);
	KLADDR->tcsr = s;
}
