/*
 *	Optimised RHP04 and RJP04 driver
 *
 *	When instructed correctly (via the ifdefs) this driver will attempt
 *	to queue io requests by drive, cylinder and sector, and endeavour
 *	to overlap seeks with data transfers for more than one drive.
 *	Method is very similar to Childrens Museum RK driver.
 *
 *	Some error correction and recovery is attempted.
 *	You have been WARNED.
 *
 *	Based on the novel by Peter Ivanov	UNSWDCS
 *
 *	Debugged and recoded by		Ian Johnstone	AGSM
 *				and
 *					Chris Maltby	DCS.
 */

#include	<param.h>
#include	<buf.h>
#include	<dir.h>
#include	<signal.h>
#include	<user.h>
#include	<errno.h>

extern struct user u;

#define	PRESEEK	1	/* 1: enable overlapped seeks on multiple drives */
#define	SWEEP	1	/* 1: enable checking to force max head travel */
#define	RJP04	0	/* 1: RP04's on the unibus */
#define	HPECC	0	/* 1: enable data check correction */
#define	HPSTATS	0	/* 1: enable stats collection */
#define	NO_OFFLINEQ 0	/* 1: stop qing of requests for offlined drives */
#define	DROP_OFFLINEQ 0	/* 1: delete drive request q when offlined */

struct
{
	int	hpcs1;	/* Control and Status register 1 */
	int	hpwc;	/* Word Count register */
	int	hpba;	/* Bus address register */
	int	hpda;	/* Desired disc address - sector/track */
	int	hpcs2;	/* Control and Status register 2 */
	int	hpds;	/* Drive status register */
	int	hper1;	/* Error register 1 */
	int	hpas;	/* Attention summary register */
	int	hpla;	/* Disc position look-ahead register */
	int	hpdb;	/* Data buffer */
	int	hpmr;	/* Maintenance register */
	int	hpdt;	/* Drive type encoding */
	int	hpsn;	/* Drive serial number */
	int	hpof;	/* Offset register - contains fm22 bit */
	int	hpdc;	/* Desired cylinder address register */
	int	hpcc;	/* Current cylinder address register */
	int	hper2;	/* Error register 2 */
	int	hper3;	/* Error register 3 */
	int	hppos;	/* Burst error bit position */
	int	hppat;	/* Burst error bit pattern */
#if	RJP04 == 0
	int	hpbae;	/* 11/70 bus extension register */
	int	hpcs3;	/* Control and Status register 3 */
#endif
};

#define	HPADDR	0176700
#define	NDRV	  2	/* the number of real rp04 drives on controller */
#define	NCYLS	411	/* cylinders per pack */
#define	NTRKS	 19	/* tracks per cylinder */
#define	NSECS	 22	/* sectors per track */
#define	TVOL	(sizeof hp_sizes/sizeof hp_sizes[0])	/* total number of volumes */
#define	INTLV	6	/* magic interleaving number */
#define	HPAGE	10	/* number of times this block may be pre-empted for io
					   before io is forced	*/
#define	HPSPL	spl5	/* priority of hp disc */

/*
 *	The following maps logical volumes of various sizes to locations
 *	on various drives. This must be altered to add extra drives.
 *	PLEASE GET THEM CORRECT FIRST TIME.
 */

struct
{
	daddr_t	nblocks;
	int		cyloff;
	int		drive;
}
hp_sizes[]
{
	{  61864,   0, 0, },	/*   0 - 147	148 cyl */
	{  24662, 148, 0, },	/* 148 - 206	 59 cyl */
	{   5016, 207, 0, },	/* 207 - 218	 12 cyl */
	{  15048, 219, 0, },	/* 219 - 254	 36 cyl */
	{  65208, 255, 0, },	/* 255 - 410	156 cyl */
	{ 171798,   0, 0, },	/*   0 - 410	411 cyl */
	{  61864,   0, 1, },	/*   0 - 147	148 cyl */
	{  24662, 148, 1, },	/* 148 - 206	 59 cyl */
	{   5016, 207, 1, },	/* 207 - 218	 12 cyl */
	{  15048, 219, 1, },	/* 219 - 254	 36 cyl */
	{  65208, 255, 1, },	/* 255 - 410	156 cyl */
	{ 171798,   0, 1, },	/*   0 - 410	411 cyl */
};

/*
 *	structure of an hp disc queue
 */

struct hpq
{
	char	hpq_flags;		/* flags for qs */
					/* assumed zero - HPOFF==0 */
#define	HPOFF		00	/* drive offline */
#if	PRESEEK
#define	HPSEEK		01	/* seeking flag for q header */
#endif
#define	HPREADY		02	/* ready for io flag for q header */
#define	HPBUSY		03	/* data transfer in progress */
#define	HPRECAL		04	/* recalibrate in progress */
#define	HPIDLE		05	/* nought doing */

#if	SWEEP
	char	hpq_dirf;		/* current direction of head movement */
#define	UP	1		/* the disc cyls are moving up */
#define	DOWN	0		/* the disc cyls are moving down */

#endif
	struct	buf	*hpq_bufp;	/* pointer to first buffer in queue for this drive */
#if	HPSTATS
	char	hpq_nreq;		/* number of requests in queue - stats */
	char	hpq_rmax;			/* high water mark for q */
	unsigned hp_cyls[NCYLS];	/* accesses per cylinder */
#endif
} hpq[NDRV];

#if	HPSTATS
/*
 *	gather statistics to decide on optimal interleaving factor.
 *	When starting an io accumulate in hpintlv the count of
 *	each segment to go to io transfer start.
 */
long hpintlv[88];	/* divide disk into 88 segments */
#endif



/*
 *	We use a few buffer variables for other purposes here, ie:
 *		b_scratch	to hold unit/cylinder address of this request
 *		b_resid		to hold track/sector address of this request
 *		av_back		to hold request age count
 */

struct { char lobyte; char hibyte; };
struct { int integ; };

#define	hp_cyl		b_scratch	/* at least an int */
#define	hp_sector	b_resid.lobyte
#define	hp_track	b_resid.hibyte
#define	hp_trksec	b_resid
#define	hp_age		av_back.integ

struct	buf	hpbuf;

/* various drive commands utilised herein */

#define	GO		 01
#define	PRESET		020	/* read-in-preset */
#define	RECAL		 06	/* recalibrate */
#define	DCLR		010	/* drive clear */
#define	SEEK		 04	/* seek */
#define	SEARCH		030	/* search */
#define	READ		070	/* read */
#define	WRITE		060	/* write */
#define	UNLOAD		 02	/* unload pack */

/*	hpds bits	*/

#define	DRY	   0200		/* drive ready */
#define	VV	   0100		/* volume valid */
#define	MOL	 010000		/* medium on line */
#define	ERR	 040000		/* the OR of error bits */
#define	ATA	0100000		/* attention bit */

/*	hpcs1 bits	*/

#define	IE	   0100		/* interrupt enable bit */
#define	RDY	   0200		/* controller ready */
#define	TRE	 040000		/* the OR of error bits */

/*	hper1 bits	*/

#define	ECH	 000100		/* hard ecc error */
#define	AOE	 001000		/* end of drive error */
#define	WLE	 004000		/* Write protect error */
#define	DTE	 010000		/* drive timing error */
#define	OPI	 020000		/* operation incomplete */
#define	UNS	 040000		/* drive unsafe */
#define	DCK	0100000		/* data check error for RHPs and RJPs (DEC) */

/*	hpcs2 bits	*/

#define	CLR	040		/* controller clear */

/*	hpof bits	*/

#define	FMT22	 010000		/* 16bit format for RHPs and RJPs (DEC) */




hpopen(dev)
{
	register int	n;

	/*
	**	If the requested drive is offline -> error
	**	Initialize it if necessary. Error if this fails.
	*/
	if((n = minor(dev)) >= TVOL)
	{
		u.u_error = ENXIO;
		return;
	}
	n = hp_sizes[n].drive;
	HPSPL();
	if( hpq[n].hpq_flags == HPOFF)
	{
		hpdinit(n);
		if(hpq[n].hpq_flags == HPOFF)
		{
			u.u_error = ENXIO;
		}
	}
	spl0();
}

hpstrategy(bp)
register struct buf	*bp;
{
	register unsigned	p1, p2;
	int dirf;

	p1 = minor(bp->b_dev);

	/*
	 *	Validate the request
	 */

	p2 = &hp_sizes[p1];
#if	NO_OFFLINEQ || DROP_OFFLINEQ
	if( p2->nblocks <= bp->b_blkno || hpq[p2->drive].hpq_flags == HPOFF )
#else
	if( p2->nblocks <= bp->b_blkno )
#endif
	{
		bp->b_flags |= B_ERROR;
		iodone(bp);
		return;
	}

#if	RJP04
	if(cputype == 70 && (bp->b_flags & B_PHYS))
	{
		mapalloc(bp);
	}
#endif

	bp->av_forw = 0;
	bp->hp_age = HPAGE;
	bp->b_error = 0;	/* clear error count */
	bp->hp_cyl = p2->cyloff;

	{
		daddr_t fred;

		fred = bp->b_blkno;
		bp->hp_cyl += fred / (NSECS * NTRKS);
		p1 = fred % (NSECS * NTRKS);
		bp->hp_track = p1 / NSECS;
		bp->hp_sector = p1 % NSECS;
	}
	p2 = &hpq[p2->drive];

/*
 *	Now "hp_cyl" contains "cylinder"
 *	and "hp_trksec" contains the "track and sector"
 */

#if	HPSTATS
	p2->hp_cyls[bp->hp_cyl]++;
	p2->hpq_nreq++;
	if( p2->hpq_nreq > p2->hpq_rmax ) p2->hpq_rmax = p2->hpq_nreq;
#endif

	HPSPL();
	if ((p1 = p2->hpq_bufp) == NULL)
	{
		/* this queue is empty */
		p2->hpq_bufp = bp;
		hpstart();
	}
	else
	{
		/*
		 * the queue is not empty, so place in queue so as to
		 * minimise head movement.
		 */
#if	SWEEP
		dirf = p2->hpq_dirf;
#endif

		p2 = p1->av_forw;
		while(p2)
		{
			/* skip any overtaken blocks */
			if( !(p2->hp_age) )
				p1 = p2;
			p2 = p2->av_forw;
		}

		for( ; p2 = p1->av_forw; p1 = p2)
		{
			if( p1->hp_cyl<=bp->hp_cyl && bp->hp_cyl<=p2->hp_cyl
			 || p1->hp_cyl>=bp->hp_cyl && bp->hp_cyl>=p2->hp_cyl )
			{
				while (bp->hp_cyl == p2->hp_cyl)
				{
				/*
				 * for a cylinder match, do the
				 * rotational optimisation.
				 * INTLV is presumably the optimal value.
				 * SEE HPSTATS as to how this could be done
				 */
					if(p2->hp_sector > p1->hp_sector)
					{
						if(bp->hp_sector > p1->hp_sector + INTLV
						&& bp->hp_sector < p2->hp_sector - INTLV)
							break;
					}
					else
					{
						if(bp->hp_sector > p1->hp_sector + INTLV
						|| bp->hp_sector < p2->hp_sector - INTLV)
							break;
					}
					p1 = p2;
					if( !(p2 = p1->av_forw) ) break;
				}
				break;
			}

#if	SWEEP
			else
			{
				if (dirf == UP)
				{
					if(p2->hp_cyl < p1->hp_cyl)
						if(bp->hp_cyl > p1->hp_cyl)
							break;
						else
							dirf = DOWN;
				}
				else
				{
					if(p2->hp_cyl > p1->hp_cyl)
						if(bp->hp_cyl < p1->hp_cyl)
							break;
						else
							dirf = UP;
				}
			}
#endif
		}

		bp->av_forw = p2;
		p1->av_forw = bp;

		while(p2)
		{
			/* count down overtaken blocks */
			p2->hp_age--;
			p2 = p2->av_forw;
		}
	}

	spl0();
}

/*
 *	start seeks as required and possibly one data transfer.
 */
hpstart()
{
	/* called at HPSPL or greater */

	register struct buf *bp;
	register struct hpq *qp;
	register int	n;

	static int	drv, ioage;	/* we assume that these are zero initially */


#if	PRESEEK
	for( n = NDRV; --n >= 0;)
	{
		qp = &hpq[n];
		if( (bp = qp->hpq_bufp) && (qp->hpq_flags == HPIDLE) )
		{
			/*
			**	for all available drives start seeking
			*/
			HPADDR->hpcs2 = n;
			if(HPADDR->hpcc == bp->hp_cyl)
			{
				qp->hpq_flags = HPREADY;
			}
			else
			{
				int xx;
#if	SWEEP
				if(bp->hp_cyl > HPADDR->hpcc)
					qp->hpq_dirf = UP;
				else if(bp->hp_cyl < HPADDR->hpcc)
					qp->hpq_dirf = DOWN;
#endif
				xx = bp->hp_sector - INTLV;
				if( xx < 0 ) xx += 22;
				xx.hibyte = bp->hp_track;
				HPADDR->hpda = xx;
				HPADDR->hpdc = bp->hp_cyl;
				qp->hpq_flags = HPSEEK;
				HPADDR->hpcs1.lobyte = IE | SEARCH | GO;
			}
		}
	}
#endif
	/*
	 *	check if possible to start an io
	 */
	for( n = NDRV; --n >= 0;)
		if(hpq[n].hpq_flags == HPBUSY ) return;

	if(HPADDR->hpcs1 & RDY)	/* ensure controller available */
	{
		/*
		**	try to start an IO
		*/
		n = NDRV;
		do
		{
			qp = &hpq[drv];
			if( (bp = qp->hpq_bufp) && qp->hpq_flags == HPREADY)
			{
#if	HPSTATS
				int a;

#endif
				qp->hpq_flags = HPBUSY;
				HPADDR->hpcs2 = drv;
				HPADDR->hpdc = bp->hp_cyl;
#if	HPSTATS
				a = bp->hp_sector * 4;
				a -= HPADDR->hpla >> 4;
				if( a <= 3 ) a += 88;
				hpintlv[a-4]++;
#endif
				{
					unsigned com;

					HPADDR->hpda = bp->hp_trksec;
					com = HPADDR->hpbae = bp->b_xmem;
					HPADDR->hpba = bp->b_addr;
					HPADDR->hpwc = -(bp->b_bcount >> 1);
					com = IE | GO | ((com & 03) << 8);
					if (bp->b_flags & B_READ)
						com |= READ;
					else
						com |= WRITE;
					HPADDR->hpcs1 = com;
				}
				if( --ioage <= 0)
				{
					ioage = HPAGE;
					if( ++drv >= NDRV)
						drv = 0;
				}
				return;
			}
			if( ++drv >= NDRV) drv=0;
			ioage = HPAGE;
		} while (--n > 0);
	}

	if( !(HPADDR->hpcs1 & IE))
		for(n=0; n<NDRV; n++)
		{
			/*
			 *	if the IE bit has not been set,
			 *	then nothing is happening and nothing
			 *	was started, so find a drive which will accept
			 *	a command and set the IE bit with a nop
			 */

			HPADDR->hpcs2 = n;
			if(HPADDR->hpds & DRY)
			{
				HPADDR->hpcs1 = IE;
				return;
			}
		}

}


hpintr()
{
	/* called at HPSPL or greater */

	register int	n;
	register struct hpq *qp;
	register struct buf *bp;

	/*
	**	An error has occured and/or a data transfer has completed.
	*/


	for(n=0; n<NDRV; n++)
	{
		HPADDR->hpcs2 = n;	/* select drive */
		qp = &hpq[n];
		if( (HPADDR->hpds & ATA) || qp->hpq_flags == HPBUSY)
		{
			bp = qp->hpq_bufp;
			if( (HPADDR->hpds & MOL) && !(HPADDR->hpds & VV) )
			{
				/* drive has come online */
				printf("HP drive %d turned on\n", n);
				hpdinit(n);
			}
			else if( !(HPADDR->hpds & MOL) )
			{
				/* drive down - disable and flag */
				printf("HP drive %d turned off\n", n);
				qp->hpq_flags = HPOFF;
#if	DROP_OFFLINEQ
		dropq:
				while(bp)
				{
					bp->b_flags |= B_ERROR;
					iodone(bp);
					bp = bp->av_forw;
				}
				qp->hpq_bufp = 0;
#endif
			}
			else if( HPADDR->hpds & ERR)
			{
				/* a drive error */
				int	er1;	/* saves the error register */

				hpregs(qp);
				er1 = HPADDR->hper1;
				HPADDR->hpcs1.lobyte = IE | DCLR | GO;
				while( !(HPADDR->hpds & DRY));
				if( er1 & (UNS | DTE | OPI) )
				{
					if( HPADDR->hper1 & UNS)
					{
						/* Still unsafe, unload for safety */
						qp->hpq_flags = HPOFF;
						HPADDR->hpcs1.lobyte = IE | UNLOAD | GO;
						printf("HP drive %d UNSAFE\n", n);
						printf("\nSTOP and re-START drive to recover\n\n");
#if	DROP_OFFLINEQ
						goto dropq;
#endif
					}
					else
					{
						/* Okay now, recal and go */
						qp->hpq_flags = HPRECAL;
						HPADDR->hpcs1.lobyte = IE | RECAL | GO;
					}
				}
#if	HPECC
				else if( (er1 & DCK) && !(er1 & ECH))
				{
					if( hpecc() )
						continue; /* IO resumed */
					else
						HPADDR->hpcs1 = TRE | IE;
						goto trcmplt; /* IO done */
				}
#endif
				switch(qp->hpq_flags)
				{
				case HPOFF:
				case HPREADY:
				case HPIDLE:
					break;
				case HPBUSY:
					HPADDR->hpcs1 = TRE | IE;
					if( er1 & AOE )
					{
						/* This might occur if hp_sizes wrong */
						HPADDR->hpcs1 = TRE | IE;
						goto trcmplt;
					}
					else if( er1 & WLE )
					{
						/* Drive is wrt protected - give up */
						bp->b_flags |= B_ERROR;
						goto unlink;
					}
#if	PRESEEK
				case HPSEEK:
					qp->hpq_flags = HPIDLE;
#else
					qp->hpq_flags = HPREADY;
#endif
					goto errec;
				case HPRECAL:
					if( qp->hpq_bufp )
						goto errec;
				}
			}
			else
			{
				/*
				 * Operation complete. Transfer or seek/recalibrate
				 */
				if( qp->hpq_flags == HPBUSY)
				{
					if(HPADDR->hpcs1 & RDY)
					{
						if( HPADDR->hpcs1 & TRE )
						{
							hpregs(qp);
							HPADDR->hpcs1 = TRE | IE;
				errec:
							if( ++bp->b_error >= 10 )
							{
								bp->b_flags |= B_ERROR;
								goto unlink;
							}
						}
						else
						{
							/* Transfer complete SUCCESS! */
				trcmplt:
							bp->b_resid = HPADDR->hpwc;
				unlink:
							qp->hpq_bufp = bp->av_forw;
							iodone(bp);
						}
					}
					else
					{
						continue;
					}
				}
#if	PRESEEK
				if( qp->hpq_flags == HPSEEK )
					qp->hpq_flags = HPREADY;
				else if(qp->hpq_flags != HPRECAL)
					qp->hpq_flags = HPIDLE;
#else
				if(qp->hpq_flags != HPRECAL)
					qp->hpq_flags = HPREADY;
#endif
			}
	clear:
			HPADDR->hpas = 1 << n;
		}
	}
	hpstart();

}

hpregs(qp)
struct hpq	*qp;
{
	static struct hperrmsgs
	{
			char *str;	/* identify which register */
			int  *reg;	/* address of device register */
	}
	hperrmsgs[]
	{
			"ER1", &HPADDR->hper1,
			"ER2", &HPADDR->hper2,
			"ER3", &HPADDR->hper3,
			"CS1", &HPADDR->hpcs1,
			"DS",  &HPADDR->hpds,
			"CS2", &HPADDR->hpcs2,
			"WC",  &HPADDR->hpwc,
			"BA",  &HPADDR->hpba,
			"DC",  &HPADDR->hpdc,
			"DA",  &HPADDR->hpda,
			"AS",  &HPADDR->hpas,
			0
	};
	register struct hperrmsgs *p = hperrmsgs;

	printf("HPERR ");
	do
	{
			printf("%s=0%o, ", p->str, *(p->reg) );
			p++;
	} while( p->str );
	printf("FLAGS=0%o\n",qp->hpq_flags);
}

/*
 *	Physical IO:
 *	truncate transfers at the ends of logical volumes
 */

hpread(dev)
int	dev;
{
	hpphys( dev , B_READ );
}

hpwrite(dev)
int	dev;
{
	hpphys( dev , B_WRITE );
}

hpphys( dev , flag )
int	dev;
int	flag;	/* B_READ or B_WRITE */
{
	register unsigned c;
	daddr_t a, b;

	a = hp_sizes[minor(dev)].nblocks;
	b = u.u_offset >> 9;
	if(b >= a)
	{
		u.u_error = ENXIO;
		return;
	}
	a -= b;
	c = u.u_count;
	if ((c - 1) / 512 >= a)
		c = a << 9;	/* a * 512 */
	a = u.u_count - c;
	u.u_count = c;
	physio(hpstrategy, &hpbuf, dev, flag);
	u.u_count += a;
}

hpdinit(n)
register n;
{
	/* called at HPSPL or greater */

	HPADDR->hpcs2 = n;
	if(HPADDR->hpds & MOL)	/* this may generate a NED error */
				/* hpintr will handle it */
	{
		HPADDR->hpcs1.lobyte = IE | DCLR | GO;
		/*
		**	possibly some risk in these while loops here and
		**	in previous code in that if the drive is maybe
		**	MOL & !(DRY) ever....... not very likely and you
		**	would be buggered if it was.....
		*/
		while( !(HPADDR->hpds & DRY) );
		HPADDR->hpcs1.lobyte = IE | PRESET | GO;
		while( !(HPADDR->hpds & DRY) );
		HPADDR->hpof = FMT22;
#if	PRESEEK
		hpq[n].hpq_flags = HPIDLE;
#else
		hpq[n].hpq_flags = HPREADY;
#endif
	}
}
