/*
 *	cr11 card reader driver for standard 12 * 80 punched cards
 *
 *	Card rows are numbered (from the top) 12 11 0 1 2 3 4 5 6 7 8 9
 *
 *	This driver supports a locally-modified buffered card reader in use
 *	at the University of NSW. All columns of the card just read are
 *	available by doing successive operations on the device registers.
 *	Two images are available: a 12-bit binary image (in an int), and
 *	a 8-bit coded image. The 12-bit image has a bit set for each hole
 *	in the card, with bit 0 representing row 9 up to bit 11 for row 12.
 *	The 8-bit coded image provides rows 8, 9, 0, 11, and 12 in bits
 *	3, 4, 5, 6, and 7. As normal alphabetic punchings have only one
 *	of bits 1 though 7 punched (or none at all - 8 possibilities),
 *	bits 0 through 2 code the row number with the punching.
 *
 *	The minor device number gives mode (bit 0: 0 for binary out,
 *	1 for ASCII out) and reader number (bits 1 through 7).
 */

#include <param.h>
#include <file.h>
#include <buf.h>
#include <dir.h>
#include <signal.h>
#include <user.h>
#include <errno.h>

extern struct user u;

#define CRIPRI	(PZERO+20)	/* sleeping priority */
#define NCOLS	80		/* no. of columns    */
#define SPLCR	spl4		/* hardware priority level */

/*
 *	card reader status bits
 */
#define ERROR		0100000
#define CARDONE		0040000		/* same as CBUFFER at UNSW */
#define HOPPER		0020000		/* not used here */
#define MOTION		0010000
#define CBUFFER		0004000
#define ONLINE		0002000		/* inverse of NOTREADY at UNSW */
#define BUSY		0001000		/* not used here */
#define NOTREADY	0000400		/* not used here */
#define COLDONE		0000200		/* not used directly here */
#define IENABLE		0000100
#define CLEAR		0000002
#define READ		0000001

/*
 *	card reader device registers
 */
struct
{
	int crstatus;	/* status register */
	int crbbin;	/* 12 bit binary buffer */
	int crbcode;	/* 8 bit encoded buffer */
};

struct
{
	char crstat0, crstat1;	/* chars of the status register */
	char crbbin0, crbbin1;	/* the two bytes of the binary buffer */
};

/*
 *	card reader handler status
 */
#define CLOSED		0
#define READING		1
#define CARDREAD	2
#define ENDFILE		3
#define VALERR		4

/*
 *	card reader mode
 */
#define o29	0
#define o26	1
#define BINARY	2

/*
 *	status info for card reader(s)
 */
struct cr11
{
	int *craddr;	/* device register address */
	char crstate;
	char crmode;
	int crvalerr;	/* count of validity errors */
	int crrdchk;	/* count of read checks */
	int crchars;	/* number of characters still in buffer */
	char *nextch;		/* buffer position pointer */
	char buffer[2 * NCOLS + 1];	/* one card buffer */
}
cr11[] =
{
    {
	0177160,	/* craddr */
	CLOSED,		/* crstate */
	o29,		/* crmode */
    },
};
#define NCR11	(sizeof cr11 / sizeof cr11[0])	/* no. of cr11 */

/*
 *	special character codes
 */
#define EOI	017	/*		  6 7 8 9 */
#define	CSUEOI	07417	/* 12 11 0 1	  6 7 8 9 */
#define EOF	015	/*		  6 7   9 */
#define EOR	007	/*		    7 8 9 */
#define CNV	025	/*		5   7   9 */

#define CNTRLF	'\06'	/* EOF returns ^f */
#define CNTRLR	'\022'	/* EOR returns ^r */

/*
 *	validity checking table
 */
int crvalid[8] =
{
	0, 1<<8, 1<<7, 1<<6, 1<<5, 1<<4, 1<<3, 1<<2
};

char crcd26[] = "\0+=()'^#>?;<[]`\"&:!?^@%";	/* care - null at end */
char crcd29[] = "\0&#%<@);^(!{\"\\_>?]}[+'=";	/* care - null at end */

/*
 *	coded to ascii translation table
 *	An entry of 0 indicates an invalid punching.
 *	A negative entry indicates an index into the o26/o29 arrays above.
 */
char crtab[256]	=
{
	' ','1','2','3','4','5','6','7',	/*		*/
	'8','`',':',-2 ,-5 ,-21,-22,-12,	/*	     8	*/
	'9', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*	   9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , CNTRLR,	/*	   9 8	*/
	'0','/','s','t','u','v','w','x',	/*	 0	*/
	'y','^',-13,',',-3 ,-14,-15,-16,	/*	 0   8	*/
	'z', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*	 0 9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*	 0 9 8	*/
	'-','j','k','l','m','n','o','p',	/*    11	*/
	'q','|',-17,'$','*',-6 ,-7 ,-8 ,	/*    11     8	*/
	'r', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*    11   9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*    11   9 8	*/
	-18,'~','S','T','U','V','W','X',	/*    11 0	*/
	'Y', 0 , 0 , 0 , 0 , 0 ,'\t',0 ,	/*    11 0   8	*/
	'Z', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*    11 0 9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/*    11 0 9 8	*/
	-1 ,'a','b','c','d','e','f','g',	/* 12		*/
	'h', 0 ,-19,'.',-4 ,-9 ,-20,-10,	/* 12	     8	*/
	'i', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12	   9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12	   9 8	*/
	-11,'A','B','C','D','E','F','G',	/* 12	 0	*/
	'H', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12	 0   8	*/
	'I', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12	 0 9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12	 0 9 8	*/
	 0 ,'J','K','L','M','N','O','P',	/* 12 11	*/
	'Q', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11     8	*/
	'R', 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11   9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11   9 8	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11 0	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11 0   8	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11 0 9	*/
	 0 , 0 , 0 , 0 , 0 , 0 , 0 , 0 ,	/* 12 11 0 9 8	*/
};



cropen(dev, flag)
{
	register struct cr11 *cr;
	register mode;

	mode = dev & 01;
	dev = minor(dev) >> 1;
	if ((flag & FWRITE) || (dev >= NCR11))
	{
		u.u_error = ENXIO;
		return;
	}
	cr = &cr11[dev];
	if (cr->crstate != CLOSED)
	{
		u.u_error = EEXIST;
		return;
	}
	cr->crmode = mode ? o29 : BINARY;
	cr->crchars = 0;
	crstart(cr);
}

crclose(dev, flag)
{
	register struct cr11 *cr;

	dev = minor(dev) >> 1;
	cr = &cr11[dev];
	cr->craddr->crstatus = 0;
	cr->crstate = CLOSED;
}

crint(dev)
{
	register struct cr11 *cr;
	register int *devreg;
	register int status;

	dev = minor(dev) >> 1;
	cr = &cr11[dev];
	devreg = cr->craddr;
	status = devreg->crstatus;
	if (status & (ERROR|CARDONE|ONLINE))
	{
		devreg->crstatus = IENABLE;	/* clear interrupt*/
		if (status & MOTION)
			cr->crrdchk++;
		else if ((status & CARDONE) == CARDONE)
		{
			devreg->crstatus = 0;
			cr->crstate = CARDREAD;
			wakeup(cr);
		}
		else if (status & ONLINE)
			crstart(cr);
	}
}

crread(dev)
{
	register struct cr11 *cr;
	register int *devreg;
	register char *pch;
	char qch;
	char *lastnonblank;

	dev = minor(dev) >> 1;
	cr = &cr11[dev];
	if (cr->crchars == 0)
	{
		if (cr->crstate == ENDFILE)
			crstart(cr);
loop:
		SPLCR();
		while (cr->crstate != CARDREAD)
			sleep(cr, CRIPRI);
		spl0();
		/*
		 * a card has been read
		 */
		pch = cr->buffer;
		devreg = cr->craddr;
		lastnonblank = pch - 1;
		do
		{
			if (cr->crmode == BINARY)
			{
				*pch++ = devreg->crbbin0;
				*pch++ = devreg->crbbin1;
			}
			else
			/*
			 * ascii conversion - 1st validity check
			 */
			{
				if (crvalid[devreg->crbcode & 07] != (devreg->crbbin & 0774))
				{
					switch(devreg->crbbin)
					{
					case EOI:
					case CSUEOI:
						cr->crstate = ENDFILE;
						cr->crmode = o29;
						return;
					case EOF:
						*pch = CNTRLF;
						lastnonblank = pch++;
						break;
					case CNV:
						if (pch == cr->buffer)
						{
							crrdcol(devreg);
							cr->crmode = devreg->crbbin == 0;
							crstart(cr);
							goto loop;
						}
					default:
						crerror(dev, cr);
						goto loop;
					}
				}
				else
				/*
				 * valid - translate
				 */
				{
					qch = crtab[devreg->crbcode & 0377];
					/*
					 * check o26 or o29 specials
					 */
					if (qch == 0)
					{
						crerror(dev, cr);
						goto loop;
					}
					if (qch < 0)
						if (cr->crmode)
							*pch = crcd26[-qch];
						else
							*pch = crcd29[-qch];
					else
						*pch = qch;
					if (*pch != ' ')
						lastnonblank = pch;
					pch++;
				}
			}
		} while (pch < &cr->buffer[NCOLS*2] && crrdcol(devreg));

		if (cr->crmode != BINARY)
		{
			pch = lastnonblank + 1;
			*pch++ = '\n';
		}
		cr->nextch = cr->buffer;
		cr->crchars = pch - cr->buffer;
		if (cr->crstate != ENDFILE)
			crstart(cr);
	}
	crpass(cr);
}

crstart(cr)
register struct cr11 *cr;
{
	register int *devreg;

	cr->crstate = READING;
	/*
	 * flush the buffer in card reader
	 */
	devreg = cr->craddr;
	devreg->crstatus = CLEAR;
	devreg->crstatus = IENABLE|READ;
	/*
	 * card in motion
	 */
}

crerror(dev, cr)
register struct cr11 *cr;
{
	register int *devreg;
	register dummy;

	devreg = cr->craddr;
	cr->crstate = VALERR;
	cr->crvalerr++;
	uprints("Invalid punch - reread last card\n");
	dummy = devreg->crbbin;		/* clear COLDONE */
	devreg->crstatus = IENABLE|CLEAR;
}

crpass(cr)
register struct cr11 *cr;
{
	register int count;

	/*
	 * send back the min of cols on card and use count
	 */
	count = (cr->crchars < u.u_count ? cr->crchars : u.u_count);
	iomove(cr->nextch, count, B_READ);
	cr->crchars -= count;
	cr->nextch += count;
}

/*
 * crrdcol(devreg)
 * register int *devreg;
 * {
 * 	int x, y;
 * 
 * 	devreg->crstatus = READ;	/* shift */
/* 	/*
 * 	 * wait a microsecond or 5
 * 	 */
/* 	x = y;		/* waste the required time */
/* 	if (devreg->crstatus & CBUFFER)
 * 		return(1);
 * 	else
 * 		return(0);
 * }
 */

#define	GIVE_UP	100	/* testing... */
int cr_hiwat;

crrdcol(devreg)
register int *devreg;
{
	register i;

	devreg->crstatus = READ;	/* shift */
	i = 0;
	while ((devreg->crstatus & (CBUFFER|COLDONE)) == CBUFFER)
		if (i++ == GIVE_UP)
		{
			printf("cr timeout\n");
			return(0);
		}
	if (devreg->crstatus & COLDONE)
	{
		if (i > cr_hiwat)
			cr_hiwat = i;
		return(1);
	}
	return(0);
}
