/
/	unix interface for spitbol
/
/	this module contains routines for the unix interface to the
/	pdp-11 macro spitbol compiler. execution of the compiler begins
/	in the interface where all initialization is performed. all
/	unix system functions are handled by subroutines in this module.
/	these subroutines have five character names starting with "sys";
/	they appear in alphabetical order starting with sysbx. termination
/	of the compiler is also handled by this module.
/
/	initialization of the compiler consists of setting up the stack
/	the dynamic area, and processing of any options for the compiler.
/	the dynamic area is carved out of the stack space at the top of
/	memory and is initially cleared.
/
/	the command line has the form:
/
/		spitbol [options] [<standard_input] [>standard_output]
/
/	where all specifications are optional.
/
/	options to the compiler are:
/
/		-e	don't send errors to terminal
/		-l	generate compiler listing
/		-c	generate compilation statistics
/		-x	generate execution statistics
/		-a	like -l -c -x
/		-p	long listing format; generates form feeds
/		-n	suppress execution (like -NOEXEC control card)
/		-mdddd	maximum size in bytes of created objects
/		-sdddd	size in words of stack space
/
/	defaults:
/
/	standard output comes only from assignments to OUTPUT or
/	from compiler error messages.
/
/		-m8192	see "defmax"
/		-s1024	see "defstk"
/
/	also, the interface determines if the terminal is the standard
/	output file and sets a compiler flag that keeps listing output
/	short. see section: tsttty.
/
/	the interface handles two modes of input/output transfers:
/
/		line mode, where records are delimited by new line
/		characters. the interface strips them on input and
/		appends them on output.
/
/		raw mode, where a predetermined number of bytes
/		are transferred.
/
/	the particular mode to be used is specified by the programmer
/	in the INPUT/OUTPUT function call and is processed by the
/	interface. the maximum length input record is determined by the
/	-l or -r argument. the length of an output record is determined
/	by the length of the string assigned to the output associated
/	variable, plus one if in line mode.
/
/	the form of the INPUT/OUTPUT function call is
/
/		INPUT/OUTPUT(.name,file_descriptor,file_name args)
/
/	where	name is the variable name to be input/output associated
/
/		file_descriptor is an integer between 4 and 14 inclusive
/		and is the actual file descriptor to be used.
/
/		file_name args is the name of the file to be used and
/		any associated arguments for the association. the file
/		name must appear before the args, and one or more spaces
/		must separate them.
/
/	file arguments:
/
/		-a	append output at end of existing file. if file
/			doesn't exist it is created. if -a is not specified
/			then the file is created.
/		-bdddd	set buffer size to dddd characters.
/		-c	like -r1.
/		-ldddd	line mode: maximum input record length is dddd
/			characters, length of output record is length
/			of string assigned to output associated variable,
/			plus one if line mode (for new line).
/		-rdddd	raw mode: maximum input record length is dddd
/			characters; length of output record is length
/			of string assigned to output associated variable.
/		-w	write records without buffering; this is useful in
/			writing to terminals.
/
/		defaults:  -b512 -l512
/			(size of disk block (blksiz) is 512)
/
/	more than one type of transfer may be associated with a file.
/	this is accomplished by calling INPUT/OUTPUT after the initial
/	call with the name, file descriptor, and file arguments. the
/	file name must not be specified on calls subsequent to the
/	first!
/
/	the interface allocates blocks within spitbol's dynamic area
/	for use by the input/output routines. these blocks conform to
/	spitbol's standards, a type flag appears in the first word of
/	the block and for these types of blocks, the length in bytes
/	of the block is stored in the second word of the block.
/
/	the ioblk is the control area for all input/output and is
/	pointed to by the compiler variable rzfil. within the ioblk
/	is an ioelt for each file descriptor that might be used in
/	an input/output association. there are three filelds in each
/	ioelt: a pointer to the file name, a pointer to the file's
/	buffer, and word of flags. both the file name and buffer are
/	blocks within spitbol's dynamic area.
/
/	for each call to INPUT/OUTPUT it is possible for a fcblk
/	to be created. within the fcblk is the file descriptor to be
/	used and the transfer mode. as previously mentioned, more than
/	one fcblk may be active per file. the fcblk is passed to the
/	interface on all input/output calls.
/
/	blocks allocated by the interface come in two flavors: external
/	relocatable (xrblk's) and external non-relocatable (xnblk's).
/
/	xrblk's contain words that are treated as addresses by spitbol.
/	thus, these addresses must conform to spitbol's standard that
/	if an address points within the dynamic area, it must point to
/	the start of a block. it is permissible for words to contain
/	addresses that do not lie in spitbol's dynamic area.
/
/	xnblk's are treated as raw data. no word within an xnblk should
/	point to another dynamic area block, since spitbol will not
/	relocate any fields in an xnblk during garbage collection.
/
/	the interface allocates the ioblk as an xrblk, bfblk an an xnblk,
/	and the fcblk as an xnblk.
/
/	when spitbol calls an interface subroutine, exit parameters for
/	the call follow the jsr instruction. an exit parameter is the
/	address of an instruction to be executed if the exit parameter
/	is taken.
/
/	thus for a succesful return from an interface routine:
/
/	add	$no_parms*2,(sp) / point return address past parms
/	rts	pc		/ return
/
/	to take parameter exit n:
/
/	add	$n-1*2,(sp)	/ point return address at parm n
/	mov	*(sp)+,pc	/ move parm address into pc
/
/	note that spitbol handles its internal errors by transfering
/	control to an odd address. this is trapped by the interface
/	and the error code is computed from the trap_pc. see sections
/	tstmod and ostrap. the text for the error messages is kept
/	in a file and is accessed by a subroutine call; see sysem.
/
/
/	register equates (for compatibility with compiler)
/
cp	= r0
wa	= r1
wb	= r2
wc	= r3
xl	= r4
xr	= r5
ia	= wc
/
/	parameter offsets
/
p1	= 2
p2	= 4
p3	= 6
p4	= 10
regs	= 14
/
/	offsets to registers on stack
/
cp.off	= 0
wa.off	= 2
wb.off	= 4
wc.off	= 6
xl.off	= 10
xr.off	= 12
/
/	globals referenced in compiler
/
.text
	.globl	sec04,sec05,sec06
	.globl	bzxnt,bzxrt,rzfil
/
/	process any options specified on command line. on entry
/	sp points to word containing number of options found
/	including the program name. following this word are pointers
/	to the ascii text for each option.
/
unxint: mov	sp,r5		/ use r5 instead of sp
	mov	(r5)+,r4	/ r4 = number of options specified
	tst	(r5)+		/ program name not interesting
	br	3f		/ process others
0:	mov	(r5)+,r0	/ r0 -> option text
	movb	(r0)+,curopt	/ pick up both bytes of option
	movb	(r0)+,curopt+1	/  and store in curopt
	mov	$opttbl,r1	/ r1 -> option table
1:	cmp	curopt,(r1)	/ ?have we found right one?
	beq	2f
	add	$optsiz,r1	/ point to next entry
	br	1b		/  and try again (will always match)
2:	jsr	pc,*optrtn(r1)	/ call option processing routine
3:	sob	r4,0b		/ loop thru all options
/
/	if the standard output file (printer) is the terminal then
/	turn on the "printer is terminal" flag in sptflg. this keeps
/	title/page lines short and prevents two copies of error
/	messages from going to terminal.
/
tsttty: mov	$1,r0		/ get the tty status
	sys	gtty ; temp	/  for file descriptor 1
	bcs	4f		/ if file descriptor 1 is a terminal
	bis	$prtich,sptflg	/  then std printer is terminal
4:
/
/	determine what happens to the pc when an odd address trap occurs.
/	this allows correct computation of error codes.
/
tstmod: sys	signal ; 10. ; 1f
				/ issue signal to catch trap
	mov	$077777,pc	/ cause odd address trap (bus error)
1:	tst	(sp)		/ if (sp) > 0
	bmi	2f
	mov	$1,pmodel	/  then set pmodel to 1
2:				/ (else set pmodel to -1, default)
/
/	in order to obtain as large a dynamic area as possible
/	the stack is extended down until unix generates a
/	segment fault. then the sp is backed off stkwds words
/	to provide a run time stack. if the user specified too
/	large a -s value it is defaulted to defstk.
/
setsp:				/ ---------- the start of the mod -------
/	clr	sp		/ start at very top of memory
/	sys	signal ; 11. ; 1f
/				/ trap segmentation error
/0:	clr	-(sp)		/ push until unix stops me
/	br	0b
/	well the above is just fine except it is boringly slow
/	lots of expand and swaps etc ... we want this bit to be fast
/	lets say that it is at least 20 times faster in getting going !!
/	(with the process sticky bitted too .... )
/	METHOD:
/		start at 1st 8kb boundary beyond end of data and
/		work up through memory until no segmentation violations
/		occur. Guaranteed to succeed in one (maybe two) goes !!
/
/					ian inc.
/
	.globl	_end
	mov	$_end+17777,r3		/ round to start
	bic	$17777,r3		/ next 8kb
	sub	$76,r3			/ adjust - really add 2 so clr -(sp) ok
1:					/ since 100 is added below
	add	$100,r3			/ step to next higher core tick
	sys	signal ; 11. ; 1b	/ catch segmentation errors
	mov	r3,sp			/ address needs to be in stack reg
	clr	-(sp)			/ try it on for size
	sys	signal ; 11. ; 0	/ well thats that then
/1:					/--------- the end of the mod ------
	mov	sp,spsave	/ save current sp
	asl	stkwds		/ cvt words to bytes
	add	stkwds,sp	/ bump over reserved space
	bcc	2f		/ if no carry then ok
	mov	spsave,sp	/  else default to
	add	$defstk*2,sp	/  defstk words of stack
2:	add	$100.*2,spsave	/ set spsave for chk processing
/
/	set pointers for top and bottom of dyanmic area and trap
/	bus errors - then off to the compiler.
/
setdyn: mov	sp,xr		/ xr -> bottom of dynamic area
	mov	$177776,xl	/ xl -> top of dynamic area
	sys	open ; errnam ; 0
				/ open error text file
	bcs	0f		/  skip next if error
	mov	r0,errfcb	/ store link number
0:	sys	signal ; 10. ; ostrap
				/ trap bus errors
	jmp	sec04		/ start compiler
/
/	on bus error transform odd pc into error code:
/
/		error code = (trap_pc + pmodel) / 2
/
ostrap: mov	(sp)+,wa	/ wa  = odd pc
	add	pmodel,wa	/  + pmodel
	asr	wa		/  divided by 2
	tst	(sp)+		/ remove useless ps
	sys	signal ; 10. ; ostrap
				/ reset intercept
	jmp	sec06		/ handle error
/
/	when there are less than one hundred words of stack remaining
/	jump to stack overflow section.
/
	.globl	chk
chk:	cmp	sp,spsave	/ if sp > spsave
	blo	0f
	rts	pc		/  then return - no stack overflow
0:	tst	(sp)+		/  else remove return address
	jmp	sec05		/	jump to stack overflow section
/
/	the following routines are used during option processing.
/
/	optclr - clear a option value
/
optclr: bic	optflg(r1),sptflg	/ reset flag
	rts	pc
/
/	optset - set a option vaule
/
optset: bis	optflg(r1),sptflg	/ set flag
	rts	pc
/
/	optnum - set numeric value from option
/
optnum: mov	r0,-(sp)	/ address of numeric string
	jsr	pc,getnum	/ number is returned in ia(r3)
	tst	(sp)+		/  not interested in scan address
	tst	r3		/ if value is zero or negative
	ble	opterr		/  then treat as error
	mov	r3,*optflg(r1)	/  else store value
	rts	pc
/
/	opterr - indicate option error
/
opterr: mov	$2,r0		/ output error message to
	sys	write ; curopt ; 4	/  file descriptor 2
	rts	pc
/
/	sysbx - before execution setup
/
/	sysbx is called after compilation and before execution to
/	allow the interface to do any required setup.
/
/	jsr	pc,sysbx	/ setup before execution
/
	.globl	sysbx
sysbx:	rts	pc		/ nothing to do
/
/	sysdc - date check
/
/	sysdc is not used by this implementation.
/
/	jsr	pc,sysdc	/ call to do date check
/
	.globl	sysdc
sysdc:	rts	pc
/
/	sysdm - dump memory
/
/	sysdm is not currently supported.
/
/	(xr)			/ arg n from DUMP(n)
/	jsr	pc,sysdm	/ call to dump memory
/
	.globl	sysdm
sysdm:	rts	pc
/
/	sysdt - get current date
/
/	sysdt is not currently supported.
/
/	jsr	pc,sysdt	/ call to get date (and time)
/	(xl)			/ date scblk pointer
/
	.globl	sysdt
sysdt:	mov	$temp,xl
	clr	sclen(xl)	/ return null string
	rts	pc
/
/	sysef - eject file
/
/	sysef is called to write a page eject to a file.
/
/	(wa)			/ fcblk pointer
/	(xr)			/ file arg1 pointer
/	jsr	pc,sysej	/ call to eject file
/		loc		/ file doesn't exist
/		loc		/ inapropiate file
/		loc		/ i/o error
/
	.globl	sysef
sysef:	jsr	r5,..sav	/ save registers
	jsr	pc,osopen	/ open file
	bcs	efer1		/ handle open error
	bit	$io.out,ioflg(wc)
	beq	efer2		/ file must be open for output
	mov	$ejcblk,xr	/ xr -> eject scblk
	jsr	pc,oswrit	/ write eject
	bit	$io.err,ioflg(wc)
	bne	efer3		/ handle i/o error
	jsr	r5,..rst	/ restore registers
	add	$6,(sp)
	rts	pc		/ return
/
efer1:	jbr	erxit1		/ file doesn' exist
efer2:	jbr	erxit2		/ inapropriate file
efer3:	jbr	erxit3		/ i/o error
/
/	sysej - end of job
/
/	sysej is called at end of job. (return is not possible.)
/	any files open for output must have their buffers flushed
/	before termination. the code keyword value is returned as
/	a status code to unix.
/
/	(wa)			/ value of abend keyword
/	(wb)			/ value of code keyword
/	jsr	pc,sysej	/ call to end job
/
	.globl	sysej
sysej:	mov	wb,-(sp)	/ save code value on stack
	mov	rzfil,wc	/ wc -> ioelt
	beq	ejxit		/ skip if no ioblk
	mov	$nmdesc,wa	/ wa  = number ioelts
	mov	$lodesc,svdesc	/ set starting file descriptor
0:	bit	$io.opn,ioflg(wc)
	beq	1f		/ skip if file not open
	bit	$io.out,ioflg(wc)
	beq	1f		/ skip if file open for input
	jsr	pc,flush	/ flush buffer for file
1:	add	$ioelt,wc	/ wc -> next ioelt
	inc	svdesc		/ increment file descriptor
	sob	wa,0b		/ loop thru all ioelts
ejxit:	mov	(sp)+,cp	/ return code keyword as status
	sys	exit		/ exit to system
/
/
/	sysem - get error message text
/
/	sysem is called to obtain the error message text for an
/	error code.
/
/	(wa)			/ error code
/	jsr	pc,sysem	/ call to get error message text
/	(xr)			/ error message text scblk pointer
/
	.globl	sysem
sysem:	jsr	r5,..sav	/ save regs
	mov	$temp,xr	/ xr -> scblk for error text
	dec	wa		/ errors are stored 0-origin
	mul	$53.,wa		/  in 53 byte chunks
	mov	wa,0f		/ store offset for seek
	mov	errfcb,r0	/ get error file link
	beq	em1		/  ?if 0 then return null?
	sys	indir; 9f
.data
9:	sys	seek ; 0: 0 ; 0 / point to proper error text
.text
	sys	read ; temp+4 ; 52.
				/ read error text
em1:	mov	r0,sclen(xr)	/ set number of bytes read
	mov	xr,xr.off(sp)	/ return pointer to scblk in xr
	jsr	r5,..rst	/ restore regs
	rts	pc
/
/	sysen - endfile
/
/	sysen is called to close a file. no further i/o operations
/	referencing the file will be preformed. all ioelt fields
/	for the file are cleared.
/
/	(wa)			/ fcblk pointer
/	(xr)			/ file arg1 pointer
/	jsr	pc,sysen	/ call to do endfile
/		loc		/ file doesn't exist
/		loc		/ endfile not allowed
/		loc		/ i/o error
/
	.globl	sysen
sysen:	jsr	r5,..sav	/ save registers
	jsr	pc,osopen	/ let osopen set up fields
	bcs	ener1		/ handle open errors
	bit	$io.out,ioflg(wc)
	beq	0f		/ if open for output
	jsr	pc,flush	/  then flush file before close
	bcs	ener3		/  handle possible i/o error
0:	mov	svdesc,cp	/ load file descriptor in cp
	sys	close		/ close file
	clr	iofnm(wc)	/ clear file name pointer
	clr	iobuf(wc)	/ clear buffer pointer
	clr	ioflg(wc)	/ clear flags
	jsr	r5,..rst	/ restore registers
	add	$6,(sp)
	rts	pc		/ return
/
ener1:	jbr	erxit1		/ file doesn't exist
ener2:	jbr	erxit2		/ endfile not allowed
ener3:	jbr	erxit3		/ i/o error
/
/	sysep - eject printer page
/
/	sysep is called to eject the printer page.
/
/	jsr	pc,sysep	/ call to eject printer page
/
	.globl	sysep
sysep:	mov	r0,-(sp)	/ save r0
	mov	$1,r0		/ standard output link
	sys	write ; ejcblk+4 ; 1
				/ write eject
	mov	(sp)+,r0	/ restore r0
	rts	pc
/
	.globl	sysex
sysex:	mov	*(sp)+,pc
/
/	sysfc - file control block routine
/
/	sysfc is the first of two calls that handle i/o associations.
/	sysio is the second call. sysfc determines the validity of
/	the file arguments and decides whether a new file control
/	block (fcblk) is needed or if an old fcblk can be used. sysfc
/	does not open the file; opening is done at the time of the
/	first read or write.
/
/	file arg1 is the file descriptor to be used, and should be
/	in the range "lodesc" to "hidesc" inclusive.
/
/	file arg2 is a string containing the file name (if any) and
/	options (if any). the file name must be first with the options
/	following, separated by one or more blanks. the options must
/	always be preceded by blank(s), even if no file name is present.
/
/	the first i/o association call must specify a file name.
/	subsequent i/o associations for the same file descriptor must
/	not specify file names. (it is assumed to be the same.) on the
/	first call for an unassociated file descriptor a new fcblk
/	and buffer are allocated. (also, if it is the first call to
/	sysfc, an ioblk is allocated.)
/
/	subsequent i/o associations for the same file descriptor may
/	require a new fcblk depending on specified options. it is
/	legal (and at times desirable) to have more than one fcblk
/	associated with a file descriptor, since this allows different
/	i/o mode access.
/
/	(wa)			/ fcblk pointer
/	(wb)			/ 0/3 for input/output association
/	(xl)			/ file arg2 scblk pointer
/	(xr)			/ file arg1 scblk pointer
/	jsr	pc,sysfc	/ call for fcblk
/		loc		/ invalid file arg
/	(wa not zero)		/ size of fcblk wanted
/	(wa zero/xl nonzero)	/ private fcblk pointer (in xl)
/	(wc)			/ 0/nonzero request xrblk/xnblk
/
	.globl	sysfc
sysfc:	jsr	r5,..sav	/ save registers
/
/	get file descriptor
/
	cmp	(xr)+,(xr)+	/ xr -> numeric string
	mov	xr,-(sp)	/ push its address
	jsr	pc,getnum	/  and convert to binary
	tst	(sp)+		/ ignore "stopper" address
	cmp	ia,$lodesc	/ insure that file decriptor
	blo	fcer1
	cmp	ia,$hidesc	/  is in valid range
	bhi	fcer1
	mov	ia,svdesc	/ save in svdesc
	sub	$lodesc,ia	/ compute index of ioelt
	mul	$ioelt,ia	/  within ioblk
	mov	ia,svofst	/  and save in svofst
/
/	handle null filename case
/
	jsr	pc,scnfnm	/ scan file name
	mov	(sp)+,lenfnm	/ if length of file name ne 0
	bne	fc10		/  then skip
	tst	wa		/ must have fcb!
	beq	fcer1
	jsr	pc,scnarg	/ scan any file arguments
	bcs	fcer1		/  terminate if error
	tst	newfcb		/ if no new fcb required
	bne	1f
	clr	wa		/  then use fcb passed in wa
	br	fcxit
1:	mov	$fcsiz,wa	/  else allocate new fcb
	br	fcxit
/
/	handle non-null filename
/
fc10:	jsr	pc,scnarg	/ scan file args to get buf size, etc
	bcs	fcer1		/  terminate if error
	tst	rzfil		/ allocate ioblk if necessary
	bne	0f
	mov	$fcsiz+bfsiz+iosiz,wa
				/ alloc fcblk, bfblk, ioblk
	add	svbfsz,wa	/  and buffer too
	br	1f		/ merge to scan args
0:	add	rzfil,ia	/ ia -> ioelt for file descriptor
	tst	ioflg(ia)	/ if file descriptor already in use
	bne	fcer1		/  then take error return
	mov	$fcsiz+bfsiz,wa /  else allocate new fcb and buffer
	add	svbfsz,wa
1:	inc	wa		/ make sure that length request
	bic	$1,wa		/  is even (bufsiz might be odd)
	mov	pc,newfcb	/ always have new fcb for 1st association
/
/	return
/
fcxit:	mov	wa,wa.off(sp)	/ return length of fcb
	mov	pc,wc.off(sp)	/ allocate as xnblk please
	clr	xl.off(sp)	/ not using private fcb's currently
	jsr	r5,..rst	/ restore regs
	add	$2,(sp)
	rts	pc
/
/	sysfc error returns
/
fcer1:	jbr	erxit1
/
/	syshs - get host information
/
/	syshs returns an scblk containing information about the
/	host computer system. three arguments are passed and currently
/	are ignored. a sclbk is returned containing the computer name,
/	operating system name, and installation name, all separated
/	by colons.
/
/	(wa)			/ arg1 pointer
/	(xl)			/ arg2 pointer
/	(xr)			/ arg3 pointer
/	jsr	pc,syshs	/ call to get host string
/	(xl)			/ host scblk pointer
/
	.globl	syshs
syshs:	mov	$hstblk,xl	/ return host information
	rts	pc
/
/	sysil - get input record length
/
/	sysil returns the length of the next record to be read. for
/	line mode reads return a length of "blksiz" since the length is
/	not known; for raw mode reads record size value in the fcblk
/	is returned.
/
/	(wa)			/ fcblk pointer
/	jsr	pc,sysil	/ call to get length
/	(wa)			/ length of next record
/
	.globl	sysil
sysil:	mov	fcrsz(wa),wa	/ pick up record size
	bgt	ilxit
	neg	wa		/ if negative make positive
ilxit:	rts	pc		/ return
/
/	sysin - read record
/
/	sysin reads the next record from a previously input associated
/	file. a sysin call is made after a call to sysil to obtain the
/	record length. if necessary, the file is opened.
/
/	(wa)			/ fcblk pointer
/	(xr)			/ buffer scblk pointer
/	jsr	pc,sysin	/ call to read record
/		loc		/ end of file
/		loc		/ i/o error
/		loc		/ record format error
/	(sclen(xr))		/ set to record length
/				/ (not set on eof!)
/
	.globl	sysin
sysin:	jsr	r5,..sav	/ save registers
	jsr	pc,osopen	/ open file for input
	bcs	iner2		/  if open error take error return
	jsr	pc,osread	/ read record
	bit	$io.err,ioflg(wc)
	bne	iner3		/ on i/o error take error exit
	bit	$io.eof,ioflg(wc)
	bne	iner1		/ on eof take error exit
	jsr	r5,..rst	/ restore registers
	add	$6,(sp)
	rts	pc		/ return
/
iner1:	jbr	erxit1		/ eof
iner2:	jbr	erxit2		/ i/o error
iner3:	jbr	erxit3		/ record format error
/
/
/	sysio - perform i/o association
/
/	sysio is called after a successful sysfc call to perform
/	the actual i/o association.
/
/	on the first i/o association for a file descriptor the
/	ioelt fields (file name pointer, buffer pointer, and
/	flags) are filled in. the fcblk and bfblk are initialized.
/	(on the first sysio call the ioblk is initialized and its
/	address is stored in rzfil.)
/
/	on subsequent i/o associations the associations validity
/	is checked and if necessary the fcblk is initialized.
/
/	(wa)			/ fcblk pointer
/	(wb)			/ 0/3 for input/output association
/	(xl)			/ file arg2 scblk pointer
/	(xr)			/ file arg1 scblk pointer
/	jsr	pc,sysio	/ call to do i/o association
/		loc		/ file doesn't exist
/		loc		/ i/o not allowed
/	(xl)			/ fcblk pointer
	.globl	sysio
sysio:	jsr	r5,..sav	/ save registers
/
/	allocate ioblk first time through
/
	mov	rzfil,wc	/ if ioblk exists then skip
	bne	0f
	mov	wa,wc		/ allocate ioblk at top of alloc'd mem
	add	$fcsiz+bfsiz+1,wc
				/ skip over fcblk and bfblk
	add	svbfsz,wc	/ skip over buffer
	bic	$1,wc		/ force even in case svbfsz is odd
	mov	$bzxrt,(wc)	/ set ioblk type - external relocatable
	mov	$iosiz,iolen(wc)
				/ set iolbk length - iosiz
	mov	wc,rzfil	/ save address of ioblk
/
/	allocate buffer if necessary
/
0:	add	svofst,wc	/ wc -> ioelt for file descriptor
	tst	iobuf(wc)	/ skip if buffer already allocated
	bne	1f
	mov	xl,iofnm(wc)	/ first fill in file name in ioelt
	mov	wa,xr
	add	$fcsiz,xr	/ xr -> buffer
	mov	xr,iobuf(wc)	/  and save address in ioelt
	mov	$bzxnt,(xr)+	/ set bfblk type - external non-relocatable
	mov	$bfsiz+1,(xr)	/ set buffer header
	add	svbfsz,(xr)	/  and buffer length
	bic	$1,(xr)+	/ force even in case svbfsz odd
	mov	svbfsz,(xr)+	/ set size of buffer
	clr	(xr)+		/  number positions left in buffer
	clr	(xr)		/  number of positions in use
	mov	svflgs,ioflg(wc) / set ioelt flags
	br	2f		/ go fill in fcb
/
/	insure that second (or later) call is consistent with first
/
1:	mov	ioflg(wc),cp	/ get flags for file descriptor
	bic	$177777-io.inp-io.out,cp / clear non-essential bits
	bit	cp,svflgs	/  and test if same io.inp/io.out bit
	beq	ioer2		/  is on - if not then error
/
/	fill in fields of fcb if necessary
/
2:	tst	newfcb		/ new fcb present?
	beq	ioxit
	mov	$fcsiz,fclen(wa) / set fclen
	mov	svdesc,fcdsc(wa)  / set file descriptor
	mov	svrcsz,fcrsz(wa)  / set record lenghth
/
/	return
/
ioxit:	mov	wa,xl.off(sp)	/ return fcb pointer in xl
	jsr	r5,..rst	/ restore regs
	add	$4,(sp)
	rts	pc
/
/	sysio error exits
/
ioer1:	jbr	erxit1
ioer2:	jbr	erxit2
/
/
/	sysld - load external function
/
/	sysld is not currently supported.
/
/	(xl)			/ library name scblk pointer
/	(xr)			/ function name scblk pointer
/	jsr	pc,sysld	/ call to load
/		loc		/ function does not exist
/		loc		/ i/o error
/
	.globl	sysld
sysld:	mov	*(sp)+,pc
/
/	sysmb - move words backwards
/
/	sysmb is not required by this implementation.
/
	.globl	sysmb
sysmb:	rts	pc
/
/	sysmc - move characters
/
/	sysmc is not required by this implementation.
/
	.globl	sysmc
sysmc:	rts	pc
/
/	sysmm - get more memory
/
/	sysmm is called in an attempt to obtain more memory for
/	the dynamic area. there is no more memory to be had!
/
/	jsr	pc,sysmm	/ call to obtain more memory
/	(xr)			/ number of words obtained
/
	.globl	sysmm
sysmm:	clr	xr		/ no more memory to be had
	rts	pc
/
/	sysmw - move words
/
/	sysmw is not required for this implementation.
/
	.globl	sysmw
sysmw:	rts	pc
/
/	sysmx - get maximum size of spitbol object
/
/	sysmx is called to obtain the size in bytes of the largest
/	spitbol object that may be created. this value may be set
/	by the -m option on the command line. the default value
/	is "defmax".
/	note: created objects must be less than this value.
/
/	jsr	pc,sysmx	/ call to get max size
/	(wc)			/ max size in bytes of object
/
	.globl	sysmx
sysmx:	mov	maxsiz,wc	/ return maximum size
	inc	wc		/  plus one (for strictly less than)
	rts	pc		/ return
/
/	sysou - output record
/
/	sysou is called to write a record to a previously ouput
/	associated file. for line mode writes a new line is appended
/	to the written record. for raw mode writes the record is
/	written without appending anything.
/
/	(wa)			/ fcblk pointer
/	(xr)			/ buffer scblk pointer
/	jsr	pc,sysou	/ call to write record
/		loc		/ file is full
/		loc		/ i/o error
/
	.globl	sysou
sysou:	jsr	r5,..sav	/ save registers
	jsr	pc,osopen	/ open file for output
	bcs	ouer2		/ handle open error
	jsr	pc,oswrit	/ write record
	bit	$io.err,ioflg(wc)
	bne	ouer1		/ handle i/o error
	jsr	r5,..rst	/ restore registers
	add	$4,(sp)
	rts	pc		/ return
/
ouer1:	jbr	erxit1		/ file is full
ouer2:	jbr	erxit2		/ i/o error
/
/	syspi - print on interactive channel
/
/	syspi is called to write a record to "an interactive channel"
/	or, simply, the terminal. syspi is typically used for issueing
/	error messages. file descriptor 2 is used.
/
/	(wa)			/ line length
/	(xr)			/ buffer scblk pointer
/	jsr	pc,syspi	/ call to write to terminal
/		loc		/ failure return
/
	.globl	syspi
syspi:	jsr	r5,..sav	/ save registers
	mov	$2,wb		/ use file descriptor 2
	br	piprt		/ merge with syspr
/
/	syspp - obtain print parameters
/
/	syspp is called to obtain print parameters and any options
/	set by the command line.
/
/	jsr	pc,syspp	/ call to obtain parameters
/	(wa)			/ length in characters of print line
/	(wb)			/ depth in lines of page
/	(wc)			/ options - see compiler flags
/
	.globl	syspp
syspp:	mov	$132.,wa	/ line length
	mov	$60.,wb		/ page depth
	mov	sptflg,wc	/ options for compiler
	rts	pc
/
/	syspr - print line on standard output file
/
/	syspr is called to print a line on the standard output file.
/	file descriptor 0 is used. syspr falls through to "piprt"
/	which is common to both syspi and syspr.
/
/	(wa)			/ length in characters of line
/	(xr)			/ buffer scblk pointer
/	jsr	pc,syspr	/ call to print line
/		loc		/ print limit exceeded
/
	.globl	syspr
syspr:	jsr	r5,..sav
	mov	$1,wb		/ use file descriptor 1
/
/	piprt is used by syspi and syspr to output line to the
/	file descriptor contained in wb.
/
piprt:	mov	wa,1f		/ compute length of record
	inc	1f		/  with new line
	cmp	(xr)+,(xr)+	/ xr -> 1st byte of record
	mov	xr,0f		/  and set for write
	add	wa,xr		/ xr -> 1st byte beyond record
	movb	(xr),-(sp)	/ save byte on stack ...
	movb	$lf,(xr)	/  ... and temporarily replace with lf
	mov	wb,r0		/ get file descriptor in r0
	sys	indir; 9f
.data
9:	sys	write ; 0: 0 ; 1: 0
.text
			/ write line
	movb	(sp)+,(xr)	/ restore saved character
	cmp	r0,1b		/ if all characters were not written
	bne	prer1		/  then it is an error
	jsr	r5,..rst	/ restore registers
	add	$2,(sp)		/ pop over error exit
	rts	pc		/ normal return
/
prer1:	jbr	erxit1		/ i/o error
/
/	sysrd - read record from standard input
/
/	sysrd is called to read a record from the standard input
/	file. file descriptor 1 is used. a sysrd call follows a
/	sysil call which sets the maximum length of the record to
/	be read.
/
/	(xr)			/ buffer scblk pointer
/	jsr	pc,sysrd	/ call to read from standard input
/		loc		/ end of file
/	(sclen(xr))		/ length of record if not end of file
/
	.globl	sysrd
sysrd:	jsr	r5,..sav	/ save registers
	mov	sclen(xr),wa	/ maximum length record
	mov	wa,wb		/ save for length computation
	mov	xr,xl		/ xl ->
	cmp	(xl)+,(xl)+	/	1st byte of buffer
rd1:	clr	r0		/ read from standard input
	mov	xl,0f		/ set buffer for read
	sys	indir; 9f
.data
9:	sys	read ; 0: 0 ; 1 / read 1 byte at a time
.text
	tst	r0		/ ?eof?
	beq	rder1
	cmpb	$lf,(xl)+	/ new line?
	beq	rd2
	sob	wa,rd1		/ read until new line or max length
	br	rd3		/ compute length without new line
rd2:	clrb	-(xl)		/ remove lf from buffer
	sub	wa,wb		/ wb = length of record
rd3:	mov	wb,sclen(xr)	/ store in scblk
	jsr	r5,..rst	/ restore regs
	add	$2,(sp)		/ pop over error exit
	rts	pc
/
rder1:	jbr	erxit1		/ take error exit
/
/	sysrl - get read length
/
/	sysrl is called to obtain the length of the next record to
/	be read from the standard input file. sysrl returns "blksiz"
/	since the length of the next record is unknown.
/
/	(wa)			/ value from latest -INddd control card
/	jsr	pc,sysrl	/ call to obtain read length
/	(wa)			/ length of next record
/
	.globl	sysrl
sysrl:	mov	$blksiz,wa
	rts	pc
/
/	sysrw - rewind file
/
/	sysrw is called to rewind a currently i/o associated file.
/	for an output associated file, its buffer is flushed before
/	the rewind is done.
/
/	(wa)			/ fcblk pointer
/	(xr)			/ file arg1 scblk pointer
/	jsr	pc,sysrw	/ call to rewind file
/		loc		/ file doesn't exsit
/		loc		/ rewind not allowed
/		loc		/ i/o error
/
	.globl	sysrw
sysrw:	jsr	r5,..sav	/ save registers
	jsr	pc,osopen	/ let osopen fill in fields
	bcs	rwer1		/ handle open error
	bit	$io.out,ioflg(wc)
	beq	0f		/ if open for output
	jsr	pc,flush	/  then flush buffer
	bcs	rwer3		/  handle any i/o error
0:	mov	svdesc,cp	/ load file descriptor in cp
	sys	seek ; 0 ; 0	/ rewind file
	bcs	rwer3		/ handle i/o error
	bic	$io.eof,ioflg(wc)
				/ clear possible eof
	clr	bfoff(xl)	/ reset buffer offset
	clr	bfrem(xl)	/  and remaining bytes in buffer
	jsr	r5,..rst	/ restore registers
	add	$6,(sp)
	rts	pc		/ return
/
rwer1:	jbr	erxit1		/ file doesn't exist
rwer2:	jbr	erxit2		/ rewind not allowed
rwer3:	jbr	erxit3		/ i/o error
/
/	systm - get execution time
/
/	systm is called to obtain the execution time used so far.
/	the value returned is in deciseconds, tenths of a second.
/
/	jsr	pc,systm	/ call to obtain time
/	(ia)			/ time used in deciseconds
/
	.globl	systm
systm:	mov	wb,-(sp)	/ save register
	sys	times ; temp	/ use temp as times buffer
	mov	temp+putime,wb	/ load time into wb/ia
	mov	temp+putime+2,ia
	div	$5,wb		/ convert to tenths of seconds mod ijh 15/5/78
	mov	wb,ia		/ return in ia
	mov	(sp)+,wb	/ restore register
	rts	pc
	rts	pc
/
/	systt - trace toggle
/
/	systt is not used by this implementation.
/
/	jsr	pc,systt	/ call to toggle trace switch
/
	.globl	systt
systt:	rts	pc
/
/	sysul - unload external function
/
/	sysul is not currently supported.
/
/
/	(xr)			/ efblk pointer
/	jsr	pc,sysul	/ call to unload external function
	.globl	sysul
sysul:	rts	pc
/
/	sysxi - exit function
/
/	sysxi is not currently supported.
/
/
/	(xl)			/ 0/string length if arg integer/string
/	(xr)			/ arg iclbk/scblk pointer
/	jsr	pc,sysxi	/ call to exit
/		loc		/ requested action not possible
/		loc		/ irrecoverable error
/
	.globl	sysxi
sysxi:	mov	*(sp)+,pc
/
/	general error exits
/
erxit1: jsr	r5,..rst	/ take error exit 1
	mov	*(sp)+,pc
/
erxit2: jsr	r5,..rst	/ take error exit 2
	add	$2,(sp)
	mov	*(sp)+,pc
/
erxit3: jsr	r5,..rst	/ take error exit 3
	add	$4,(sp)
	mov	*(sp)+,pc
/
/	register save and restore routines
/
..sav:	mov	r4,-(sp)
	mov	r3,-(sp)
	mov	r2,-(sp)
	mov	r1,-(sp)
	mov	r0,-(sp)
	mov	12(sp),-(sp)
	rts	r5
/
..rst:	tst	(sp)+
	mov	(sp)+,r0
	mov	(sp)+,r1
	mov	(sp)+,r2
	mov	(sp)+,r3
	mov	(sp)+,r4
	rts	r5
/
/	flush and flush1 write out any records still held in the
/	buffer associated with file descriptor in svdesc. entry
/	point flush computes the buffer address from the ioelt;
/	entry point flush1 requires buffer address to be in wb.
/
/	/(wc)P
/	(wc)			/ ioelt pointer
/	jsr	pc,flush	/ flush buffer
/	(xl)			/ bfblk pointer
/	(cp,wb)			/ destroyed
/
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	jsr	pc,flush1	/ flush buffer
/	(cp,wb)			/ destroyed
/
/
flush:	mov	iobuf(wc),xl	/ xl -> bfblk
	mov	xl,wb		/ wb ->
	add	$bfbuf,wb	/	buffer
/
flush1: mov	bfoff(xl),1f	/ only do write if there's
	beq	flflsh		/  anything left in buffer
	mov	wb,0f		/ set buffer address for write
	mov	svdesc,cp	/ load file descriptor in cp
	sys	indir; 9f
.data
9:	sys	write ; 0: 0 ; 1: 0
.text
	bcs	flxit		/ return if error
flflsh: clr	bfoff(xl)	/ reset buffer offset
	mov	bfbsz(xl),bfrem(xl)
				/ bytes remaining = buffer size
flxit:	rts	pc		/ return (with carry set if error)
/
/	getc obtains the next character from a file and stores
/	it in a buffer.
/
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	(xr)			/ buffer address
/	jsr	pc,getc		/ call to get character
/	bcs	...		/ eof or error
/	(cp,wb)			/ destroyed
/
getc:	mov	xl,wb		/ xl ->
	add	$bfbuf,wb	/	buffer in bfblk
	tst	bfrem(xl)	/ any characters left?
	bne	get1
/
	mov	wb,0f		/ store buffer address
	mov	bfbsz(xl),1f	/  and size of buffer
	mov	svdesc,cp	/ get file descriptor
	sys	indir; 9f
.data
9:	sys	read ; 0: 0 ; 1: 0
.text
	bcs	geter		/ handle i/o error
	clr	bfoff(xl)	/ reset buffer offset
	mov	cp,bfrem(xl)	/ set number bytes read
	bne	get1
	sec			/ signal eof
	rts	pc		/ return
/
get1:	add	bfoff(xl),wb	/ point to byte in buffer
	inc	bfoff(xl)	/ inc offset
	dec	bfrem(xl)	/ dec bytes remaining in buffer
	movb	(wb),(xr)	/ store character
	rts	pc		/ return
/
geter:	bis	$io.err,ioflg(wc)
	rts	pc		/ return with carry and io.err bits set
/
/	getnum converts a decimal number to binary. the result is
/	returned in r3 (ia). conversion stops on the first non-
/	numeric character encountered - the address of this character
/	is returned on the stack.
/
/	calling sequence:
/
/	mov	$string,-(sp)	/ address of string
/	jsr	pc,getnum	/ call to getnum
/	mov	(sp)+,...	/ address following number
/	(ia)			/ value returned in ia
/
getnum: mov	r5,-(sp)	/ save regs
	mov	r4,-(sp)
	mov	p1+4(sp),r4	/ r4 -> ascii string
	clr	r3		/ r3 is accumulator
0:	cmpb	$60,(r4)	/ a valid decimal digit
	bhi	1f
	cmpb	$71,(r4)	/  is 60 <= d <= 71
	blo	1f
	movb	(r4)+,r5	/ get ascii digit
	sub	$60,r5		/  and strip off unnecessary bits
	mul	$10.,r3		/ multiply accum by 10
	add	r5,r3		/  and add in digit
	br	0b		/ loop back until non-digit
1:	mov	r4,p1+4(sp)	/ return address of next byte
	mov	(sp)+,r4	/ restore regs
	mov	(sp)+,r5
	rts	pc		/ return
/
/	osopen opens or creates files for input/output calls. there
/	are three cases: input associations are opened for input,
/	output associations are either created or opened for output
/	and "seeked" to end for in case of append flag. in addition
/	to pointers to the ioelt and bfblk are returned.
/
/	(wa)			/ fcb pointer
/	jsr	pc,osopen	/ call to open
/	bcs	...		/ carry set if open error
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	svdesc			/ file descriptor
/	svrcsz			/ record size
/	(cp,wb)			/ destroyed
/
osopen: cmp	$bzxnt,(wa)	/ must have fcb!
	bne	opner1
	mov	fcdsc(wa),wc	/ wc = file descriptor
	mov	wc,svdesc	/  and save in svdesc
	mov	fcrsz(wa),svrcsz
				/ save record length-type
	sub	$lodesc,wc	/ compute address
	mul	$ioelt,wc	/  of ioelt
	add	rzfil,wc	/  within ioblk
/
/	open file if necessary.
/
	bit	$io.opn,ioflg(wc)
	bne	opnxit		/ if file open simple exit
/
	mov	$-1,-(sp)	/ set stack marker
	mov	iofnm(wc),xl	/ xl -> file name
	jsr	pc,scnfnm	/ obtain length of file name
	cmp	(xl)+,(xl)+	/ xl -> 1st character of file name
	mov	$temp,wb	/ holding area for file name
	mov	(sp)+,cp	/ length of file name
0:	movb	(xl)+,(wb)+	/ move file name into work area
	sob	cp,0b
	clrb	(wb)		/ set stopper for unix
/
/	open for input
/
	bit	$io.inp,ioflg(wc)
	beq	1f
	sys	open ; temp ; 0 / open for input
	bcs	opner		/ exit if error
	br	opnmr0		/ merge to get file descriptor
/
/	open for output
/
1:	bit	$io.app,ioflg(wc)
	bne	3f
2:	sys	creat ; temp ; 644
	bcs	opner		/ exit if error
	br	opnmr0
/
/	append output
/
3:	sys	open ; temp ; 1 / open for output
	bcs	2b		/ try create if non-existant
	mov	cp,wb		/ save file descriptor
	sys	seek ; 0 ; 2	/ seek to eof
	bcs	opner		/ exit if error
/
/	insure that file descriptor of opened file matches that
/	requested by program.
/
opnmr1: mov	wb,cp		/ get current file descriptor in cp
opnmr0: cmp	cp,fcdsc(wa)	/ is it right one
	beq	opnset		/  yes - close any duplicated descriptors
	mov	cp,-(sp)	/ save current descriptor
	sys	dup		/ duplicate descriptor
	bcs	opner		/ exit if error
	br	opnmr0		/ see if new descriptor is right one
opnset: mov	(sp)+,cp	/ pop previous descriptor
	bmi	0f		/ if negative then none left
	sys	close		/  else close descriptor
	br	opnset		/ loop until all closed
0:	bis	$io.opn,ioflg(wc)
				/ indicate file opened
/
/	return to caller
/
opnxit: mov	iobuf(wc),xl	/ return pointer to bfblk
	rts	pc		/ return
/
opner:	mov	(sp)+,cp	/ must close any descriptors
	bmi	opner1		/ negative number terminates list
	sys	close		/ close descriptor
	br	opner
opner1: sec			/ indicate open error
	rts	pc		/ return
/
/	osread reads the next record into the scblk buffer provided.
/	the value in svrcsz determines whether the read should be
/	line or raw mode. line mode records are delimited by new lines
/	with the new lines filtered out. raw mode records are simply
/	the next "svrcsz" bytes.
/
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	(xr)			/ scblk pointer
/	jsr	pc,osread	/ call to read record
/	bit	io.xxx,ioflg(wc)
/				/ check for eof or i/o error
/	(cp,wa,wb)		/ destroyed
/
osread: mov	xr,-(sp)	/ save scblk pointer
	cmp	(xr)+,(xr)+	/ xr -> 1st byte in scblk buffer
	clr	svflgs		/ inidcate no line read
	mov	svrcsz,wa	/ line or raw mode transfer?
	bmi	rdraw		/ if negative then raw mode
/
/	line mode: read up to next new line and throw away new line.
/
rdlin:	jsr	pc,getc		/ read next byte
	bcs	rdmrg		/ merge on error or eof
	mov	pc,svflgs	/ set flags nonzero
	cmpb	$lf,(xr)+	/ stop on new line
	beq	0f
	sob	wa,rdlin	/ loop until new line or max record len
	br	rdmrg		/ merge on end of record
0:	clrb	-(xr)		/ zap unwanted new line
	br	rdmrg		/ merge on end of record
/
/	raw mode: read next "svrcsz" bytes or until eof.
/
rdraw:	neg	wa		/ make record length positive
0:	jsr	pc,getc		/ read next byte
	bcs	rdmrg		/ merge on error or eof
	inc	xr		/ advance scblk buffer pointer
	sob	wa,0b		/ loop until end of record
/
/	check for i/o error and eof. otherwise set record length.
/
rdmrg:	mov	xr,wb		/ wb -> byte past end of record
	mov	(sp)+,xr	/ restore scblk pointer
	bit	$io.err,ioflg(wc)
	bne	rdxit		/ exit if i/o error
	sub	xr,wb		/ compute length
	sub	$4,wb		/  of record
	bne	0f
	tst	svflgs		/ if record length = 0 and svflgs not 0
	bne	0f		/  then read null line
	bis	$io.eof,ioflg(wc)
				/  else read eof
	br	rdxit		/ if 0 then set eof
0:	mov	wb,sclen(xr)	/ set record length if all went ok
rdxit:	rts	pc		/ return
/
/	oswrit writes a record from the scblk to an output file.
/	the only difference between line and raw mode is a new
/	line is tacked on the end of records in line mode.
/
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	(xr)			/ scblk pointer
/	jsr	pc,oswrit	/ call to write record
/	bit	$io.err,ioflg(wc)
/				/ check for i/o error
/
/	(cp,wa,wb)		/ destroyed
/
oswrit: mov	xr,-(sp)	/ save scblk pointer
	mov	sclen(xr),wa	/ pick up record length
	beq	1f		/ handle null records
	cmp	(xr)+,(xr)+	/ xr -> 1st byte of record
0:	jsr	pc,putc		/ write character
	bcs	wrer		/ handle i/o error
	sob	wa,0b		/ loop thru record
1:	tst	svrcsz		/ if line mode
	bmi	wrxit
	mov	$nuline,xr	/  then append new line
	jsr	pc,putc		/	at end of record
	bcs	wrer		/ check for i/o error
wrxit:	bit	$io.wrc,ioflg(wc)
	beq	0f		/ if io.wrc flag set
	jsr	pc,flush	/  then don't buffer records
0:	mov	(sp)+,xr	/ restore scblk pointer
	rts	pc		/ return
/
wrer:	bis	$io.err,ioflg(wc)
	br	wrxit
/
/	putc writes one character pointed to by xr into a file
/	buffer. when the buffer is filled, flush1 is called
/	to write it out.
/
/	(wc)			/ ioelt pointer
/	(xl)			/ bfblk pointer
/	(xr)			/ character pointer
/	jsr	pc,putc		/ call to write character
/	(xr)			/ 1 byte past character
/	(cp,wb)			/ destroyed
/
putc:	mov	xl,wb		/ wb ->
	add	$bfbuf,wb	/	buffer
	tst	bfrem(xl)	/ any room left in buffer?
	bne	0f
	jsr	pc,flush1	/ no flush it out
	bcs	putxit		/ exit if i/o error
0:	add	bfoff(xl),wb	/ wb -> buffer position
	inc	bfoff(xl)	/ inc buffer offset
	dec	bfrem(xl)	/ dec number remaining bytes
	movb	(xr)+,(wb)	/ move byte into buffer
putxit: rts	pc		/ return
/
/	scnarg is used by sysfc to scan out any arguments to be used
/	with the i/o association. in addition it sets the input/output
/	flag for the association. information is stored in locations
/	labelled svxxxx; newfcb indicates whether a new fcb is needed.
/
/	calling sequence:
/
/	jsr	pc,scnarg	/ call to scan out args
/	bcs	...		/ carry set if error
/
scnarg: jsr	r5,..sav	/ save registers
	mov	$blksiz,svrcsz	/ set default record size& line mode
	mov	$blksiz,svbfsz	/ set default buffer size to block size
	tst	wb		/ check for input/output
	bne	0f
	mov	$io.inp,svflgs	/  ... input - set io.inp flag
	br	1f
0:	mov	$io.out,svflgs	/  .. output - set io.out flag
1:	clr	newfcb		/ assume no fcb is needed
/
/	point to beginning of args
/
	mov	sclen(xl),xr	/ xr = length of file name and args
	beq	argxit		/ if 0 then no args
	sub	lenfnm,xr	/ if nothing past file name then no args
	beq	argxit
	cmp	(xl)+,(xl)+	/ xl
	add	lenfnm,xl	/    -> 1st blank past file name
/
/	move args into temp buffer
/
	cmp	xr,$tmpsiz-1	/ args (and \0) must fit in temp
	bhi	arger1		/  then definite error
	mov	$temp,cp	/ use temp as buffer
0:	movb	(xl)+,(cp)+	/ move args
	sob	xr,0b
	clrb	(cp)		/ set stopper for getnum
	mov	$temp,xl	/ scan out of temp
/
/	scan over a span(blanks)
/
scnar1: cmpb	$40,(xl)	/ skip over span(blanks)
	bne	1f
	inc	xl		/ bump over blank
	br	scnar1		/ and continue span
/
/	scan arguments after blank(s)
/
1:	movb	(xl)+,curarg	/ pick up first byte
	beq	argxit		/ if 0 then end of args
	movb	(xl)+,curarg+1	/ pick up second byte
	bisb	$40,curarg+1	/ insure lower case letter
	mov	$argtbl,wa	/ wa -> arg table
0:	cmp	curarg,(wa)	/ have we found a match?
	beq	1f
	add	$argsiz,wa	/ point to next entry
	br	0b		/  and try again
1:	jsr	pc,*argrtn(wa)	/ call arg processing routine
	mov	pc,newfcb	/ indicate new fcb is needed
	br	scnar1		/ loop to scan next arg
/
argxit: jsr	r5,..rst	/ restore registers
	rts	pc
/
arger:	tst	(sp)+		/ remove return address
arger1: jsr	r5,..rst	/ restore registers
	sec			/ carry set indicates error
	rts	pc		/ error return
/
/	set arg flag
/
argset: bis	argflg(wa),svflgs
	rts	pc		/ set arg's flag in svflgs
/
/	set buffer size
/
argb:	jsr	pc,argnum	/ get number
	tst	*argflg(wa)	/ if zero
	beq	arger		/  then error
	rts	pc		/  else return
/
/	set c access
/
argc:	mov	$-1,svrcsz	/ set 1 character records - raw mode
	rts	pc
/
/	set line mode access
/
argl:	jsr	pc,argnum	/ get number for line length
	tst	svrcsz		/ if record length is 0
	bne	0f
	mov	$blksiz,svrcsz	/  then default to disk block size
0:	rts	pc
/
/	get number and store at *argflg(wa)
/
argnum: mov	xl,-(sp)	/ address of number(?)
	jsr	pc,getnum	/ convert to numeric if possible
	mov	(sp)+,xl	/ point xl past number
	mov	ia,*argflg(wa)	/ store value
	bmi	0f		/ if negative then error
	rts	pc		/  else return
0:	tst	(sp)+		/ error - pop return address
	br	arger		/ jump to error return
/
/	set raw mode access
/
argr:	jsr	pc,argnum	/ get number for record length
	tst	svrcsz		/ if record length is 0
	bne	0f
	mov	$blksiz,svrcsz	/  then default to disk block size
0:	neg	svrcsz		/ negate record length for raw mode
	rts	pc
/
/	scnfnm determines the length of the filename portion
/	of the scblk string pointed to by xl.
/
/	calling sequence
/
/	(xl)			/ pointer to scblk
/	jsr	pc,scnfnm	/ call to determine length
/	mov	(sp)+,...	/ length returned on stack
/
scnfnm: mov	(sp),-(sp)	/ make room for returned length
	clr	p1(sp)		/ default to 0
	jsr	r5,..sav	/ save regs
	mov	sclen(xl),xr	/ xr = length of file name and args
	beq	fnmxit		/ skip if null string
	cmp	(xl)+,(xl)+	/ point to 1st character
	mov	xl,wc		/ save in wc
0:	cmpb	$40,(xl)+	/ compare to blank
	beq	1f
	sob	xr,0b		/ loop back
	inc	xl		/ bump xl past (missing) blank
1:	sub	wc,xl		/ compute length
	dec	xl		/  file name
	mov	xl,p1+regs(sp)	/  and return on stack
fnmxit: jsr	r5,..rst	/ restore regs
	rts	pc		/ return
/
/
/	unix equates
/
/	process time block
/
putime	= 0			/ process user time
pstime	= 4			/ process system time
cutime	= 10			/ child user time
cstime	= 14			/ child system time
timsiz	= 20			/ size in bytes of time block
/
/	data areas
/
/
/	option processing data
/
/	flags for compiler
/
errors	= 1			/ send errors to terminal
prtich	= 2			/ standard printer is terminal
nolist	= 4			/ suppress compilation listing
nocmps	= 10			/ suppress compilation statistics
noexcs	= 20			/ suppress execution statistics
lnglst	= 40			/ generat page ejects
noexec	= 100			/ suppress program execution
trmnal	= 200			/ terminal i/o asociation
/
/	default compiler flags
/
deflag = errors+nolist+nocmps+noexcs	/ default flags
/
/	option table
/
opttxt	= 0			/ option characters
optflg	= opttxt+2		/ option flag - flags or address
optrtn	= optflg+2		/ addres of option processing routine
optsiz	= optrtn+2		/ size in bytes of option entry
/
.data
opttbl:
		<-e>	; errors	; optclr
		<-l>	; nolist	; optclr
		<-c>	; nocmps	; optclr
		<-x>	; noexcs	; optclr
		<-a>	; nolist+nocmps+noexcs	; optclr
		<-p>	; lnglst	; optset
		<-n>	; noexec	; optset
		<-s>	; stkwds	; optnum
		<-m>	; maxsiz	; optnum
curopt:		<  ?>
	.byte		lf		; opterr
/
sptflg:		deflag		/ flags for compiler
/
/	stack data areas
/
defstk	= 1024.			/ reserve 1024 words for stack (default)
stkwds:		defstk		/ number of words to reserve for stack
spsave:		0		/ sp save area
defmax	= 8192.			/ default value for max object size
maxsiz:		defmax		/ value for max object size
/
/	pmodel contains value used in computing error code from an
/	odd pc address. if the pdp-11 model doesn't increment the
/	odd pc then its value is 1 else its value is -1.
/
pmodel:		-1		/ default value
/
/	input/output association data areas
/
lodesc	= 4			/ lowest valid file descriptor
hidesc	= 14.			/ highest valid file descriptor
nmdesc	= hidesc-lodesc+1	/ number of valid descriptors
/
/	ioblk holds information about active input/output assocaitions.
/	rzfil in the compiler's relocatable work area points to the
/	current ioblk or contains a zero if no ioblk is active.
/
iotyp	= 0			/ spitbol block type (external relocatable)
iolen	= 2			/ spitbol block length (iosize)
iofnm	= 4			/ pointer to file name for file descriptor
iobuf	= 6			/ pointer to buffer for file descriptor
ioflg	= 10			/ flags for file descriptor
/
io.inp	= 1			/ input association flag
io.out	= 2			/ output association flag
io.app	= 4			/ append output flag
io.opn	= 10			/ file open flag
io.eof	= 20			/ eof flag
io.err	= 40			/ i/o error flag
io.sys	= 100			/ system file flag
io.wrc	= 200			/ write each record (no buffering)
/
ioelt	= ioflg+2-iofnm		/ fields iofnm -> ioflg is called ioelt
iosiz	= iofnm+[hidesc-lodesc+1*ioelt] / size of ioblk
/
/
/	the fcblk contains information about an individual i/o association.
/	and is pointed to by a trap block hanging off a variable block. the
/	fcblk contains the file descriptor and the i/o mode. more than one
/	fcblk may be active per file descriptor.
/
fctyp	= 0			/ spitbol block type (external non-relocatable)
fclen	= 2			/ spitbol block length
fcdsc	= 4			/ file descriptor
fcrsz	= 6			/ record size (raw mode) or 0 (line mode)
fcsiz	= 10			/ size of fcblk
/
/	the bfblk contains th buffer and related information for a file
/	descriptor. two fields of related information are kept: the
/	number of bytes left in the buffer and an offset to the next
/	byte to be accessed in the buffer.
/
bftyp	= 0			/ spitbol block type (external non-relocatable)
bflen	= 2			/ spitbol block length
bfbsz	= 4			/ buffer size in bytes
bfrem	= 6			/ number of bytes remaining in buffer
bfoff	= 10			/ offset in buffer to next byte
bfbuf	= 12			/ start of buffer
bfsiz	= 12			/ buffer header size (buffer size excluded)
/
blksiz	= 512.			/ disk block size
/
/	data areas used during processing of i/o associations.
/
svbfsz:		0		/ saved buffer size
/
/	file argument table
/
argtxt	= 0			/ arg characters
argflg	= 2			/ arg flag - flags or address
argrtn	= 4			/ address of arg processing routine
argsiz	= 6			/ size in bytes of arg entry
/
/
/
argtbl:
		<-a>	; io.app	; argset
		<-b>	; svbfsz	; argb
		<-c>	; 1		; argc
		<-l>	; svrcsz	; argl
		<-r>	; svrcsz	; argr
		<-w>	; io.wrc	; argset
curarg:		<  >	; 0		; arger
/
/
/
svdesc:		0		/ saved file descriptor
svflgs:		0		/ saved flags for svdesc
svofst:		0		/ saved offset into ioblk for svdesc
svrcsz:		0		/ saved record size for svdesc
lenfnm:		0		/ length of file name
newfcb:		0		/ newfcb flag: not 0 (new fcb)/0 (no new fcb)
spdesc	= 2			/ special file descriptor (for terminal)
/
/	temporary buffer
/
.bss
temp:	.=.+blksiz
tmpsiz	= blksiz		/ size of temp area
/
/	scblk - spitbol's internal string block
/
sctyp	= 0			/ block type word - bzscl
sclen	= 2			/ string length in characters
scchr	= 4			/ string starts here
/
/	file name and file descriptor for error text file
/
.text
errnam:		</usr/lib/spiterr>
	.byte	0
	.even
.data
errfcb:		0		/ link to error file
/
/	define new line character (line feed) and eject scblk
/
.text
lf	= 12			/ line feed character (new line)
nuline:		lf		/ store new line character
ejcblk	= .-2			/ don't need type word
		1 ; 14		/ sclen = 1 ; string = ff
/
/
/	host block contains site dependent information
/
hstblk:		0
		hstlen
		<pdp-11/70:unix:UNSW-DCS>
hstlen	= .-hstblk-4
	.even
topint: / doesn't appear to be referenced !!
