MODULE communications;

{
This program enables a console on the PDP-11
to be used as a terminal on the DEC-10,  by
transferring characters to and from the DEC-10
via a DL-11 line interface.
}

CONST controlS = 23C; { suspends tty output }
      controlQ = 21C; { resumes tty output }
      controlO = 17C; { kills tty output }



DEVICE MODULE inline[4];
DEFINE get, full, empty, kill;

CONST bsize = 256;    
VAR buffer: ARRAY 0:bsize-1 OF char;
    in, out: integer; 
    chars: integer;
    full, empty, nonempty: signal;

PROCEDURE kill;
BEGIN
  in := 0; out := 0; chars := 0
END kill;

PROCEDURE get(VAR c: char);
BEGIN
  IF chars=0 THEN send(empty); wait(nonempty) END;
  out := (out+1) MOD bsize;
  dec(chars);
  c := buffer[out]
END get;

PROCESS driver[310B];
  VAR rsr[175610B]: bits; { receiver status register }
      rbr[175612B]: char; { receiver buffer register }
BEGIN
  LOOP
    rsr[6] := true;
    doio;
    in := (in+1) MOD bsize;
    buffer[in] := rbr;
    inc(chars);
    IF chars>bsize-10
    THEN send(full)
    ELSE send(nonempty)
    END
  END
END driver;

BEGIN { in line }
  kill; driver
END inline;


DEVICE MODULE outline[4];
DEFINE put;

CONST bsize = 8;      
VAR buffer: ARRAY 0:bsize-1 OF char;
    in, out: integer; 
    chars: integer;  
    nonempty: signal;

PROCEDURE put(c: char);
BEGIN
  in := (in+1) MOD bsize;
  buffer[in] := c;
  inc(chars);
  send(nonempty)
END put;

PROCESS driver[314B];
  VAR tsr[175614B]: bits;
      tbr[175616B]: char;
BEGIN
  LOOP
    IF chars=0 THEN wait(nonempty) END;
    out := (out+1) MOD bsize;
    tbr := buffer[out];
    dec(chars);
    tsr[6] := true;
    doio;
    tsr[6] := false
  END
END driver;

BEGIN { out line }
  in := 0; out := 0;
  chars := 0; driver
END outline;


PROCESS stopper;
BEGIN
  LOOP
    wait(full);
    put(controlS);
    wait(empty);
    put(controlQ)
  END
END stopper;


DEVICE MODULE keyboard[4];
DEFINE ttyin;

CONST bsize = 32;     
VAR buffer: ARRAY 0:bsize-1 OF char;
    in, out: integer; 
    chars: integer;
    nonempty: signal;

PROCEDURE ttyin(VAR c: char);
BEGIN
  IF chars=0 THEN wait(nonempty) END;
  out := (out+1) MOD bsize;
  c := char(integer(buffer[out]) MOD 128);
  dec(chars)
END ttyin;

PROCESS driver[60B];
  VAR ksr[177560B]: bits; { keyboard status register }
      kbf[177562B]: char; { keyboard buffer register }
BEGIN
  LOOP
    ksr[6] := true;
    doio;
    in := (in+1) MOD bsize;
    buffer[in] := kbf;
    inc(chars);
    send(nonempty)
  END
END driver;

BEGIN { keyboard }
  in := 0; out := 0;
  chars := 0; driver
END keyboard;


DEVICE MODULE printer[4];
DEFINE ttyout, suspend, resume;

CONST bsize = 32;       
VAR buffer: ARRAY 0:bsize-1 OF char;
    in, out: integer;   
    chars: integer;
    suspended: boolean;
    nonfull, nonempty, continue: signal;

PROCEDURE suspend;
BEGIN
  suspended := true
END suspend;

PROCEDURE resume;
BEGIN
  suspended := false;
  send(continue)
END resume;

PROCEDURE ttyout(c: char);
BEGIN
  IF chars=bsize THEN wait(nonfull) END;
  in := (in+1) MOD bsize;
  buffer[in] := c;
  inc(chars);
  send(nonempty)
END ttyout;

PROCESS driver[64B];
  VAR psr[177564B]: bits; { printer status register }
      pbr[177566B]: char; { printer buffer register }
BEGIN
  LOOP
    IF suspended THEN wait(continue) END;
    IF chars=0 THEN wait(nonempty) END;
    out := (out+1) MOD bsize;
    pbr := buffer[out];
    dec(chars);
    send(nonfull);
    psr[6] := true;
    doio;
    psr[6] := false
  END
END driver;

BEGIN { printer }
  in := 0; out := 0;
  chars := 0; suspended := false;
  driver
END printer;

PROCESS receiver;
  VAR c: char;
BEGIN
  LOOP get(c); ttyout(c) END
END receiver;



PROCESS transmitter;
  VAR c: char;
BEGIN
  LOOP
    ttyin(c);
    IF   c=controlS
    THEN suspend
    ELSIF c=controlQ
    THEN resume
    ELSIF c=controlO
    THEN kill; put(c)
    ELSE put(c)
    END
  END
END transmitter;



BEGIN { communicating }
  stopper;
  transmitter;
  receiver
END communications.

{
.bp
}
