{Chris Bond
3rd year project
Interpretation of morse code input on PDP-11 computer

To read and display morse code in several forms:
  Raw morse as lengths and histograms.
  Actual morse (symbols and spaces)
  Text (translated morse).
}
MODULE morse;

TYPE spec = (key, disk, none); {Source / destination possibilities of morse 
                                                                       code}

VAR getfromkey, getfromdisk : signal;
    processlengths, processmorse, allprocessed : signal;
    stoplengths, morsetodate, texttodate, alltodate : boolean;

    {Signals and booleans used to start and stop the process of reading 
     morse and converting it from lengths to text}

    input, output : spec;
DEVICE MODULE keyboard [4];

{ This Module provides the interface to the keyboard.
Characters are stored in the circular buffer keybuf and
retrieved by calls to the procedure read.}

DEFINE read;

CONST keysiz = 32;

VAR keybuf : ARRAY 0 : keysiz - 1 OF char;
    in, out, nchars : integer;
    ne, nf : signal;

PROCEDURE read (VAR ch : char);
BEGIN
  IF nchars = 0 THEN wait (ne) END;
  out := (out + 1) MOD keysiz;
  ch := keybuf[out];
  dec (nchars); send (nf)
END read;

PROCESS driver [60B];
  VAR kcsr[177560B] : bits;
      kdbr[177562B] : integer;
BEGIN
  LOOP
    IF nchars = keysiz THEN wait (nf) END;
    kcsr[6] := true; doio; kcsr[6] := false;
    in := (in + 1) MOD keysiz;
    keybuf[in] := char (kdbr MOD 128);
    inc (nchars); send (ne)
  END
END driver;

BEGIN {keyboard}
  in := 0; out := 0; nchars := 0;
  driver
END keyboard;
DEVICE MODULE timing[6];

{ This module services the line clock, sending a signal
(tick) to each process waiting on the clock.  The clock
runs at 50Hz, thus the process 'driver' sends on 'tick'
(the signal) every 1/50 th of a second.}

DEFINE tick, delay;

VAR tick : signal;

PROCEDURE delay (n : integer);

{This procedure delays the calling process by n * 1/50 ths of a second}

  VAR count : integer;
BEGIN
  count := n;
  REPEAT wait (tick); dec (count) UNTIL count <= 0
END delay;

PROCESS clock [100B];
  VAR lcsr[177546B] : bits;
BEGIN
  LOOP
    lcsr[6] := true; doio; lcsr[6] := false;
    WHILE awaited (tick) DO send (tick) END
  END
END clock;

BEGIN {timing}
  clock
END timing;
DEVICE MODULE bell[4];

{This module implements the procedure to ring the bell in the display processor}

DEFINE ring;

PROCEDURE ring;
  VAR dsr[172002B] : integer;
BEGIN
  dsr := 1
END ring;

END bell;
MODULE communications;

{This module implements the communications with the host machine which
enables morse (in the form of lengths) to be filed on disk on the larger machine
and retrieved at a later date.  It is necessary to have the program 'morsecom'
running on the host line to provide the other half of the communication.}


DEFINE acknowledge, openfile, createfile, getm, putm, setupmorsebuf,
       stopprog, fileempty;

USE delay, disk, input, output, ring;

CONST rxm = 1C;  {ctrl A - Sent by GT40 instructing host to receive Morse }
      txm = 2C;  {ctrl B - Sent by GT40 instructing host to transmit Morse}
      ack = 6C;  {ctrl F - Sent by GT40 acknowledging receipt of a block of
                           morse and asking for the next (host is transmitting)
                           Sent by host to ackowledge successful creation or
                           lookup of a file.}
      nak = 25C; {ctrl U - As above but implies negative acknowledgement}
      bof = 31C; {ctrl Y - unique symbol for marking beginning of file when
                           host is transmitting a file of morse.}
      eof = 32C; {ctrl Z - Symbol transmitted by host to indicate end of file.
                           When transmitted to host it instructs the morsecom
                           program to exit.}

      nl = 12C;
      space = 40C;

VAR acknol, getcount, putcount : integer;
    fileempty : boolean;
DEVICE MODULE receiver [5];

{ This Module provides the interface to the DL-11 receiver.
Characters are stored in the circular buffer rxbuf and retrieved by calls
to the procedure get.
An exception to this system are the command characters ack and nak.  These
are not put in the buffer but instead set the value of acknol to 1 and -1
respectively, representing acknowledgement and -ve acknowledgement from host.
If the buffer fills these characters are still detectable but other characters
are ignored.}

DEFINE get;

USE nl, space, acknol, ack, nak;

CONST rxsiz = 64;

VAR rxbuf : ARRAY 0 : rxsiz - 1 OF char;
    in, out, nchars : integer;
    ne : signal;

PROCEDURE get (VAR ch : char);
BEGIN
  IF nchars = 0 THEN wait (ne) END;
  out := (out + 1) MOD rxsiz;
  ch := rxbuf[out];
  dec (nchars)
END get;

PROCESS driver [300B];
  VAR rcsr[175610B] : bits;
      rdbr[175612B] : integer;
      ch : char;
BEGIN
  LOOP
    rcsr[6] := true; doio; rcsr[6] := false;
    ch := char (rdbr MOD 128);
    IF ch = ack THEN acknol := 1
    ELSIF ch = nak THEN acknol := -1
    ELSE
      IF nchars < rxsiz THEN
        in := (in + 1) MOD rxsiz;
        rxbuf[in] := char (rdbr MOD 128);
        inc (nchars)
      END;
      send (ne)
    END
  END
END driver;

BEGIN {receiver}
  in := 0; out := 0; nchars := 0;
  driver
END receiver;
DEVICE MODULE transmitter [5];

{ This Module provides the interface to the DL-11 transmitter.
Characters are placed in the circular buffer txbuf by calls
to the procedure put and output from there by the driver.}

DEFINE put;

CONST txsiz = 64;

VAR txbuf : ARRAY 0 : txsiz - 1 OF char;
    in, out, nchars : integer;
    ne, nf : signal;

PROCEDURE put (ch : char);
BEGIN
  IF nchars = txsiz THEN wait (nf) END;
  in := (in + 1) MOD txsiz;
  txbuf[in] := ch;
  inc (nchars); send (ne)
END put;

PROCESS driver [304B];
  VAR tcsr[175614B] : bits;
      tdbr[175616B] : integer;
BEGIN
  LOOP
    IF nchars = 0 THEN wait (ne) END;
    out := (out + 1) MOD txsiz;
    tdbr := integer (txbuf[out]);
    dec (nchars); send (nf);
    tcsr[6] := true; doio; tcsr[6] := false;
  END
END driver;

BEGIN {transmitter}
  in := 0; out := 0; nchars := 0;
  driver
END transmitter;
DEVICE MODULE bodgestop[5];

{This module implements the procedure 'stopprog' which causes a jump from
morse into the bootstrap program of the GT40 after first sending eof to stop
morsecom running under host.  It operates by changing the interrupt vector
address of the dl-11 transmitter to point to the bootstrap (166000B) and then
deliberately causes an interrupt to effect the jump.}

DEFINE stopprog;

USE put, eof, nl, delay;

VAR txint[304B] : integer;
    tcsr[175614B] : bits;

PROCEDURE stopprog;
BEGIN
  put (eof); delay (2); txint := 166000B; tcsr[6] := true
END stopprog;

END bodgestop;
PROCEDURE acknowledge : integer;
  VAR i : integer;

{Function procedure that returns the value of acknol to the caller.  If acknol
is zero it waits up to 5 seconds for it to change before returning zero which
will indicate that host is not connected (or morsecom is not running).  Acknol
of 1 indicates +ve acknowledgement, -1  -ve.}

BEGIN
  i := 0;
  WHILE (acknol = 0) AND (i < 25) DO
    delay (10); inc(i)
  END;
  acknowledge := acknol;
  acknol := 0
END acknowledge;

PROCEDURE putstring (s : ARRAY integer OF char);
  VAR i : integer;

{Writes a string of characters to dl-11 output buffer using put.  The string
should be terminated by a newline character (nl).}

BEGIN
  i := low (s);
  REPEAT
    put (s[i]); inc (i)
  UNTIL (i > high (s)) OR (s[i-1] = nl)
END putstring;

PROCEDURE openfile (fname : ARRAY integer OF char) : integer;

{Sets up host to transmit morse (lengths) to GT40, asking host to open file
fname.  It waits for an acknowledgement and returns it to the caller as its
function value.}

BEGIN
  put (txm);
  putstring (fname);
  openfile := acknowledge
END openfile;

PROCEDURE createfile (fname : ARRAY integer OF char) : integer;

{Sets up host to receive morse asking it to create the file fname.  It waits
for an acknowledgement and returns it to the caller as its function value.}

BEGIN
  put (rxm);
  putstring (fname);
  createfile := acknowledge
END createfile;

PROCEDURE setupmorsebuf;
  VAR ch : char;

{If diskinput is set, this procedure reads from the input buffer until it
detects the bof character.  It then requests host to transmit the 1st block.
If diskinput is not set then it zeros the count of characters output.}

BEGIN
  IF input = disk THEN
    REPEAT
      get (ch)
    UNTIL (ch = bof);
    put (ack); getcount := 0;
    fileempty := false
  ELSIF output = disk THEN
    putcount := 0
  END
END setupmorsebuf;

PROCEDURE getm (VAR length : integer);
  VAR ch : char;

{Reads lengths from input buffer requesting new blocks when it reads newline
characters.  It recognises error conditions but does not do any correction.
It does not attempt to read if the file is empty but simulates an eof.}

BEGIN
  IF NOT fileempty THEN
    REPEAT
      IF getcount = 61 THEN 
        ring; delay (10); ring; {error}
        put (ack); {get another record anyway}
        getcount := 0
      END;
      get (ch); inc (getcount);
      IF ch = nl THEN
        IF getcount <> 61 THEN ring; delay (10); ring {error} END;
        put (ack); getcount := 0
      END
    UNTIL ((ch > 40C) AND (ch < 135C) OR (ch = eof))
  ELSE ch := eof END;
  length := integer (ch) - 40B;
  IF length < 0 THEN fileempty := true END
END getm;

PROCEDURE putm (length : integer);

{Writes lengths out to dl-11 buffer for transmission to host inserting spaces
and newlines as required by the buffer protocol.}

BEGIN
  put (space); put (char (length + 40B));
  inc (putcount, 2);
  IF putcount >= 60 THEN
    putcount := 0; put (nl)
  END
END putm;

BEGIN {communications}
  acknol := 0;  fileempty := false
END communications;
DEVICE MODULE tapper [4];

{ Tapper is the interface to the morse code key.  It implements
the procedure mark which senses the state of the key. }

DEFINE mark;
VAR camstr [160000B] : integer;  {station reg}
    camcsr [160002B] : bits;     {status reg}
    camxcr [160004B] : bits;     {exec reg}
    camswr [162040B] : bits;     {switch register (bit 0 is morse key)}

PROCEDURE mark : boolean;
BEGIN
  mark := camswr[0]
END mark;

BEGIN {tapper}
  camstr := 5;
  camcsr := [];
  camxcr := [0, 7, 8, 14]
END tapper;
MODULE analysis;

{This module encompasses the modules which contain procedures for
analysing the input from the morse key, beginning with lengths and
finally ending with the production of text on the screen.}

DEFINE windowsiz, newwindow, startreading, setlength,
       setmorse, settext, discriminate,
       displayhists, displaylengths, displaymorse, displaytext,
       displayhelptext,
       marks, spaces, writeinfo, rescalehistfiles,
       lcount, lmcount, mcount, mtcount,
       dotpnt, iegpnt, icgpnt, sympnt, on, off, state, left, right, direction,
       pointers, shiftpointer, pointerblink, evaluatepointers;

USE getfromkey, getfromdisk, stoplengths, getm, putm,
    input, output, key, disk, none,
    morsetodate, texttodate, alltodate, ring;

CONST pntmod = 117620B;  {Point mode}
      chrmod = 103620B;  {Character mode}
      lvdmod = 113627B;  {Long Vector Dot dashed line}
      lvhmod = 113524B;  {Long Vector solid (Heavy) line}
      lvlmod = 113525B;  {Long Vector Long dash}
      lvsmod = 113626B;  {Long Vector Short dash}
      dstop = 173400B;  {Display stop and interrupt}
      ch = 24; {character height}
      cw = 14; {character width}
      seeline = 40000B;  {Intensify vector or point}
      xinc = cw + seeline;  {Steps on x-axis for plotting histograms}
      blink = 10B;  {Blink on bit to make display flash}

TYPE header = ARRAY 1 : 4 OF integer;  {To set graphics mode of GT40}
     len  = RECORD
            head : header;
            labl : ARRAY 1 : 8 OF char;
            body : ARRAY 1 : 12, 1 : 74 OF char;
            tail : integer
            END;  {Format of a Lengths file}
     hist = ARRAY 0 : 60 OF integer;  {Numerical Histogram}
     ms = (marks, spaces);
     sympnt = (dotpnt, iegpnt, icgpnt);  {Identifies pointers on histograms}
     state = (on, off);  {State of blink}
     direction = (left, right);  {Direction in which pointer is to move}
     dispmode = (lengthmode, histmode, morsemode, textmode, helpmode);
     hfile = RECORD
             head : header;
             body : ARRAY 1 : 120 OF integer;
             tail : integer
             END;  {Displayable histogram format}
     mtfile = RECORD
              head : header;
              body : ARRAY 1 : 26, 1 : 74 OF char;
              tail : integer
              END;  {Format of Morse and Text files}
     pntholder = RECORD
                 head : header;
                 pntr : ARRAY 1 : 11 OF integer;
                 labl : ARRAY 1 : 5 OF char;
                 tail : integer
                 END;  {For holding a displayable pointer}

VAR windowsiz : integer;  {Number of lengths over which histograms are held}
    mode : dispmode;  {What to display on GT40}
    lengths : ARRAY marks : spaces OF len; {For holding lengths (raw morse)}
    histograms : ARRAY marks : spaces OF hist;  {Numerical values in histograms}
    morsefile, textfile : mtfile;  {Morsecode and text displayfiles}
    pointers : ARRAY dotpnt : icgpnt OF pntholder;  {Pointers on histograms}
    info : RECORD
           ihead : header;
           ibody : ARRAY 1 : 5, 1 : 74 OF char;
           nhead : header;
           nbody : ARRAY 1 : 3 OF char;
           tail : integer
           END;  {Indicates what the program is doing}
    axes : RECORD
           lhead : header;
           lines : ARRAY 1 : 6 OF integer;
           fhead : header;
           flabl : ARRAY 1 : 4 OF char;
           nhead : header;
           nbody : ARRAY 1 : 3 OF char;
           thead : header;
           tlabl : ARRAY 1 : 66 OF char;
           tail : integer
           END;  {Axes and labels on histograms}
    histfile : ARRAY marks : spaces OF hfile;  {The displayable histograms}
    helpfile : RECORD
               head : header;
               body : ARRAY 1 : 23, 1 : 74 OF char;
               tail : integer
               END;  {Helpfile for the program}

PROCEDURE cleardisplayfile (VAR dfile : ARRAY integer, integer OF char; imax : 
                                                                    integer);
  VAR i, j : integer;

{Sets to spaces (clears) a standard displayfile which is 74 chars wide (72 chars
+ <cr> <lf>) and imax lines long.  This type of file is displayed by the GT40
in character mode.}

BEGIN
  i := 1;
  REPEAT
    j := 1;
    REPEAT
      dfile[i,j] := ' '; inc (j)
    UNTIL j > 72;
    dfile[i,73] := 15C;
    dfile[i,74] := 12C;
    inc (i)
  UNTIL i > imax
END cleardisplayfile;

PROCEDURE updatedpos (VAR rpos, cpos : integer; rmax : integer);

{Moves pointers by 1 position through standard displayfile, skipping over the
<cr><lf> at the end of each line.  Wrap around occurs from the bottom to the top
of the file if rpos exceeds rmax.}

BEGIN
  inc (cpos);
  IF cpos > 72 THEN
    cpos := 1; inc (rpos);
    IF rpos > rmax THEN rpos := 1 END
  END
END updatedpos;
MODULE lenhist;

{Module lenhist contains those routines which manipulate the lengths
and histogram structures.  Lengths is in a form which can immediately
be displayed on the gt40 screen, histograms is not.  Instead, it is
represented by histfile (in displayfiles) which can be displayed.}

DEFINE clearlengths, clearhists, setms, lcount;

USE header, ms, marks, spaces, windowsiz, pntmod, chrmod, dstop, ch, cw,
    histograms, lengths, cleardisplayfile, updatedpos;

VAR i, j, rpos, cpos, lcount : integer;

{Lcount is the number of lengths written into the two lengths files (marks)
to one and spaces to the other).  Rpos and cpos indicate the position of the
last elements written into these standard display files.}

PROCEDURE clearlengths;

{Initialises lengths files, associated counters and pointers}

BEGIN
  cleardisplayfile (lengths[marks].body, 12);
  cleardisplayfile (lengths[spaces].body, 12);
  rpos := 1; cpos := 0; lcount := 0
END clearlengths;

PROCEDURE clearhists;

{Reset the numerical histograms.  All positions are zeroed except zero position
which is set to windowsize to normalise the histograms.}

BEGIN
  histograms[marks,0] := windowsiz; histograms[spaces,0] := windowsiz;
  i := 1;
  REPEAT
    histograms[marks,i] := 0; histograms[spaces,i] := 0;
    inc (i)
  UNTIL i > 60
END clearhists;

PROCEDURE setms (ident : ms; newlen : integer; VAR oldlen : integer);

{Setms writes a mark / space (newlen) into one of the lengths files, returning
the one overwritten in the variable oldlen.  It then adjusts the histograms
incrementing at newlen and decrementing at oldlen to preserve normalisation.}

BEGIN
  IF ident = marks THEN
    inc (lcount);
    IF lcount > windowsiz THEN 
      lcount := 1; rpos := 1; cpos := 1
    ELSE
      updatedpos (rpos, cpos, 13)
    END
  END;
  oldlen := integer (lengths[ident].body[rpos,cpos]) - 40B;
  lengths[ident].body[rpos,cpos] := char (newlen + 40B);
  dec (histograms[ident,oldlen]); inc (histograms[ident,newlen]);
END setms;

BEGIN {lenhist}
  WITH lengths[marks] DO
    head := (pntmod, 0, 25 * ch, chrmod);
    labl := ('M', 'A', 'R', 'K', 'S', ' ', 15C, 12C);
    tail := dstop
  END;
  WITH lengths[spaces] DO
    head := (pntmod, 0, 11 * ch, chrmod);
    labl := ('S', 'P', 'A', 'C', 'E', 'S', 15C, 12C);
    tail := dstop
  END;
  clearlengths;
  clearhists
END lenhist;
MODULE displayfiles;

{Module displayfiles manipulates those display files which are purely
used for display purposes.  This means info, histfiles, axes and the pointers.
Device Module gt40 then displays these on the screen.  The particular display
file is selected by means of calls to procedures displayxxx where xxx represents
the type of display required.}

DEFINE writeinfo, adjusthistfile, rescalehistfiles, shiftpointer,
       pointerblink, evaluatepointers,
       displaylengths, displayhists, displaymorse, displaytext,
       displayhelptext;

USE ms, marks, spaces, windowsiz, cleardisplayfile, ring,
    pntmod, chrmod, lvhmod, lvsmod, dstop, ch, cw, seeline, xinc, blink,
    lvdmod, lvlmod, left, right, direction, on, off, state,  dotpnt,
    iegpnt, icgpnt, sympnt,
    info, axes, pointers, lengths, histograms, histfile,
    mode, lengthmode, histmode, morsemode, textmode, helpmode;

CONST  fnegbit = 20000B;  {Bit to indicate -ve vector (ie move down or left)}

VAR factr, fscale, i, j : integer;

{Fscale is the vertical (frequency) scaling factor for the histograms.  It is
calculated as fscale = 26 * ch * factr / windowsiz where factr can be changed
by calling rescalehistfiles so as to change the scaling on the frequency axis.}

PROCEDURE writeinfo (s : ARRAY integer OF char; n : integer);
  VAR i, j : integer;

{Writes a line of information into the 'info' file.  The line is identified by
its position 1 to 5 in the displayfile.}

BEGIN
  WITH info DO
    j := 1;
    REPEAT ibody[n,j] := ' '; inc (j) UNTIL j > 72;
    j := 1; i := low (s);
    WHILE (i <= high (s)) AND (j <= 72) DO
      ibody[n,j] := s[i]; inc (i); inc (j)
    END
  END
END writeinfo;

PROCEDURE writenum (n : integer; VAR pos : ARRAY integer OF char);
  VAR quot : integer;

{Writes an integer n (max 3 digits) into a specified position (pos) in a
displayfile}

BEGIN
  i := 1;
  REPEAT
    pos[i] := ' ';
    inc (i)
  UNTIL i > 3;
  i := 3;
  REPEAT
    quot := n / 10;
    pos[i] := char (n - quot * 10 + 60B);
    dec (i);
    n := quot
  UNTIL (n = 0) OR (i = 0);
END writenum;

PROCEDURE inithistfile (ident : ms);

{Initialise the histogram files setting all the x steps to xinc}

BEGIN
  i := 1;
  REPEAT
    histfile[ident].body[i] := xinc;
    inc (i, 2)
  UNTIL i > 120
END inithistfile;

PROCEDURE adjusthistfile (ident : ms; l : integer);
  VAR newf : integer;

{Adjusts one point in a histogram file.  As the histograms are drawn with
relative vectors, this usually involves changing two vectors.  Fscale is the
scaling factor used in drawing the histograms from their absolute values held
in the two histogram arrays (as opposed to the histfiles which are for display
only).}

BEGIN
  IF l = 1 THEN
    histfile[ident].body[2] := histograms[ident,1] * fscale
  ELSIF l <> 0 THEN
    newf := (histograms[ident,l] - histograms[ident,l-1]) * fscale;
    IF newf < 0 THEN newf := -newf + fnegbit END;
    histfile[ident].body[l*2] := newf
  END;
  IF (l <> 60) AND (l <> 0) THEN
    newf := (histograms[ident,l+1] - histograms[ident,l]) * fscale;
    IF newf < 0 THEN newf := -newf + fnegbit END;
    histfile[ident].body[(l+1)*2] := newf
  END
END adjusthistfile;

PROCEDURE resethistfile (ident : ms);
  VAR lasth, thish, newf : integer;

{Completely resets the y scale (frequency) values of an histogram file by
calculating the relative positions from the correct histogram array.}

BEGIN
  lasth := histograms[ident,1];
  histfile[ident].body[2] := lasth * fscale;
  i := 2;
  REPEAT
    thish := histograms[ident,i];
    newf := (thish - lasth) * fscale;
    IF newf < 0 THEN newf := -newf + fnegbit END;
    histfile[ident].body[2*i] := newf;
    lasth := thish; inc (i)
  UNTIL i > 60
END resethistfile;

PROCEDURE rescalehistfiles (scaleinc : integer);

{Changes frequency scaling factor on the displayable histograms}

BEGIN
  factr := factr + scaleinc;
  IF factr < 1 THEN ring; factr := 1 END;
  IF factr > 15 THEN ring; factr := 15 END;
  fscale := 26 * ch * factr / windowsiz;
  resethistfile (marks);
  resethistfile (spaces);
  writenum (26 * ch / fscale, axes.nbody);
  writenum (windowsiz, info.nbody)
END rescalehistfiles;

PROCEDURE initpointers;

{Sets up the pointers on the histograms to indicate the dot - dash separation,
the inter element/character separation and the inter character/word
separation.  These originally have default values but can be adjusted by the
user when the program is running.}

BEGIN
  WITH pointers[dotpnt] DO
    head := (pntmod, 3 * cw, 25 * ch, lvdmod);
    pntr := (5 * cw, 0, seeline, 6 * ch + fnegbit,
             0, 17 * ch + fnegbit, seeline, 2 * ch + fnegbit,
             3 * cw + fnegbit, 25 * ch,
             chrmod);
    labl := 'X:dot';
    tail := dstop
  END;
  WITH pointers[iegpnt] DO
    head := (pntmod, 3 * cw, 24 * ch, lvlmod);
    pntr := (9 * cw, 0, seeline, 6 * ch + fnegbit,
             0, 16 * ch + fnegbit, seeline, 2 * ch + fnegbit,
             3 * cw + fnegbit, 24 * ch,
             chrmod);
    labl := 'Y:ieg';
    tail := dstop
  END;
  WITH pointers[icgpnt] DO
    head := (pntmod, 3 * cw, 23 * ch, lvsmod);
    pntr := (15 * cw, 0, seeline, 6 * ch + fnegbit,
             0, 15 * ch + fnegbit, seeline, 2 * ch + fnegbit,
             3 * cw + fnegbit, 23 * ch,
             chrmod);
    labl := 'Z:icg';
    tail := dstop
  END
END initpointers;

PROCEDURE shiftpointer (p : sympnt; d : direction);
  VAR tmp : integer;

{Move pointer on histograms along the time axis.}

BEGIN
  WITH pointers[p] DO
    tmp := pntr[1];
    IF d = right THEN
      tmp := tmp + cw;
      IF tmp > 59 * cw THEN ring
      ELSE pntr[1] := tmp END
    ELSE
      tmp := tmp - cw;
      IF tmp < 0 THEN ring
      ELSE pntr[1] := tmp END
    END
  END
END shiftpointer;

PROCEDURE pointerblink (p : sympnt; b : state);

{Make specific pointer blink (or not) as required.}

BEGIN
  WITH pointers[p] DO
    IF b = on THEN
      head[4] := head[4] + blink; pntr[11] := pntr[11] + blink
    ELSE
      head[4] := head[4] - blink; pntr[11] := pntr[11] - blink
    END
  END
END pointerblink;

PROCEDURE evaluatepointers (VAR p1, p2, p3 : integer);

{Sense positions of pointers and use them to discriminate morse code}

BEGIN
  p1 := pointers[dotpnt].pntr[1] / cw + 1;
  p2 := pointers[iegpnt].pntr[1] / cw + 1;
  p3 := pointers[icgpnt].pntr[1] / cw + 1
END evaluatepointers;


{Below are the procedures to select the information to be displayed on the
screen of the gt40.}

PROCEDURE displaylengths;
BEGIN
  mode := lengthmode
END displaylengths;

PROCEDURE displayhists;
BEGIN
  mode := histmode
END displayhists;

PROCEDURE displaymorse;
BEGIN
  mode := morsemode
END displaymorse;

PROCEDURE displaytext;
BEGIN
  mode := textmode
END displaytext;

PROCEDURE displayhelptext;
BEGIN
  mode := helpmode
END displayhelptext;

BEGIN {displayfiles}
  windowsiz := 40;

  WITH info DO
    ihead := (pntmod, 0, 31 * ch, chrmod);
    nhead := (pntmod, 52 * cw, 31 * ch, chrmod);
    tail := dstop
  END;
  cleardisplayfile (info.ibody, 5);

  WITH axes DO
    lhead := (pntmod, 3 * cw, 26 * ch, lvhmod);
    lines := (0 + seeline, 25 * ch + fnegbit,
              60 * cw + seeline, 0,
              0 + seeline, 25 * ch);
    fhead := (pntmod, 0, 26 * ch, chrmod);
    flabl := 'Freq';
    nhead := (pntmod, 0, 25 * ch, chrmod);
    thead := (pntmod, 3 * cw, 0, chrmod);
    tlabl := '01  05   10   15   20   25   30   35   40   45   50   55   60 Time';
    tail := dstop
  END;
  initpointers;

  WITH histfile[marks] DO
    head := (pntmod, 3 * cw, ch, lvhmod);
    tail := dstop
  END;
  WITH histfile[spaces] DO
    head := (pntmod, 3 * cw, ch, lvsmod);
    tail := dstop
  END;
  inithistfile (marks);
  inithistfile (spaces);
  factr := 2;
  rescalehistfiles (0);
  displayhelptext
END displayfiles;
MODULE morsecode;

{This module contains all procedures used in converting lengths into proper
morse code.  The procedures are called by the process 'lengthstomorse'
that implements the conversion.}

DEFINE discriminate, setmorse, lmcount, mcount, clearmorse;

USE windowsiz, marks, spaces, pntmod, chrmod, dstop, ch, cw,
    lengths, morsefile, cleardisplayfile, lcount, updatedpos, evaluatepointers;

VAR dotmax, iegmax, icgmax : integer;
    lmcount, mcount, lrpos, lcpos, mrpos, mcpos : integer;

{Dotmax is the maximum value for a dot length ie it represents the dot / dash
separation point.  Likewise ieg (inter element gap max) and icg (inter character
gap max) represent the inter element/character separation and inter character/
word separation points respectively.
Lmcount counts the number of lengths (a mark and its corresponding space) that
have been converted into morsecode.  Mcount counts the number of morsecode
symbols that are written into the morse file.  This module uses lrpos and
lcpos for stepping through the lengths files and mrpos and mcpos for moving
through the morse file.}

PROCEDURE clearmorse;

{Initialise morsefile and associated counters}

BEGIN
  cleardisplayfile (morsefile.body, 26);
  lrpos := 1; lcpos := 0; mrpos := 1; mcpos := 0;
  lmcount := 0; mcount := 0
END clearmorse;

PROCEDURE discriminate;

{Discrimination of morse lengths is effected by this procedure.  At the moment
this consists soley of sensing the positions of the pointers on the histograms
which are moved by the user.}

BEGIN
  evaluatepointers (dotmax, iegmax, icgmax)
END discriminate;

PROCEDURE setmorse;
  VAR length : integer;

{Converts 1 mark and its corresponding space into morse code symbols, writing
the result(s) into the morsefile.}

BEGIN
  inc (lmcount);
  IF lmcount > windowsiz THEN
    lmcount := 1; lrpos := 1; lcpos := 1
  ELSE
    updatedpos (lrpos, lcpos, 13)
  END;
  updatedpos (mrpos, mcpos, 26);
  IF integer (lengths[marks].body[lrpos,lcpos]) - 40B <= dotmax THEN
    morsefile.body[mrpos,mcpos] := '.'
  ELSE
    morsefile.body[mrpos,mcpos] := '_'
  END;
  inc (mcount);
  length := integer (lengths[spaces].body[lrpos,lcpos]) - 40B;
  IF length > iegmax THEN
    updatedpos (mrpos, mcpos, 26);
    IF length > icgmax THEN
      morsefile.body[mrpos,mcpos] := '|'
    ELSE
      morsefile.body[mrpos,mcpos] := ' '
    END;
    inc (mcount)
  END
END setmorse;

BEGIN {morsecode}
  WITH morsefile DO
    head := (pntmod, 0, 25*ch, chrmod);
    tail := dstop
  END;
  clearmorse
END morsecode;
MODULE text;

{This module contains all procedures needed to convert morse into text.
The procedures are used by the process 'morsetotext' that controls the
conversion.}

DEFINE settext, cleartext, mtcount;

USE pntmod, chrmod, dstop, ch, cw, morsefile, textfile, mcount,
    cleardisplayfile, updatedpos;

VAR factr, nsymbols, val : integer;  {Variables used in conversion}
    mtcount, tcount, mrpos, mcpos, trpos, tcpos : integer;

{Mtcount is a count of the number of morse symbols translated to text, the
position indicators mrpos and mcpos being used to step through the morse file.
Trpos and tcpos step through the text file and tcount is a count of the number
of symbols written into the text file (which incidentally is irrelevant!}

    codemap1 : ARRAY 0 : 1 OF char; codemap2 : ARRAY 0 : 3 OF char;
    codemap3 : ARRAY 0 : 7 OF char; codemap4 : ARRAY 0 : 15 OF char;
    codemap5 : ARRAY 0 : 31 OF char;  {Arrays for holding the codemap tables}
    codemap6 : ARRAY 1 : 8, 1 : 2 OF char;  {Last codemap is special}

PROCEDURE cleartext;

{Initialise textfile, associated counters, pointers and local variables}

BEGIN
  cleardisplayfile (textfile.body, 26);
  mrpos := 1; mcpos := 0; trpos := 1; tcpos := 0;
  mtcount := 0; tcount := 0; factr := 1; val := 0; nsymbols := 0;
END cleartext;

PROCEDURE settext;
  VAR ch, c : char;
      i : integer;

{Symbols are read from the morse file and the dots and dashes converted to
binary, a dot being 0 and a dash 1.  Variable val holds the binary number being
produced, nsymbols being a count of the number of symbols that have gone into
making this character.  Factr indicates the correct power of 2 to be added on to
val each time.  When a space or end of word symbol '|' is encountered, val is
then used to index a codemap table to give the correct character.  The table
is found by checking the value of nsymbols.  Values which contain too many
symbols to be looked up are translated as '!', values which do not correspond
to any characters in the codemaps are printed as '*'.}

BEGIN
  inc (mtcount); updatedpos (mrpos, mcpos, 26);
  ch := morsefile.body[mrpos,mcpos];
  IF (ch = ' ') OR (ch = '|') THEN
    updatedpos (trpos, tcpos, 26);
    IF nsymbols > 6 THEN
      textfile.body[trpos,tcpos] := '!'
    ELSE
      CASE nsymbols OF
        1 : BEGIN textfile.body[trpos,tcpos] := codemap1[val] END;
        2 : BEGIN textfile.body[trpos,tcpos] := codemap2[val] END;
        3 : BEGIN textfile.body[trpos,tcpos] := codemap3[val] END;
        4 : BEGIN textfile.body[trpos,tcpos] := codemap4[val] END;
        5 : BEGIN textfile.body[trpos,tcpos] := codemap5[val] END;
        6 : BEGIN
              c := '*'; i := 1;
              REPEAT
                IF codemap6[i,1] = char (val) THEN
                  c := codemap6[i,2]
                END;
                inc (i)
              UNTIL (i > 8) OR (c <> '*');
              textfile.body[trpos,tcpos] := c
            END
      END
    END;
    inc (tcount);
    IF ch = '|' THEN
      updatedpos (trpos, tcpos, 26);
      textfile.body[trpos,tcpos] := ' ';
      inc (tcount)
    END;
    nsymbols := 0; val := 0; factr := 1
  ELSE
    IF ch = '_' THEN val := val + factr END;
    inc (nsymbols); factr := factr * 2
  END
END settext;

BEGIN {text}
  codemap1 := ('E', 'T');
  codemap2 := ('I', 'N', 'A', 'M');
  codemap3 := ('S', 'D', 'R', 'G', 'U', 'K', 'W', 'O');
  codemap4 := ('H', 'B', 'L', 'Z', 'F', 'C', 'P', '*',
               'V', 'X', '*', 'Q', '*', 'Y', 'J', '*');
  codemap5 := ('5', '6', '*', '7', '*', '*', '*', '8',
               '*', '/', '*', '*', '*', '*', '*', '9',
               '4', '*', '*', '*', '*', '*', '*', '*',
               '3', '*', '*', '*', '2', '*', '1', '0');
  codemap6 := ((07C, ':'), (14C, '?'), (22C, '"'), (36C, ''''),
               (41C, '-'), (52C, '.'), (55C, '|'), (63C, ','));

  WITH textfile DO
    head := (pntmod, 0, 25*ch, chrmod);
    tail := dstop
  END;
  cleartext
END text;
MODULE helpinfo;

USE helpfile, pntmod, chrmod, dstop, ch, cw;

CONST helpmes = (
'Commands available for controlling Morse program                        ',
'                                                                        ',
'C          Continue reading Morse after having stopped                  ',
'E          Exit from program                                            ',
'G          Graphical Mode - display histograms                          ',
'H          Help Mode - display this text                                ',
'I          Select Morse input from Disk                                 ',
'K          Select Morse input from Morse Key                            ',
'L          Length Mode - display lengths                                ',
'M          Morse mode - display morse code                              ',
'N          Select Null device for Morse output                          ',
'O          Select disk for Morse output                                 ',
'R          Begin Reading morse input                                    ',
'S          Stop reading morse input                                     ',
'T          Text mode - display interpretation of morse code             ',
'W          Wind input file back to the beginning (ie re - read it)      ',
'X          Shift maximum dot length indicator (dot dash separator)      ',
'Y          Shift maximum inter element gap indicator                    ',
'Z          Shift maximum inter character gap indicator                  ',
'>          Increase window size by 10                                   ',
'<          Decrease window size by 10                                   ',
'UPARROW    Increase scale factor on histograms                          ',
'DOWNARROW  Decrease scale factor on histograms                          ');

VAR i, j : integer;

PROCEDURE inithelpfile;

{Copy above constant help message into help file to initialise it}

BEGIN
  WITH helpfile DO
    i := 1;
    REPEAT
      j := 1;
      REPEAT
        body[i,j] := helpmes[i,j];
        inc (j)
      UNTIL j > 72;
      body[i,73] := 15C;
      body[i,74] := 12C;
      inc (i)
    UNTIL i > 23
  END
END inithelpfile;

BEGIN {helpinfo}
  WITH helpfile DO
    head := (pntmod, 0, 25 * ch, chrmod);
    tail := dstop
  END;
  inithelpfile
END helpinfo;
DEVICE MODULE gt40 [4];

{ Module gt40 provides the interface to the display processor, taking
information from the display files info, lengths, axes, pointers,
histfile, morsefile, textfile and helpfile, displaying it on the
screen of the graphics processor (GT40).  The file displayed at a
particular time is indicated by the variable mode which gt40 reads.}

USE marks, spaces, info, lengths, axes, pointers, histfile, helpfile,
    dotpnt, iegpnt, icgpnt,
    morsefile, textfile,
    mode, lengthmode, histmode, morsemode, textmode, helpmode;

PROCESS screen [320B];
  VAR dpc[172000B] : integer;
BEGIN
  LOOP
    dpc := adr (info.ihead[1]);
    doio;
    IF mode = lengthmode THEN
      dpc := adr (lengths[marks].head[1]);
      doio;
      dpc := adr (lengths[spaces].head[1]);
      doio
    ELSIF mode = histmode THEN
      dpc := adr (axes.lhead[1]);
      doio;
      dpc := adr (histfile[marks].head[1]);
      doio;
      dpc := adr (histfile[spaces].head[1]);
      doio;
      dpc := adr (pointers[dotpnt].head[1]);
      doio;
      dpc := adr (pointers[iegpnt].head[1]);
      doio;
      dpc := adr (pointers[icgpnt].head[1]);
      doio
    ELSIF mode = morsemode THEN
      dpc := adr (morsefile.head[1]);
      doio
    ELSIF mode = textmode THEN
      dpc := adr (textfile.head[1]);
      doio
    ELSIF mode = helpmode THEN
      dpc := adr(helpfile.head[1]);
      doio
    END
  END
END screen;

BEGIN {gt40}
  screen
END gt40;
{Further declarations in MODULE analysis}

PROCEDURE startreading;
BEGIN
  clearlengths;
  clearhists;
  rescalehistfiles (0);
  clearmorse;
  cleartext;
  stoplengths := false; morsetodate := false; texttodate := false;
  alltodate := false;
  IF input = disk THEN send (getfromdisk)
  ELSIF input = key THEN send (getfromkey) END
END startreading;

PROCEDURE newwindow (windowinc : integer);
BEGIN
  windowsiz := windowsiz + windowinc;
  IF windowsiz = 0 THEN windowsiz := 10
  ELSIF windowsiz > 500 THEN windowsiz := 500
  END;
  rescalehistfiles (0)
END newwindow;

PROCEDURE setlength (ident : ms; newlen : integer);
  VAR oldlen : integer;
BEGIN
  IF output = disk THEN putm (newlen) END;
  setms (ident, newlen, oldlen);
  adjusthistfile (ident, oldlen);
  adjusthistfile (ident, newlen)
END setlength;

BEGIN {analysis}
END analysis;
{Further declarations in MODULE morse}

PROCEDURE initmorse;

{Called at top level to initialise the whole system on program startup.
Indicates no morse being read at the start although the key is selected}

BEGIN
  stoplengths := true; morsetodate := false; texttodate := false;
  alltodate := true;
  input := key; output := none
END initmorse;
PROCESS morsefromkey;

{ This process times the durations of the different states
of the morse key (mark or space) using the signal tick, and
by means of calls to setlength, writes them to the display
files, lengths and histfile, as well as to the histograms.
Morsefromkey is activated by the signal getfromkey. }

  VAR length : integer;
BEGIN
  LOOP
    wait (getfromkey);
    WHILE (NOT mark) AND (NOT stoplengths) DO wait (tick) END;

    WHILE NOT stoplengths DO
      length := 0;
      REPEAT wait (tick); inc (length) UNTIL NOT mark;
      IF length > 60 THEN length := 60 END;
      setlength (marks, length);
      length := 0;
      REPEAT wait (tick); inc (length) UNTIL (mark) OR (stoplengths) OR (length 
                                                                    >60);
      IF length > 60 THEN length := 60 END;
      setlength (spaces, length);
      send (processlengths);
      WHILE (NOT mark) AND (NOT stoplengths) DO wait (tick) END
    END;
    morsetodate := true; send (processlengths)
  END
END morsefromkey;
PROCESS morsefromdisk;

{This process is activated when morse is retrieved from disk
rather than being read direct from the morse key.  The process
waits on the signal getfromdisk and is paused by the boolean
stoplengths.  It also checks the boolean fileempty and stops
if it is true.  Procedure setlength is called to write the lengths
into the displayfiles as before.  It delays between each read
of mark and space so that the system doesn't run too fast for
the operator to intervene if he so wishes.}

  VAR length : integer;
BEGIN
  LOOP
    wait (getfromdisk);
    REPEAT
      getm (length);
      IF length > 0 THEN
        setlength (marks, length);
        getm (length);
        setlength (spaces, length)
      END;
      send (processlengths);
      delay (5)
    UNTIL (stoplengths) OR (fileempty);
    morsetodate := true; send (processlengths);
    IF fileempty THEN
      writeinfo ('Morse input file exhausted : Please type a command', 5);
      stoplengths := true
    END
  END
END morsefromdisk;
PROCESS lengthstomorse;

{This process reads from the lengths file and produces morsecode, using
pointers on the histograms for discriminating. The process remains 'lim'
symbols behind the key unless the boolean 'morsetodate' becomes true
when it comes up to date.  It instructs 'morsetotext' to come up to
date by setting the boolean 'texttodate'.  The process is controlled
by the signal 'processlengths'.  The conversion of lengths to morse
is carried out by the procedure setmorse (in Module morsecode).}

CONST lim = 0;
      discrimrate = 5;

BEGIN
  LOOP
    wait (processlengths);
    WHILE (lmcount < lcount - lim) OR
          ((lmcount > lcount) AND (lmcount - windowsiz < lcount - lim)) OR
          ((morsetodate) AND (lcount <> lmcount))
    DO
      IF lmcount MOD discrimrate = 0 THEN discriminate END;
      setmorse
    END;
    IF (morsetodate) AND (lcount = lmcount) THEN texttodate := true END;
    send (processmorse)
  END
END lengthstomorse;
PROCESS morsetotext;

{This process reads from the morsefile and produces text, running one
symbol behind 'lengthstomorse'.  When 'texttodate' becomes true it comes
up to date and sends the signal 'allprocessed' which informs the controlling
process (commands) that all translation is over.  The process is activated
by the signal 'processmorse'.  The conversion of morsecode to text is carried
out by procedure settext (in Module text).}

BEGIN
  LOOP
    wait (processmorse);
    WHILE (mtcount < mcount) OR
          ((texttodate) AND (mtcount <> mcount))
    DO
      settext
    END;
    IF (mtcount = mcount) AND (texttodate) THEN
      alltodate := true; send (allprocessed)
    END
  END
END morsetotext;
PROCESS commands;

{ Commands controls the whole system, reading from the
keyboard using read and then activating the necessary
processes or procedures to execute the commands. }

  CONST nl = 12C;
        cr = 15C;
        uarrow = 32C;
        darrow = 13C;
        larrow = 10C;
        rarrow = 30C;

        {Messages to be displayed on gt40}

   nohost = ('Communications not available');
   askfile = ('Please type file-spec terminated by <cr>');
   innone = ('      Input : None');
   outnone = ('     Output : None');
   inkey = ('      Input : Morse Key');
indisk = ('      Input : Disk :                                                    ');
 outdisk = ('     Output : Disk :                                                    ');
 offset = 22;  {Amount of offset in 'inline' and 'outline
                      strings to avoid overwriting 'input/output :
                      Disk :' messages}

  VAR ch : char;
      inline, outline : ARRAY 1 : 72 OF char;  {Strings for displaying filenames}
      filename : ARRAY 1 : 30 OF char;  {Holds filename (sent from here to host)}
      result : integer;  {Temp storage of result from open/create-file calls}

PROCEDURE readstring (VAR s1, s2 : ARRAY integer OF char; n : integer);
  VAR ch : char;
      i1, i2 : integer;

{Reads filename for host from keyboard.  S1 is the string to be passed to
host via the dl-11.  S2 is the string to be written on to the gt40 screen
using writeinfo, n being the line it will be written into.  The string is
terminated by <cr> which is replaced by <lf> for host.}

BEGIN
  i1 := low (s1); i2 := low (s2) + offset;
  REPEAT
    read (ch);
    IF ch = 177C THEN {rubout one character}
      IF i1 > low (s1) THEN
        dec (i1); dec (i2);
        s2[i2] := ' ';
        writeinfo (s2, n)
      END
    ELSIF (i1 < high (s1)) AND (ch <> cr) AND (ch <> nl) THEN
      s1[i1] := ch; inc (i1);
      s2[i2] := ch; inc (i2);
      writeinfo (s2, n)
    END
  UNTIL ch = cr;
  s1[i1] := nl;
END readstring;

PROCEDURE movepointer (p : sympnt);
  VAR ch : char;

{Moves one of the pointers along the time axis of the histograms.  The
addressed pointer will blink.  It is moved by using the left and right
arrows on the keyboard.  Any other character will stop addressing of
this pointer and return to 'command level'.}

BEGIN
  pointerblink (p, on);
  REPEAT
    read (ch);
    IF ch = larrow THEN shiftpointer (p, left)
    ELSIF ch = rarrow THEN shiftpointer (p, right)
    END
  UNTIL (ch <> larrow) AND (ch <> rarrow);
  pointerblink (p, off)
END movepointer;

BEGIN {commands}
  writeinfo ('For help text type "H"                (windowsize =     )', 1);
  writeinfo (inkey, 2);
  writeinfo (outnone, 3);
  writeinfo ('Please type a command',5);
  LOOP
    read (ch);
    IF (ch >= 'A') AND (ch <= 'Z') THEN {Convert to lower case}
      ch := char (integer (ch) + 40B)
    END;
    CASE ch OF
      'c' : BEGIN  {Continue reading morse after stopping}
              IF (stoplengths) AND (input <> none) THEN
                morsetodate := false; texttodate := false;
                stoplengths := false; alltodate := false;
                IF input = disk THEN send (getfromdisk)
                ELSIF input = key THEN send (getfromkey) END;
                writeinfo ('Reading morse, S to stop', 5)
              ELSE ring END
            END;

      'e' : BEGIN  {Runs the bootstrap and stops morsecom}
              stopprog
            END;

      'g' : BEGIN
              displayhists
            END;

      'h' : BEGIN
              displayhelptext
            END;

 'i', 'w' : BEGIN  {Select new input file or rewind current one (w command)}
              IF (stoplengths) AND NOT ((input <> disk) AND (ch = 'w')) THEN
                IF ch = 'i' THEN {ask for new filename etc}
                  IF output = disk THEN
                    output := none; writeinfo (outnone, 3)
                  END;
                  inline := indisk;  writeinfo (indisk, 2);
                  writeinfo (askfile, 5);  readstring (filename, inline, 2);
                END;
                result := openfile (filename);
                IF result < 1 THEN
                  ring;
                  input := none;  writeinfo (innone, 2);
                  IF result = 0 THEN writeinfo (nohost, 5)
                  ELSE writeinfo ('Cannot open file', 5) END;
                  delay (150); writeinfo ('Please type a command', 5);
                ELSE
                  input := disk;  setupmorsebuf;
                  writeinfo ('Input file selected : Please type a command', 5);
                END;
              ELSE ring END
            END;

      'k' : BEGIN  {Select input from morse key}
              IF stoplengths THEN
                input := key;
                writeinfo (inkey, 2);
                writeinfo ('Morse key enabled : Please type a command', 5)
              ELSE ring END
            END;

      'l' : BEGIN
              displaylengths
            END;

      'm' : BEGIN
              displaymorse
            END;

      'n' : BEGIN  {Output to Null}
              IF stoplengths THEN
                writeinfo (outnone, 3); output := none
              ELSE ring END
            END;

      'o' : BEGIN  {Select output to disk}
              IF stoplengths THEN
                IF input = disk THEN
                  input := none; writeinfo (innone, 2)
                END;
                outline := outdisk;  writeinfo (outline, 3);
                writeinfo (askfile, 5);  readstring (filename, outline, 3);
                result := createfile (filename);
                IF result < 1 THEN
                  ring;
                  output := none;  writeinfo (outnone, 3);
                  IF result = 0 THEN writeinfo (nohost, 5)
                  ELSE writeinfo ('Cannot create file', 5) END;
                  delay (150);
                  writeinfo ('Please type a command', 5)
                ELSE
                  output := disk;  setupmorsebuf;
                  writeinfo ('Output file selected : Please type a command', 5)
                END
              ELSE ring END
            END;

      'r' : BEGIN  {Begin reading morse}
              IF (stoplengths) AND (input <> none) THEN
                startreading;
                writeinfo ('Reading Morse, S to stop ', 5)
              ELSE ring END
            END;

      's' : BEGIN  {Stop reading morse}
              IF NOT stoplengths THEN
                stoplengths := true;
                IF NOT alltodate THEN wait (allprocessed) END;
                delay (2);  {Allow other processes to settle down}
                writeinfo ('Translation paused : Please type a command ', 5)
              ELSE ring END
            END;

      't' : BEGIN
              displaytext
            END;

      'x' : BEGIN  {Change dot dash discriminator}
              displayhists;
              movepointer (dotpnt)
            END;

      'y' : BEGIN  {Change inter element/character gap discriminator}
              displayhists;
              movepointer (iegpnt)
            END;

      'z' : BEGIN  {Change inter character/word gap discriminator}
              displayhists;
              movepointer (icgpnt)
            END;

   uarrow : BEGIN
              displayhists;
              rescalehistfiles (1)
            END;

   darrow : BEGIN
              displayhists;
              rescalehistfiles (-1)
            END;

      '>' : BEGIN
              IF stoplengths THEN newwindow (10) ELSE ring END
            END;

      '<' : BEGIN
              IF stoplengths THEN newwindow (-10) ELSE ring END
            END
    END
  END
END commands;

BEGIN {morse}
  initmorse;
  morsefromkey;
  morsefromdisk;
  lengthstomorse;
  morsetotext;
  commands
END morse.
{
.bp
}
