#include "param.h"
#include "systm.h"
#include "dir.h"
#include "signal.h"
#include "seg.h"
#include "ipm.h"
#include "sid.h"
#include "user.h"
#include "proc.h"
#include "errno.h"
#include "file.h"
#include "inode.h"
#include "tty.h"
#include "conf.h"
#include "fcntl.h"

/*
 * read system call
 */
read()
{
	rdwr(FREAD);
}

/*
 * write system call
 */
write()
{
	rdwr(FWRITE);
}

/*
 * common code for read and write calls:
 * check permissions, set base, count, and offset,
 * and switch out to readi or writei code.
 */
rdwr(mode)
	register mode;
{
	register struct file *fp;
	register struct inode *ip;
	register struct a {
		int	fdes;
		char	*cbuf;
		unsigned count;
	} *uap;
	int	type;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if (fp == NULL)
		return;
	if ((fp->f_flag & mode) == 0) {
		u.u_error = EBADF;
		return;
	}
	u.u_base = (caddr_t)uap->cbuf;
	u.u_count = uap->count;
	if (uap->count > 0x7fffffff)
		return;
	u.u_segflg =  0;
	if (fp->f_flag & FPIPE) {
		if (mode == FREAD)
			readp(fp);
		else
			writep(fp);
	} else if (fp->f_flag & FSID)  {
		u.u_error = EBADF;
		return;
	} else {
		ip = fp->f_inode;
		u.u_offset = fp->f_offset;
		if ((ip->i_mode & (IFCHR&IFBLK)) == 0)
			plock(ip);
		if (mode == FREAD)
			readi(ip);
		else
			writei(ip);
		if ((ip->i_mode & (IFCHR&IFBLK)) == 0)
			prele(ip);
		fp->f_offset += uap->count - u.u_count;
	}
	u.u_rval1 = uap->count-u.u_count;
}

/*
 * open system call
 * Note that the value of mode passed to open1 is incremented by
 * one.  This changes the least significant hex digit to
 * match FREAD and FWRITE.  It leaves the rest unchanged.
 */
open()
{
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	mode;
		int	fmode;	/* for O_CREAT */
	} *uap;
	register m;

	uap = (struct a *)u.u_ap;
	if (((m = uap->mode) & 03) == 03) {
		u.u_error = EINVAL;
		return;
	}
	ip = namei(uchar, uap->mode & O_CREAT ? 1 : 0);
	if (ip == NULL) {
		if (u.u_error)
			return;
		if (m & O_CREAT) {
			ip = maknode(uap->fmode & 07777 & (~ISVTX));
			if (ip == NULL)
				return;
			open1(ip, m + 1, 2);
			return;
		} else
			return;
	}
	if (m & O_CREAT && m & O_EXCL) {
		iput(ip);
		u.u_error = EEXIST;
		return;
	}
	open1(ip, m+1, m & O_TRUNC ? 1 : 0);
}

/*
 * creat system call
 */
creat()
{
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	fmode;
	} *uap;

	uap = (struct a *)u.u_ap;
	ip = namei(uchar, 1);
	if (ip == NULL) {
		if (u.u_error)
			return;
		ip = maknode(uap->fmode & 07777 & (~ISVTX));
		if (ip == NULL)
			return;
		open1(ip, FWRITE, 2);
	} else
		open1(ip, FWRITE, 1);
}

/*
 * common code for open and creat.
 * Check permissions, allocate an open file structure,
 * and call the device open routine if any.
 */
open1(ip, mode, trf)
register struct inode *ip;
register mode;
{
	register struct file *fp;
	int i;

	if (trf != 2) {
		if (mode & FREAD)
			access(ip, IREAD);
		if (mode & FWRITE) {
			access(ip, IWRITE);
			if ((ip->i_mode & IFMT) == IFDIR)
				u.u_error = EISDIR;
		}
	}
	if (u.u_error)
		goto out;
	if (trf == 1)
		itrunc(ip);
	prele(ip);
	if ((fp = falloc()) == NULL)
		goto out;
	fp->f_flag = mode & (FREAD | FWRITE);
	if (mode & O_APPEND)
		fp->f_flag |= FAPPEND;
	fp->f_inode = ip;
	i = u.u_rval1;
	openi(ip, mode & FWRITE);
	if (u.u_error == 0)
		return;
	u.u_ofile[i] = NULL;
	fp->f_count--;

out:
	iput(ip);
}

/*
 * close system call
 */
close()
{
	register struct file *fp;
	register struct a {
		int	fdes;
	} *uap;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if (fp == NULL)
		return;
	u.u_ofile[uap->fdes] = NULL;
	closef(fp);
}

/*
 * seek system call
 */
seek()
{
	register struct file *fp;
	register struct inode *ip;
	register struct a {
		int	fdes;
		off_t	off;
		int	sbase;
	} *uap;

	uap = (struct a *)u.u_ap;
	fp = getf(uap->fdes);
	if (fp == NULL)
		return;
	ip = fp->f_inode;
	if (fp->f_flag & FPIPE) {
		u.u_error = ESPIPE;
		return;
	}
	if (uap->sbase == 1)
		uap->off += fp->f_offset;
	else if (uap->sbase == 2)
		uap->off += fp->f_inode->i_size;
	fp->f_offset = uap->off;
	u.u_roff = uap->off;
}

/*
 * link system call
 */
link()
{
	register struct inode *ip, *xp;
	register struct a {
		char	*target;
		char	*linkname;
	} *uap;

	uap = (struct a *)u.u_ap;
	ip = namei(uchar, 0);
	if (ip == NULL)
		return;
	if ((ip->i_mode & IFMT) == IFDIR && !suser())
		goto out;
	/*
	 * Unlock to avoid possibly hanging namei.
	 * Sadly, this means races. (Suppose someone
	 * deletes the file in the meantime?)
	 * Nor can it be locked again later
	 * because then there will be deadly
	 * embraces.
	 */
	prele(ip);
	u.u_dirp = (caddr_t) uap->linkname;
	xp = namei(uchar, 1);
	if (xp != NULL) {
		u.u_error = EEXIST;
		iput(xp);
		goto out;
	}
	if (u.u_error)
		goto out;
	if (u.u_pdir->i_dev != ip->i_dev) {
		iput(u.u_pdir);
		u.u_error = EXDEV;
		goto out;
	}
	wdir(ip);
	if (u.u_error == 0) {
		/* should check for max link here */
		ip->i_nlink++;
		ip->i_flag |= ICHG;
	}
out:
	iput(ip);
}

/*
 * mknod system call
 */
mknod()
{
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	fmode;
		int	dev;
	} *uap;

	uap = (struct a *)u.u_ap;
	if (suser()) {
		ip = namei(uchar, 1);
		if (ip != NULL) {
			iput(ip);
			u.u_error = EEXIST;
			goto out;
		}
	}
	if (u.u_error)
		return;
	ip = maknode(uap->fmode);
	if (ip == NULL)
		return;
	ip->i_rdev = (dev_t)uap->dev;
out:
	iput(ip);
}

/*
 * access system call
 */
saccess()
{
	register svuid, svgid;
	register struct inode *ip;
	register struct a {
		char	*fname;
		int	fmode;
	} *uap;

	uap = (struct a *)u.u_ap;
	svuid = u.u_uid;
	svgid = u.u_gid;
	u.u_uid = u.u_ruid;
	u.u_gid = u.u_rgid;
	ip = namei(uchar, 0);
	if (ip != NULL) {
		if (uap->fmode&(IREAD>>6))
			access(ip, IREAD);
		if (uap->fmode&(IWRITE>>6))
			access(ip, IWRITE);
		if (uap->fmode&(IEXEC>>6))
			access(ip, IEXEC);
		iput(ip);
	}
	u.u_uid = svuid;
	u.u_gid = svgid;
}

/*
 * ioctl system call
 * Check legality, execute common code, and switch out to individual
 * device routine.
 */
ioctl()
{
	register struct file *fp;
	register struct inode *ip;
	register struct a {
		int	fdes;
		int	cmd;
		caddr_t	cmarg;
	} *uap;
	register dev_t dev;
	register fmt;

	uap = (struct a *)u.u_ap;
	if ((fp = getf(uap->fdes)) == NULL)
		return;
	if (uap->cmd==FIOCLEX) {
		u.u_pofile[uap->fdes] |= EXCLOSE;
		return;
	}
	if (uap->cmd==FIONCLEX) {
		u.u_pofile[uap->fdes] &= ~EXCLOSE;
		return;
	}
	if (fp->f_flag & FSID) {
		sidioctl(fp, uap->cmd, uap->cmarg);
		return;
	}
	ip = fp->f_inode;
        fmt = ip->i_mode & IFMT;
        if (fmt != IFCHR ) {
                u.u_error = ENOTTY;
                return;
        }
        dev = (dev_t)ip->i_rdev;
	if (ip->i_stream) {
		stioctl(ip->i_stream, uap->cmd, uap->cmarg, dev);
		return;
	}
        (*cdevsw[major(dev)].d_ioctl)(dev, uap->cmd, uap->cmarg, fp->f_flag);
}

/*
 * select system call
 */
int	selwait;

#define	bitget(x, y)	(copyin((caddr_t)(x), (caddr_t)(y), (nfds+31)>>5<<2))
#define	bitset(x, y)	(copyout((caddr_t)(x), (caddr_t)(y), (nfds+31)>>5<<2))


select()
{
	register struct a {
		int	nfds;
		fd_set	*readfds;
		fd_set	*writefds;
		int	timeout;
	} *uap;
	fd_set rfds, wfds, readbits, writebits;
	int nfds, n;

	uap = (struct a *) u.u_ap;
	nfds = min(NOFILE, uap->nfds);
	if (uap->readfds == NULL)
		FD_ZERO(rfds);
	else if (bitget(uap->readfds, &rfds) < 0) {
		u.u_error = EFAULT;
		return;
	}
	if (uap->writefds == NULL) {
		FD_ZERO(wfds);
	} else if (bitget(uap->writefds, &wfds) < 0) {
		u.u_error = EFAULT;
		return;
	}
	FD_ZERO(readbits);
	FD_ZERO(writebits);
	u.u_procp->p_seltim = (uap->timeout + 999) / 1000;
retry:
	u.u_procp->p_flag |= SSEL;
	n = readyscan(nfds, rfds, &readbits, FREAD);
	n += readyscan(nfds, wfds, &writebits, FWRITE);
	splimp();
	if (n > 0 || u.u_procp->p_seltim == 0) {
		spl0();
		if (uap->readfds != NULL)
		if (bitset(&readbits, uap->readfds) < 0) {
			u.u_error = EFAULT;
			return;
		}
		if (uap->writefds != NULL)
		if (bitset(&writebits, uap->writefds) < 0) {
			u.u_error = EFAULT;
			return;
		}
		u.u_rval1 = n;
		return;
	}
	if ((u.u_procp->p_flag & SSEL) == 0) {
		spl0();
		goto retry;
	}
	sleep(&selwait, PZERO+1);
	spl0();
	goto retry;
}

#undef	bitset
#undef	bitget


readyscan(n, fds, bits, mode)	/* check requested fds */
	register n;
	fd_set fds;
	register fd_set *bits;
{
	register struct file *fp;
	register struct stdata *sp;
	register count = 0;

	while (n--) {
		if (! FD_ISSET(n, fds))
			continue;
		fp = u.u_ofile[n];
		if (fp == NULL || fp->f_inode == NULL) {
			u.u_error = EBADF;
			return;
		}
		if (fp->f_flag & FSID) {
			if (sidselect(fp, mode)) {
				FD_SET(n, *bits);
				count++;
			}
		} else if (sp = fp->f_inode->i_stream) {
			if (stselect(sp, mode)) {
				FD_SET(n, *bits);
				count++;
			}
		} else {
			FD_SET(n, *bits);
			count++;
                }
	}
	return count;
}
