/* udp.c: udp_ld and udp dev rolled into a single source file */
/*		some of these routines use a lot of register
 *		If the code is ported to a machine with a small
 *		number of registers, it may execute faster if the
 * 		routines are broken down into separate functions.
 */
#include "param.h"
#include "types.h"
#include "errno.h"
#include "in.h"
#include "inio.h"
#include "ip_var.h"
#include "stream.h"
#include "udp_user.h"
#include "proc.h"
#include "inode.h"
#include "dir.h"
#include "ipm.h"
#include "sid.h"
#include "seg.h"
#include "signal.h"
#include "user.h"


#define	IP_UDP_SIZE 	(IP_SIZE + UDP_SIZE)
#define	IP_SIZE		(20)
#define	UDP_SIZE	(8)

#define	respond(q, bp)\
	((*OTHERQ(q)->next->qinfo->putp)(OTHERQ(q)->next, (bp)))

struct	udpdev	udpdev[NUDPDEV];

/* values for state */

#define	CLOSED		0	/* can open */
#define	UNBOUND		1	/* opened, but waiting first write */
#define	DATAGRAM	2	/* setup for datagram messages */
#define	CONNECTED	3	/* setup for auto-header messages */
#define	LISTENING	4	/* waiting for first recv'ed to connect */

struct	udplds	{		/* udp line discipline state */
	struct	udpdev	*dev;	/* destination device for current packet */
	int		state;	/* what's happening now */
	struct	block	*bp;	/* lonely ip header */
} udplds;

/* values for state */

#define	UIDLE		0	/* Nothing going on */
#define	UIPSEEN		1	/* src and dest are valid */
#define	UHDRSEEN	2	/* dev is valid */
#define	UTOSSING	3	/* trowing all blocks away */

struct	queue	*udpqueue;	/* down stream connection */

int	udpldups(), udpldqopen(), udpldqclose();
extern	putq(), srv(), nullsys();

/*
 * Udp line discipline.  These are the routines that make up the
 * udp modlue.  They accecpt incomming ip packets and hang them
 * on the proper udp socket.
 */

struct	qinit	udpldqinit[] = {
/* up */ { putq, udpldups, udpldqopen, udpldqclose, 2*1520, 64},
/* dn */ { putq, srv, nullsys, nullsys, 2 * 1520, 64 }
};

udpldqopen(dev, q)
	dev_t dev;
	register struct queue *q;
{
	udpqueue = OTHERQ(q);
	return 0;
}

udpldqclose(q)
	register struct queue *q;
{
	udpqueue = NULL;
	return 0;
}

udpldups(q)	/* service incoming udp datagrams */
	register struct queue *q;
{
	register struct udpdev *dp;
	struct block *bp;
	register struct iphdr *ip;
	register struct udp *up;
	register len;

	while (q->first && (q->next->flag & QFULL) == 0) {
		bp = getq(q);
		if (bp->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, bp);
			continue;
		}
		udpld_input(bp);
	}
	if (q->first)
		q->next->flag |= QWANTW;
}


udpld_input(bp)	/* process incoming udp block */
	register struct block *bp;
{
	register struct udplds *u;
	register delim;

	u = &udplds;
	delim = bp->class & S_DELIM;
	switch (u->state) {
	case UIDLE: {			/* first block */
		register struct iphdr *ip;
		register len;

		ip = (struct iphdr *)bp->rptr;
		len = (ip->ver_ihl & 0xf) << 2;
		u->state = UIPSEEN;
		u->bp = bp;
		if (bp->wptr - bp->rptr == len)
			return;
		/* fall through */
	}
	case UIPSEEN: {
		register struct udp *up;
		register in_addr src, dest;
		register struct iphdr *ip;
		register struct udpdev *dp;

		ip = (struct iphdr *) u->bp->rptr;
		up = (u->bp == bp)
			? (struct udp *) (bp->rptr + IP_SIZE)
			: (struct udp *)  bp->rptr;
		src = ip->src;
		dest = ip->dest;
		for (dp = udpdev; dp < &udpdev[NUDPDEV]; dp++) {
			if (dp->state == CLOSED)
				continue;
			if (dp->lport != up->sport)
				continue;
			if (dp->laddr != dest)
				continue;
			break;
		}
		if (dp >= &udpdev[NUDPDEV])
			/* Well, maybe an INADDR_ANY wants it. */
			for (dp = udpdev; dp < &udpdev[NUDPDEV]; dp++) {
				if (dp->state == CLOSED)
					continue;
				if (dp->laddr != INADDR_ANY)
					continue;
				if (dp->lport == up->dport)
					break;
			}
		if (dp >= &udpdev[NUDPDEV]) {
			if (u->bp != bp) {
				freeb(u->bp);
				u->bp = NULL;
			}
			freeb(bp);	/* drop it */
			u->state = delim ? UIDLE : UTOSSING;
			return;
		}
		if (u->bp != bp)	/* ip in a diffrent block */
			(*dp->queue->qinfo->putp)(dp->queue, u->bp);
		u->bp == NULL;
		(*dp->queue->qinfo->putp)(dp->queue, bp);
		if (!delim) {
			u->state = UHDRSEEN;
			u->dev = dp;
		} else
			u->state = UIDLE;
		return;
	}
	case UHDRSEEN:
		(*u->dev->queue->qinfo->putp)(u->dev->queue, bp);
		if (delim)
			u->state = UIDLE;
		return;
	case UTOSSING:
		freeb(bp);
		if (delim)
			u->state = UIDLE;
	}
}

/*
 * udp device routines.  This is the pseudo device driver
 * that is the read/write interface to the user datagram protocol.
 * These streams decend to a single udp pushed on /dev/ipudp.
 * Upstream ip packets get routined based on there port number.
 */

int	udpdns(), udpups(), udpqopen(), udpqclose();

struct	qinit	udpqinit[] = {
/* up */ { putq, udpups, udpqopen, udpqclose, 2*1520, 64},
/* dn */ { putq, udpdns, nullsys, nullsys, 2*1520, 64}
};

udpqopen(dev, q)
	dev_t dev;
	register struct queue *q;
{
	register socket;
	register struct udpdev *p;

	if ((socket = minor(dev)) < 0 || socket > NUDPDEV) {
		u.u_error = ENXIO;
		return;
	}
	if ((p = &udpdev[socket])->state != CLOSED) {
		u.u_error = ENXIO;
		return;
	}
	if (udpqueue == NULL) {
		u.u_error = ENXIO;
		return;
	}
	p->state = 0;	/* make sure it's zero */
	p->laddr = 0;
	p->lport = 0;
	p->faddr = 0;
	p->fport = 0;
	q->ptr = (char *)p;
	OTHERQ(q)->ptr = (char *)p;
	p->queue = q;
	OTHERQ(q->next)->flag |= QBIGB;
}

udpopen(dev)	/* device open.  Enforce open-once */
	dev_t dev;
{
	register socket;

	if ((socket = minor(dev)) < 0 || socket > NUDPDEV) {
		u.u_error = ENXIO;
		return;
	}
	if (udpdev[socket].state) {
		u.u_error = ENXIO;
		return;
	}
	udpdev[socket].state = UNBOUND;
}

udpqclose(dev, q)		/* shut the device down */
	dev_t dev;
	register struct queue *q;
{
	register socket;

	if ((socket = minor(dev)) < 0 || socket > NUDPDEV)
		return ;
	udpdev[socket].state = CLOSED;
}

/*
 * ip/udp/data frames queued.
 * if datagram
 *	udpaddr/data passed along
 * if connected
 *	data passed along
 */

udpups(q)		/* service upstream datagrams */
	register struct queue *q;
{
	register struct block *bp;

	while (q->first && (q->next->flag & QFULL) == 0) {
#ifdef notdef
		if (q->first->type == M_IOCTL) {
			udp_ioctl(q, bp);
			continue;
		}
#endif
		if (q->first->type != M_DATA) {
			(*q->next->qinfo->putp)(q->next, getq(q));
			continue;
		}
		/* must be a data block */
		udp_input(q, getq(q));
	}
	if (q->first)
		q->next->flag |= QWANTW;
}


udp_input(q, bp)	/* convert blocks and pass them up stream */
	register struct queue *q;
	register struct block *bp;
{
	register struct udpdev *dp;
	register delim;

	dp = (struct udpdev *)q->ptr;
	delim = bp->class & S_DELIM;
	switch (dp->qstate) {
	case UF_IDLE: {
		register struct iphdr *ip;

		ip = (struct iphdr *)bp->rptr;
		dp->src = ip->src;
		dp->dest = ip->dest;
		bp->rptr += (ip->ver_ihl & 0xf) << 2;
		if (bp->wptr - bp->rptr <= 0) {
			freeb(bp);
			if (delim)
				return;
			dp->qstate = UF_IPSEEN;
			return;
		}
		/* fall through */
	}
	case UF_IPSEEN: {	/* now look at the udp header */
		register in_addr src, dest;
		register struct udp *up;
		register struct udpuser *uu;
		register struct block *bp2;

		src = dp->src;
		dest = dp->dest;
		up = (struct udp *)(bp->rptr);
		dp->len = up->udplen;
		switch (dp->state) {
		case CLOSED:
			freeb(bp);
			dp->qstate = delim ? UF_IDLE : UF_TOSS;
			return;
		case DATAGRAM:
			((struct udpaddr *)up)->port = up->sport;
			((struct udpaddr *)up)->host = src;
			dp->len -= bp->wptr - bp->rptr;
			(*q->next->qinfo->putp)(q->next, bp);
			break;
		case LISTENING:	/* assertion: first message recv */
			dp->state = CONNECTED;
			dp->faddr = src;
			if (dp->laddr == INADDR_ANY)
				dp->laddr = dest;
			dp->fport = up->sport;
			bp2 = allocb(sizeof (struct udpuser));
			uu = (struct udpuser *)bp2->rptr;
			bp2->wptr += sizeof (struct udpuser);
			uu->code = UDPC_OK;
			uu->faddr = dp->faddr;
			uu->fport = dp->fport;
			uu->laddr = dp->laddr;
			uu->lport = dp->lport;
			(*q->next->qinfo->putp)(q->next, bp2);
			/* fall thru */
		case CONNECTED:	/* check orig */
			if (src != dp->faddr 
			||  up->sport != dp->fport) {
				freeb(bp);	/* don't want to see it */
				dp->qstate = delim ? UF_IDLE : UF_TOSS;
				return;
			}
			dp->len -= bp->wptr - bp->rptr;
			bp->rptr += 8;
			if (bp->wptr - bp->rptr)
				(*q->next->qinfo->putp)(q->next, bp);
			break;
		default:
			panic("udp_input: can't happen");
		}
		dp->qstate = delim ? UF_IDLE : UF_HDRSEEN;
		return;
	}
	case UF_HDRSEEN: {
		register n;

		n = min(bp->wptr - bp->rptr, dp->len);
		dp->len -= n;
		bp->wptr = bp->rptr + n;
		if (n > 0)
			(*q->next->qinfo->putp)(q->next, bp);
		else
			freeb(bp);
		if (delim)
			dp->qstate = UF_IDLE;
		return;
	}
	case UF_TOSS:	/* don't care about them */
		freeb(bp);
		if (delim)
			dp->qstate = UF_IDLE;
		return;
	default:
		printf("udp_input: bad qstate %d\n", dp->qstate);
	}
}

udp_ioctl(q, bp)		/* process ioctl requests */
	register struct queue *q;
	register struct block *bp;
{
	bp->wptr = bp->rptr;
	bp->type = M_IOCNAK;
	qreply(q, bp);
}



udpdns(q)		/* service out going datagrams */
	register struct queue *q;
{
	register struct udpdev *dp;
	register struct block *bp;
	udp_port udp_gport();

	dp = (struct udpdev *)q->ptr;
	while (q->first) {
		if (q->first->type == M_IOCTL) {
			udp_ioctl(q, getq(q));
			continue;
		}
		if (q->first->type != M_DATA) {
			freeb(getq(q));
			continue;
		}
		if (udpqueue->flag & QFULL)
			return;
		/* M_DATA then */
		bp = getq(q);
		switch (dp->state) {
		case CLOSED:
			panic("udpdns: can't happen");
			break;
		case UNBOUND: {
			register struct udpuser *uu;

			uu = (struct udpuser *)bp->rptr;
			switch (uu->code) {
			case UDPC_LISTEN:
				uu->code = UDPC_OK;
				dp->laddr = uu->laddr;
				dp->lport = udp_gport(uu->lport);
				if (dp->lport) {
					dp->state = LISTENING;
					dp->faddr = 0;
					dp->fport = 0;
				} else
					uu->code == UDPC_SORRY;
				respond(q, bp);
				break;
			case UDPC_CONNECT:
				uu->code = UDPC_OK;
				dp->laddr = uu->laddr;
				dp->lport = udp_gport(uu->lport);
				if (dp->lport) {
					dp->faddr = uu->faddr;
					dp->fport = uu->fport;
					dp->state = CONNECTED;
				} else
					uu->code = UDPC_SORRY;
				respond(q, bp);
				break;
			case UDPC_DATAGRAM:
				dp->laddr = uu->laddr;
				dp->lport = udp_gport(uu->lport);
				if (dp->lport) {
					dp->fport = 0;
					dp->faddr = 0;
					dp->state = DATAGRAM;
					uu->code = UDPC_OK;
				} else
					uu->code = UDPC_SORRY;
				respond(q, bp);
				break;
			default:
				uu->code = UDPC_SORRY;
				respond(q, bp);
				break;
			}
			break;
		}
		case CONNECTED: {
			register struct block *bp2;
			register struct iphdr *ip;
			register struct udp *up;

			bp2 = allocb(IP_UDP_SIZE);
			bp2->wptr += IP_UDP_SIZE;
			ip = (struct iphdr *)bp2->rptr;
			up = (struct udp *)((char *)ip + IP_SIZE);
			ip->tlen = IP_UDP_SIZE + bp->wptr - bp->rptr;
			ip->ver_ihl = (20 >> 2);
			ip->proto = IP_PROTO_UDP;
			ip->src = dp->laddr;
			ip->dest = dp->faddr;
			up->sport = dp->lport;
			up->dport = dp->fport;
			up->udplen = 8 + bp->wptr - bp->rptr;
			up->ucksum = 0;
			(*udpqueue->qinfo->putp)(udpqueue, bp2);
			(*udpqueue->qinfo->putp)(udpqueue, bp);
			break;
		}
		case LISTENING:
			freeb(bp);	/* don't know who to send it to */
			break;
		/* this has a bug: What if the write is > 1024? */
		case DATAGRAM: {
			register struct iphdr *ip;
			register struct udp *up;
			register struct udpaddr *ua;
			register struct block *bp2;
			register host, port;

			/* turn udpaddr structure into udp header */
			ua = (struct udpaddr *)bp->rptr;
			port = ua->port;
			host = ua->host;
			up = (struct udp *)bp->rptr;
			up->dport = port;
			up->sport = dp->lport;
			up->udplen = bp->wptr - bp->rptr; /* includes udphdr */
			up->ucksum = 0;	/* we don't so this yet */
			/* get block and make an ip header */
			bp2 = allocb(sizeof(struct iphdr));
			bp2->wptr += sizeof(struct iphdr);
			ip = (struct iphdr *)bp2->rptr;
			ip->ver_ihl = (20 >> 2);
			ip->tlen = sizeof (struct iphdr) + up->udplen;
			ip->proto = IP_PROTO_UDP;
			ip->src = dp->laddr;
			ip->dest = host;
			/* send iphdr then udp header + data */
			(*udpqueue->qinfo->putp)(udpqueue, bp2);
			(*udpqueue->qinfo->putp)(udpqueue, bp);
			break;
		}
		default:
			panic("udpdns: bad state");
			break;
		}
	}
}

/*
 * if port == 0 assign unused port
 * if port != 0 just check to make sure it isn't already in use.
 */

udp_port
udp_gport(port)		/* handle setting port */
	register udp_port port;
{
	static int scrport = 1024;	/* scratch port */
	register struct udpdev *dp;

	if (port != 0) {	/* just validate port given */
		for (dp = udpdev; dp < &udpdev[NUDPDEV]; dp++)
			if (port == dp->lport)
				return 0;	/* sorry */
		return port;
	}
	/* here means user wants us to assign port for him */
	/* port name space is so large we will never run out of numbers */
	for (;;) {
		port = scrport;
		if (++scrport > 0xffff)
			scrport = 512;
		for (dp = udpdev; dp < &udpdev[NUDPDEV]; dp++)
			if (dp->state != CLOSED && port == dp->lport)
				break;
		if (dp >= &udpdev[NUDPDEV])
			return port;
	}
}
