/*
 * savecore
 */

#include <stdio.h>
#include <nlist.h>
#include <sys/param.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <sys/filsys.h>
#include <time.h>
#include <ctype.h>

#define	DAY	(60L*60L*24L)
#define	LEEWAY	(3*DAY)

#define eq(a,b) (!strcmp(a,b))
#define ok(number) ((number)&0x7fffffff)

#define SHUTDOWNLOG "/usr/adm/shutdownlog"
#define	DEBUG

struct nlist nl[] = {
#define X_DUMPDEV	0
	{ "_dumpdev" },
#define X_DUMPLO	1
	{ "_dumplo" },
#define X_END		2
	{ "_end" },
#define X_TIME		3
	{ "_time" },
	{ 0 },
};


char	*system;
char	*dirname;			/* directory to save dumps in */
char	*ddname;			/* name of dump device */
char	*find_dev();
dev_t	dumpdev;			/* dump device */
int	dumplo;				/* where dump starts on dumpdev */
time_t	now;				/* current date */
time_t	dumptime;			/* crash date from corefile */
char	*path();
unsigned malloc();
char	*ctime();
off_t	lseek();
off_t	Lseek();

main(argc, argv)
	char **argv;
	int argc;
{

	if (argc != 2 && argc != 3) {
		fprintf(stderr, "usage: savecore dirname [ system ]\n");
		exit(1);
	}
	dirname = argv[1];
	if (argc == 3)
		system = argv[2];
	if (access(dirname, 2) < 0) {
		perror(dirname);
		exit(1);
	}
	(void) time(&now);
	read_kmem();
	if (get_crashtime() && check_space()) {
		log_entry();
		save_core();
	} else
		exit(1);
}

char *
find_dev(dev, type)
	register dev_t dev;
	register int type;
{
	register int dfd = Open("/dev", 0);
	struct direct dir;
	struct stat statb;
	static char devname[DIRSIZ + 1];
	char *dp;

	strcpy(devname, "/dev/");
	while(Read(dfd, (char *)&dir, sizeof dir) > 0) {
		if (dir.d_ino == 0)
			continue;
		strncpy(devname + 5, dir.d_name, DIRSIZ);
		devname[DIRSIZ] = '\0';
		if (stat(devname, &statb)) {
			perror(devname);
			continue;
		}
		if ((statb.st_mode&S_IFMT) != type)
			continue;
		if (dev == statb.st_rdev) {
			close(dfd);
			dp = (char *)malloc(strlen(devname)+1);
			strcpy(dp, devname);
			return dp;
		}
	}
	close(dfd);
	fprintf(stderr, "Can't find device %d,%d\n", major(dev), minor(dev));
	exit(1);
	/*NOTREACHED*/
}

#define	X	(0x80000000)	/* Kludge, remove */

read_kmem()
{
	int kmem;
	FILE *fp;
	register char *cp;

	nlist("/unix", nl);
	if (nl[X_DUMPDEV].n_value == 0) {
		fprintf(stderr, "/unix: dumpdev not in namelist\n");
		exit(1);
	}
	if (nl[X_DUMPLO].n_value == 0) {
		fprintf(stderr, "/unix: dumplo not in namelist\n");
		exit(1);
	}
	if (nl[X_END].n_value == 0) {
		fprintf(stderr, "/unix: end not in namelist\n");
		exit(1);
	}
	if (nl[X_TIME].n_value == 0) {
		fprintf(stderr, "/unix: time not in namelist\n");
		exit(1);
	}
	kmem = Open("/dev/kmem", 0);
	Lseek(kmem, (long)nl[X_DUMPDEV].n_value - X, 0);
	Read(kmem, (char *)&dumpdev, sizeof dumpdev);
	Lseek(kmem, (long)nl[X_DUMPLO].n_value - X, 0);
	Read(kmem, (char *)&dumplo, sizeof dumplo);
	dumplo *= BSIZE;
	ddname = find_dev(dumpdev, S_IFBLK);
}


char *
path(file)
	char *file;
{
	register char *cp = (char *)malloc(strlen(file) + strlen(dirname) + 2);

	(void) strcpy(cp, dirname);
	(void) strcat(cp, "/");
	(void) strcat(cp, file);
	return cp;
}

check_space()
{
	struct stat dsb;
	register char *ddev;
	register int dfd;
	struct filsys sblk;

	if (stat(dirname, &dsb) < 0) {
		perror(dirname);
		exit(1);
	}
	ddev = find_dev(dsb.st_dev, S_IFBLK);
	dfd = Open(ddev, 0);
	Lseek(dfd, 1L<<BSHIFT, 0);
	Read(dfd, (char *)&sblk, sizeof sblk);
	close(dfd);
	if (read_number("minfree") > sblk.s_tfree) {
		fprintf(stderr, "Dump omitted, not enough space on device\n");
		return (0);
	}
	return (1);
}

read_number(fn)
	char *fn;
{
	char lin[80];
	register FILE *fp;

	if ((fp = fopen(path(fn), "r")) == NULL)
		return 0;
	if (fgets(lin, 80, fp) == NULL) {
		fclose(fp);
		return 0;
	}
	fclose(fp);
	return atoi(lin);
}

save_core()
{
	register int n;
	char buffer[8*BSIZE];
	register char *cp = buffer;
	register int ifd, ofd, bounds;
	register FILE *fp;
	register dumpsize;

	bounds = read_number("bounds");
	ifd = Open("/unix", 0);
	ofd = Create(path(sprintf(cp, "unix.%d", bounds)), 0666);
	while((n = Read(ifd, cp, sizeof(buffer))) > 0)
		Write(ofd, cp, n);
	close(ifd);
	close(ofd);
	dumpsize = nl[X_END].n_value - 0x80000000;
	ifd = Open(system?system:ddname, 0);
	ofd = Create(path(sprintf(cp, "core.%d", bounds)), 0666);
	Lseek(ifd, (off_t)dumplo, 0);
	printf("Saving %d bytes of image in core.%d\n", dumpsize, bounds);
	dumpsize = (dumpsize + 511) & ~0777;
	while(dumpsize > 0) {
		n = Read(ifd, cp, (dumpsize > 8*BSIZE ? 8*BSIZE : dumpsize));
		Write(ofd, cp, n);
		dumpsize -= n;
	}
	close(ifd);
	close(ofd);
	fp = fopen(path("bounds"), "w");
	fprintf(fp, "%d\n", bounds+1);
	fclose(fp);
}

char *days[] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

char *months[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep",
	"Oct", "Nov", "Dec"
};

log_entry()
{
	FILE *fp;
	struct tm *tm, *localtime();

	tm = localtime(&now);
	fp = fopen("/usr/adm/shutdownlog", "a");
	if (fp == 0)
		return;
	fseek(fp, 0L, 2);
	fprintf(fp, "%02d:%02d  %s %s %2d, %4d.  Reboot; ", tm->tm_hour,
		tm->tm_min, days[tm->tm_wday], months[tm->tm_mon],
		tm->tm_mday, tm->tm_year + 1900);
	fprintf(fp, "crashtime: %d\n", dumptime);
	fclose(fp);
}

time_t
last_saved()
{
	FILE *fp;
	register i;
	register char *timep;
	char line[80];

	fp = fopen("/usr/adm/shutdownlog", "r");
	if (fp == 0)
		return;
	if (fgets(line, sizeof(line), fp) == NULL) {
		fclose(fp);
		return 0;
	}
	if (!strlen(line))
		return 0;
	fseek(fp, (long)-(strlen(line)), 2);
	memset(line, NULL, sizeof(line));
	fgets(line, sizeof(line), fp);
	fclose(fp);
	timep = &line[strlen(line) - 1];
	*timep-- = NULL;	/* take off <cr> */
	while (isdigit(*timep))
		timep--;
	return(atol(timep));
}


get_crashtime()
{
	int dumpfd;

	if ((dumpfd = Open(ddname, 0)) < 0) 
		printf("get_crashtime: failed to open %s\n", ddname);
	
	Lseek(dumpfd, (off_t)(dumplo+ctob(USIZE)+ok(nl[X_TIME].n_value)), 0);
	if (!Read(dumpfd, (char *)&dumptime, sizeof dumptime)) 
		printf("get_crashtime: failed read of %s\n", ddname);
	close(dumpfd);
	if (dumptime == 0L) {
#ifdef DEBUG
		printf("dump time is 0\n");
#endif
		return 0;
	}
	printf("System core file time %s", ctime(&dumptime));
	if (dumptime < now - LEEWAY || dumptime > now + LEEWAY) {
		printf("Dump time is unreasonable\n");
		return 0;
	}
	if (last_saved() == dumptime) {
		printf("savecore: already saved dump for %s\n", 
			ctime(&dumptime));
		return 0;
	}
	return 1;
}

/*
 * Versions of std routines that exit on error.
 */

Open(name, rw)
	char *name;
	int rw;
{
	int fd;

	if ((fd = open(name, rw)) < 0) {
		perror(name);
		exit(1);
	}
	return fd;
}

Read(fd, buff, size)
	int fd, size;
	char *buff;
{
	int ret;

	if ((ret = read(fd, buff, size)) < 0) {
		perror("read");
		abort (0);
		exit(1);
	}
	return ret;
}

off_t
Lseek(fd, off, flag)
	int fd, flag;
	long off;
{
	long ret;

	if ((ret = lseek(fd, off, flag)) == -1L) {
		perror("lseek");
		exit(1);
	}
	return ret;
}

Create(file, mode)
	char *file;
	int mode;
{
	register int fd;

	if ((fd = creat(file, mode)) < 0) {
		perror(file);
		exit(1);
	}
	return fd;
}

Write(fd, buf, size)
	int fd, size;
	char *buf;

{

	if (write(fd, buf, size) < size) {
		perror("write");
		exit(1);
	}
}


