
/*
 * $Log:	ops.c,v $
 * Revision 1.1  86/03/11  10:07:25  root
 * Initial revision
 * 
 */


/*
 * Machine operators.
 */
#include "defs.h"
#include "machine.h"
#include "mappings.h"
#include "ops.h"


#ifndef public
typedef unsigned short Opcode;

#define OP_BPT 0x4E4F		/* break point fault, (actually = trap #14) */
#define OP_RTD 0x4E74		/* return from procedure, deallocate params */
#define OP_RTS 0x4E75		/* return from procedure */
#define OP_JSR 0x4E80		/* jump to subroutine */
#define OP_JMP 0x4EC0		/* jump */
#define OP_DBT  0x50C8		/* decr & branch true */
#define OP_DBRA 0x51C8		/* decr & branch */
#define OP_DBHI 0x52C8		/* decr & branch on high */
#define OP_DBLS 0x53C8		/* decr & branch on low or same */
#define OP_DBCC 0x54C8		/* decr & branch on carry clear */
#define OP_DBCS 0x55C8		/* decr & branch on carry set */
#define OP_DBNE 0x56C8		/* decr & branch on not equal */
#define OP_DBEQ 0x57C8		/* decr & branch on equal */
#define OP_DBVC 0x58C8		/* decr & branch on overflow clear */
#define OP_DBVS 0x59C8		/* decr & branch on overflow set */
#define OP_DBPL 0x5AC8		/* decr & branch on positive */
#define OP_DBMI 0x5BC8		/* decr & branch on negative */
#define OP_DBGE 0x5CC8		/* decr & branch on greater or equal */
#define OP_DBLT 0x5DC8		/* decr & branch on less */
#define OP_DBGT 0x5EC8		/* decr & branch on greater */
#define OP_DBLE 0x5FC8		/* decr & branch on less or equal */
#define OP_BRA 0x6000		/* branch */
#define OP_BSR 0x6100		/* branch to subroutine */
#define OP_BHI 0x6200		/* branch on high */
#define OP_BLS 0x6300		/* branch on low or same */
#define OP_BCC 0x6400		/* branch on carry clear */
#define OP_BCS 0x6500		/* branch on carry set */
#define OP_BNE 0x6600		/* branch on not equal */
#define OP_BEQ 0x6700		/* branch on equal */
#define OP_BVC 0x6800		/* branch on overflow clear */
#define OP_BVS 0x6900		/* branch on overflow set */
#define OP_BPL 0x6A00		/* branch on positive */
#define OP_BMI 0x6B00		/* branch on negative */
#define OP_BGE 0x6C00		/* branch on greater or equal */
#define OP_BLT 0x6D00		/* branch on less */
#define OP_BGT 0x6E00		/* branch on greater */
#define OP_BLE 0x6F00		/* branch on less or equal */

#endif public

private Address ldot;
private Boolean print_inst;
private Address dest_addr;

private char   *badop = "???(0x%x)";

private int     omove(), obranch(), oimmed(), oprint();
private int	oneop(), soneop(), oreg(), ochk();
private int     olink(), omovem(), oquick(), omoveq();
private int	otrap(), oscc(), opmode(), oshift();
private int     extend(), obit(), oconst(), odbcc();
private int	omovep(), omovec(), omoves(), mvspec();
private int	olea(), odiv();
private int	ocallm(), opack(), ochk2(), bfoneop(), bftwoop();
private int	ocas(), ocas2(), otrapcc(), regpair();
private int	fops(), fbcc(), fdbcc(), fscc(), ftrapcc();
private Word	instfetch();

private struct opdesc {
    Opcode  mask, match;
    int     (*opfun) ();
    char    *farg;
} opdecode[] =
{	/* order is important below */

	/* move instructions */
	0xF000, 0x1000, omove,	"b",
	0xF000, 0x2000, omove,	"l",
	0xF000, 0x3000, omove,	"w",

	/* branches */
	0xF000, 0x6000, obranch, 0,

	/* op class 0  */
	0xFFC0, 0x00C0, ochk2,	"b",
	0xFF00, 0x0000, oimmed,	"or",
	0xFFC0, 0x02C0, ochk2,	"w",
	0xFF00, 0x0200, oimmed,	"and",
	0xFFC0, 0x04C0, ochk2,	"l",
	0xFF00, 0x0400, oimmed,	"sub",
	0xFFF8, 0x06C0, oreg,	"rtm\td%d",
	0xFFF8, 0x06C8, oreg,	"rtm\ta%d",
	0xFFC0, 0x06C0, ocallm, 0,
	0xFF00, 0x0600, oimmed,	"add",
	0xFFFF, 0x0AFC, ocas2,	"b",
	0xFFC0, 0x0AC0, ocas,	"b",
	0xFF00, 0x0A00, oimmed,	"eor",
	0xFFFF, 0x0CFC, ocas2,	"w",
	0xFFC0, 0x0CC0, ocas,	"w",
	0xFF00, 0x0C00, oimmed,	"cmp",
	0xFFFF, 0x0EFC, ocas2,	"l",
	0xFFC0, 0x0EC0, ocas,	"l",
	0xFFC0, 0x0E00, omoves,	"b",
	0xFFC0, 0x0E40, omoves,	"w",
	0xFFC0, 0x0E80, omoves,	"l",
	0xF138, 0x0108, omovep,	"movep",
	0xF100, 0x0100, obit,	0,
	0xF800, 0x0800, obit,	0,

	/* op class 4 */
	0xFFC0, 0x40C0, mvspec,	"sr,",
	0xFF00, 0x4000, soneop,	"negx",
	0xFFC0, 0x42C0, mvspec,	"cc,",
	0xFF00, 0x4200, soneop,	"clr",
	0xFFC0, 0x44C0, mvspec,	",cc",
	0xFF00, 0x4400, soneop,	"neg",
	0xFFC0, 0x46C0, mvspec,	",sr",
	0xFF00, 0x4600, soneop,	"not",
	0xFFF8, 0x4808, olink,	0,
	0xFFC0, 0x4800, oneop,	"nbcd",
	0xFFF8, 0x4840, oreg,	"swap\td%d",
	0xFFF8, 0x4848, oreg,	"bkpt\t#%d",
	0xFFC0, 0x4840, oneop,	"pea",
	0xFFF8, 0x4880, oreg,	"extw\td%d",
	0xFFF8, 0x48C0, oreg,	"extl\td%d",
	0xFFF8, 0x49C0, oreg,	"extbl\td%d",
	0xFB80, 0x4880, omovem,	0,
	0xFFC0, 0x4AC0, oneop,	"tas",
	0xFF00, 0x4A00, soneop,	"tst",
	0xFFC0, 0x4C00, regpair,	"mul",
	0xFFC0, 0x4C40, regpair,	"div",
	0xFFF0, 0x4E40, otrap,	0,
	0xFFF8, 0x4E50, olink,	0,
	0xFFF8, 0x4E58, oreg,	"unlk\ta%d",
	0xFFF8, 0x4E60, oreg,	"move\ta%d,usp",
	0xFFF8, 0x4E68, oreg,	"move\tusp,a%d",
	0xFFFF, 0x4E70, oprint,	"reset",
	0xFFFF, 0x4E71, oprint,	"nop",
	0xFFFF, 0x4E72, oconst,	"stop",
	0xFFFF, 0x4E73, oprint,	"rte",
	0xFFFF, 0x4E74, oconst,	"rtd",
	0xFFFF, 0x4E75, oprint,	"rts",
	0xFFFF, 0x4E76, oprint,	"trapv",
	0xFFFF, 0x4E77, oprint,	"rtr",
	0xFFFE, 0x4E7A, omovec, 0,
	0xFFC0, 0x4E80, oneop,	"jsr",
	0xFFC0, 0x4EC0, oneop,	"jmp",
	0xF1C0, 0x4180, ochk,	"w",
	0xF1C0, 0x4100, ochk,	"l",
	0xF1C0, 0x41C0, olea,	"lea",

	0xF0F8, 0x50C8, odbcc,	0,
	0xF0FF, 0x50FA, otrapcc,	0,
	0xF0FF, 0x50FB, otrapcc,	0,
	0xF0FF, 0x50FC, otrapcc,	0,
	0xF0C0, 0x50C0, oscc,	0,
	0xF100, 0x5000, oquick,	"addq",
	0xF100, 0x5100, oquick,	"subq",

	0xF000, 0x7000, omoveq,	0,

	0xF1C0, 0x80C0, odiv,	"divu",
	0xF1F0, 0x8100, extend,	"sbcd",
	0xF1F0, 0x8140, opack,	"pack",
	0xF1F0, 0x8180, opack,	"unpk",
	0xF1C0, 0x81C0, odiv,	"divs",
	0xF000, 0x8000, opmode,	"or",

	0xF1C0, 0x91C0, opmode,	"sub",
	0xF130, 0x9100, extend,	"subx",
	0xF000, 0x9000, opmode,	"sub",

	0xF1C0, 0xB1C0, opmode,	"cmp",
	0xF138, 0xB108, extend,	"cmpm",
	0xF100, 0xB000, opmode,	"cmp",
	0xF100, 0xB100, opmode,	"eor",

	0xF1C0, 0xC0C0, odiv,	"mulu",
	0xF1C0, 0xC1C0, odiv,	"muls",
	0xF1F8, 0xC188, extend,	"exg",
	0xF1F8, 0xC148, extend,	"exg",
	0xF1F8, 0xC140, extend,	"exg",
	0xF1F0, 0xC100, extend,	"abcd",
	0xF000, 0xC000, opmode,	"and",

	0xF1C0, 0xD1C0, opmode,	"add",
	0xF130, 0xD100, extend,	"addx",
	0xF000, 0xD000, opmode,	"add",

	0xF9C0, 0xE8C0, bfoneop, 0,
	0xF9C0, 0xE9C0, bftwoop, 0,
	0xF100, 0xE000, oshift,	"r",
	0xF100, 0xE100, oshift,	"l",

	0xFFFF, 0xF27A, ftrapcc,	0,
	0xFFFF, 0xF27B, ftrapcc,	0,
	0xFFFF, 0xF27C, ftrapcc,	0,
	0xFFF8, 0xF248, fdbcc,	0,
	0xFFC0, 0xF240, fscc,	0,
	0xFFC0, 0xF280, fbcc,	"w",
	0xFFC0, 0xF2C0, fbcc,	"l",
	0xFFC0, 0xF300, oneop,	"fsave",
	0xFFC0, 0xF340, oneop,	"frestore",
	0xFE00, 0xF200, fops,	0,
	0, 0, 0, 0
};

public Address disasm (print, addr, dest)
Boolean print;
Address addr;
Address *dest;
{
    register struct opdesc *p;
    register Opcode inst;

    ldot = addr;
    print_inst = print;
    dest_addr = -1;

    inst = instfetch(2);
    for (p = opdecode; p->mask; p++)
	if ((inst & p->mask) == p->match)
	    break;
    if (p->mask != 0) {
	(*p->opfun) (inst, p->farg);
    } else
	Printf (badop, inst);

    *dest = dest_addr;
    return ldot;
}

private Word instfetch (size)
int     size;
{
    Word    w;
    unsigned short   hw;

    if (size == 4) {
	iread(&w, ldot, sizeof(w));
	ldot += 4;
    	return (w);
    } else {
	iread(&hw, ldot, sizeof(hw));
	ldot += 2;
    	return (hw);
    }
}

private printea (mode, reg, size)
int     mode, reg;
int     size;
{
    Word    index, disp;
    union kludge {
	double d;
	float f;
	long l[2]
    } kluj;

    switch (mode) {
	case 0: 
	    Printf ("d%d", reg);
	    break;

	case 1: 
	    Printf ("a%d", reg);
	    break;

	case 2: 
	    Printf ("a%d@@", reg);
	    break;

	case 3: 
	    Printf ("a%d@@+", reg);
	    break;

	case 4: 
	    Printf ("a%d@@-", reg);
	    break;

	case 5: 
	    Printf ("a%d@@(%d)", reg, (short)instfetch (2));
	    break;

	case 6: 
	    indexwd (reg);
	    break;

	case 7: 
	    switch (reg) {
		case 0: 
		    pr_symbol (instfetch (2));
		    Printf(":w");
		    break;

		case 1: 
		    pr_symbol (instfetch (4));
		    break;

		case 2: 
		    disp = (short)instfetch (2);
		    dest_addr = ldot + disp + 4;
		    Printf ("pc@@(%d)", disp);
		    break;

		case 3: 
		    indexwd (16);
		    break;

		case 4: 
			if (size <= 2)
				Printf ("#0x%x", instfetch (2));
			else if (size == 4) {
				Printf ("#0x%x", instfetch (4));
			} else if (size == 6) {
				/* floating single precision */
				/* KLUDGE: size is really 4,
				 * but need to distinguish from long case 
				 */
				kluj.l[0] = instfetch(4);
				Printf ("#0f%g", kluj.f);
			} else if (size == 8) {
				/* floating double precision */
				kluj.l[0] = instfetch(4);
				kluj.l[1] = instfetch(4);
				Printf ("#0d%g", kluj.d);
			} else if (size == 12) {
				/* Floating extended and packed decimal */
				/* can't give real value, so do in hex */
				Printf ("#0x%08x", instfetch (4));
				Printf ("%08x", instfetch (4));
				Printf ("%08x", instfetch (4));
			}
			break;

		default: 
		    Printf ("???");
		    break;
	    }
	    break;

	default: 
	    Printf ("???");
    }
}

private printEA (inst)
Opcode inst;
{
    printea ((inst >> 3) & 07, inst & 07, 0);
}

private pr_symbol (v)
Address	 v;
{
    Symbol f;

    f = whatblock (v);
    if (codeloc (f) == v)
	Printf ("%s", symname(f));
    else
	Printf ("0x%x", v);
    dest_addr = v;
}

union index_word {
	unsigned short word;
	struct {
#ifdef vax
		unsigned short disp: 8;
		unsigned short usefull: 1;
		unsigned short scale: 2;
		unsigned short sz: 1;
		unsigned short reg: 3;
		unsigned short isareg: 1;
#else
		unsigned short isareg: 1;
		unsigned short reg: 3;
		unsigned short sz: 1;
		unsigned short scale: 2;
		unsigned short usefull: 1;
		unsigned short disp: 8;
#endif vax
	} brief;
	struct {
#ifdef vax
		unsigned short ixsel: 3;
		unsigned short zfill: 1;
		unsigned short bdsize: 2;
		unsigned short isupp: 1;
		unsigned short bsupp: 1;
		unsigned short usefull: 1;
		unsigned short scale: 2;
		unsigned short sz: 1;
		unsigned short reg: 3;
		unsigned short isareg: 1;
#else
		unsigned short isareg: 1;
		unsigned short reg: 3;
		unsigned short sz: 1;
		unsigned short scale: 2;
		unsigned short usefull: 1;
		unsigned short bsupp: 1;
		unsigned short isupp: 1;
		unsigned short bdsize: 2;
		unsigned short zfill: 1;
		unsigned short ixsel: 3;
#endif vax
	} full;
};

indexwd (reg)
int reg;
{
	int bdisp, odisp;
	union index_word ixwd;

	ixwd.word = instfetch (2);
	if (!ixwd.brief.usefull) {
		bdisp = (char) ixwd.brief.disp;
		if (reg == 16) {
			bdisp += 2;
			Printf ("pc@@");
		} else
			Printf ("a%d@@", reg);
		Printf ("(%d,%c%d:%c*%c)",
			bdisp,
			(ixwd.brief.isareg ? 'a' : 'd'),
			ixwd.brief.reg, (ixwd.brief.sz ? 'l' : 'w'),
			"1248"[ixwd.brief.scale]);
		return;
	}

	/* now comes the really groady part */
	if (reg == 16)
		Printf ("%spc@@(", (ixwd.full.bsupp ? "z" : ""));
	else if (ixwd.full.bsupp)
		Printf ("za@@(");
	else
		Printf ("a%d@@(", reg);

	switch (ixwd.full.bdsize) {
	case 0: 		/* this is really an error, but oh, well... */
	case 1: 
		Printf ("0");
		break;
	case 2: 
		bdisp = (short)instfetch (2);
		Printf ("%d", bdisp);
		break;
	case 3: 
		bdisp = instfetch (4);
		pr_symbol (bdisp);
		break;
	}

	if (ixwd.full.ixsel & 04) {
		Printf (")[%c%d:%c*%c,",
			(ixwd.full.isareg ? 'a' : 'd'),
			ixwd.full.reg, (ixwd.full.sz ? 'l' : 'w'),
			"1248"[ixwd.full.scale]);
	} else {
		if (ixwd.full.isupp)
			Printf (")");
		else
			Printf (",%c%d:%c*%c)",
				(ixwd.full.isareg ? 'a' : 'd'),
				ixwd.full.reg, (ixwd.full.sz ? 'l' : 'w'),
				"1248"[ixwd.full.scale]);
		if (ixwd.full.ixsel != 0)
			Printf ("[");
	}
	if (ixwd.full.ixsel != 0) {
		switch (ixwd.full.ixsel & ~04) {
		case 0: 	/* this is really an error, but oh, well... */
		case 1: 
			Printf ("0]");
			break;
		case 2: 
			odisp = (short)instfetch (2);
			Printf ("%d]", odisp);
			break;
		case 3: 
			odisp = instfetch (4);
			pr_symbol (odisp);
			Printf ("]");
			break;
		}
	}
}

private mapsize (inst)
register Opcode inst;
{
    inst >>= 6;
    inst &= 03;
    return ((inst == 0) ? 1 : (inst == 1) ? 2 : (inst == 2) ? 4 : -1);
}

private char suffix (size)
register int size;
{
    return ((size == 1) ? 'b' : (size == 2) ? 'w' : (size == 4) ? 'l' : '?');
}

private omove (inst, s)
Opcode inst;
char *s;
{
    int     size;

    Printf ("mov%c\t", *s);
    size = ((*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
    printea ((inst >> 3) & 07, inst & 07, size);
    Putchar (',');
    printea ((inst >> 6) & 07, (inst >> 9) & 07, size);
}

private char *br_names[16] = {
    "ra", "sr", "hi", "ls", "cc", "cs", "ne", "eq",
    "vc", "vs", "pl", "mi", "ge", "lt", "gt", "le"
};

private obranch (inst, dummy)
Opcode inst;
char *dummy;
{
    short disp = inst & 0377;
    char  leng;
    Address save_dot;

    save_dot = ldot;
    if (disp == 0) {
	leng = ' ';
	disp = (short)instfetch (2);
    } else if (disp == 255) {
	leng = 'l';
	disp = instfetch (4);
    } else if (disp > 127) {
	leng = 's';
	disp |= ~0377;
    }
    Printf ("b%s%c\t", br_names[((inst >> 8) & 017)], leng);
    pr_symbol (save_dot + disp);
}

private char *cond_names[16] = {
    "t", "f", "hi", "ls", "cc", "cs", "ne", "eq",
    "vc", "vs", "pl", "mi", "ge", "lt", "gt", "le"
};

private odbcc (inst, dummy)
Opcode inst;
char *dummy;
{
    register int    cond = (inst >> 8) & 017;
    Address save_dot;

    save_dot = ldot;
    Printf ("db%s\td%d,",
	(cond == 1 ? "ra" : cond_names[cond]),
		/* use "dbra" instead of "dbf" */
	inst & 07);
    pr_symbol (save_dot + (short)instfetch(2));
}

private oscc (inst, dummy)
Opcode inst;
char *dummy;
{
    register int    cond = (inst >> 8) & 017;

    Printf ("s%s\t", cond_names[cond]);
    printea ((inst >> 3) & 07, inst & 07, 1);
}

private obit (inst, dummy)
Opcode inst;
char *dummy;
{
    static char   *bit_ops[4] = {
	"btst", "bchg", "bclr", "bset"
    };

    Printf ("%s\t", bit_ops[((inst >> 6) & 03)]);
    if (inst & 0x0100)
	Printf ("d%d,", inst >> 9);
    else {
	Printf ("#%d,", instfetch (2));
    }
    printEA (inst);
}

private opmode (inst, opcode)
Opcode inst;
char *opcode;
{
    register int    mode = (inst >> 6) & 07;
    register int    reg = (inst >> 9) & 07;
    int     size;

    size = (mode == 0 || mode == 4) ? 1 :
	(mode == 1 || mode == 3 || mode == 5) ? 2 : 4;
    Printf ("%s%c\t", opcode, suffix (size));
    if (mode >= 4 && mode <= 6) {
	Printf ("d%d,", reg);
	printea ((inst >> 3) & 07, inst & 07, size);
    } else {
	printea ((inst >> 3) & 07, inst & 07, size);
	Printf (",%c%d", (mode <= 2) ? 'd' : 'a', reg);
    }
}

private oshift (inst, ds)
Opcode inst;
char *ds;
{
    int     rx, ry;
    char   *opcode;
    static  char   *shift_ops[4] = {
	"as", "ls", "rox", "ro"
    };

    if ((inst & 0xC0) == 0xC0) {
	opcode = shift_ops[(int) ((inst >> 9) & 03)];
	Printf ("%s%s\t", opcode, ds);
	printEA (inst);
    } else {
	opcode = shift_ops[(int) ((inst >> 3) & 03)];
	Printf ("%s%s%c\t", opcode, ds, suffix (mapsize (inst)));
	rx = (int) ((inst >> 9) & 07);
	ry = (int) (inst & 07);
	if ((inst >> 5) & 01)
	    Printf ("d%d,d%d", rx, ry);
	else
	    Printf ("#%d,d%d", (rx == 0 ? 8 : rx), ry);
    }
}

private oimmed (inst, opcode)
Opcode inst;
char  *opcode;
{
    int    size = mapsize (inst);
    int    mode = (inst >> 3) & 07;
    int    reg = inst & 07;
    long   const;

    if (size > 0) {
	const = instfetch (size == 4 ? 4 : 2);
	Printf ("%s%c\t#0x%x,", opcode, suffix (size), const);
	if (mode == 7 && reg == 4)
		/* cc or sr is destination */
	    Printf ("%s", (size == 1 ? "cc" : "sr"));
	else
	    printea (mode, reg, size);
    } else
	Printf (badop, inst);
}

private oreg (inst, opcode)
Opcode inst;
char  *opcode;
{
    Printf (opcode, (inst & 07));
}

private extend (inst, opcode)
Opcode inst;
char   *opcode;
{
    int     size = mapsize (inst);
    int     ry = (inst & 07), rx = ((inst >> 9) & 07);
    char    c;

    c = ((inst & 0x1000) ? suffix (size) : ' ');
    Printf ("%s%c\t", opcode, c);
    if (*opcode == 'e') {
	if (inst & 0x0080)
	    Printf ("d%d,a%d", rx, ry);
	else if (inst & 0x0008)
	    Printf ("a%d,a%d", rx, ry);
	else
	    Printf ("d%d,d%d", rx, ry);
    } else if ((inst & 0xF000) == 0xB000)
	Printf ("a%d@@+,a%d@@+", ry, rx);
    else if (inst & 0x8)
	Printf ("a%d@@-,a%d@@-", ry, rx);
    else
	Printf ("d%d,d%d", ry, rx);
}

private olink (inst, dummy)
Opcode inst;
char *dummy;
{
    int disp;

    if (inst & 0x8)
	disp = instfetch (4);
    else
	disp = (short)instfetch (2);
    Printf ("link\ta%d,#%d", inst & 07, disp);
}

private otrap (inst, dummy)
Opcode inst;
char *dummy;
{
    Printf ("trap\t#%d", inst & 017);
}

private mvspec (inst, opcode)
Opcode inst;
char *opcode;
{
    Printf ("movw\t");
    if (*opcode == ',') {
	printEA (inst);
	Printf ("%s", opcode);
    } else {
	Printf ("%s", opcode);
	printEA (inst);
    }
}

private oneop (inst, opcode)
Opcode inst;
char  *opcode;
{
    Printf ("%s\t", opcode);
    printEA (inst);
}

private pregmask (mask)
register int    mask;
{
    register int    i;
    register int    flag = 0;

    Printf ("#<");
    for (i = 0; i < 16; i++) {
	if (mask & 1) {
	    if (flag)
		Putchar (',');
	    else
		flag++;
	    Printf ("%c%d", (i < 8) ? 'd' : 'a', i & 07);
	}
	mask >>= 1;
    }
    Printf (">");
}

private omovem (inst, dummy)
Opcode inst;
char *dummy;
{
    register int    i, list = 0, mask = 0100000;
    register int    regmask = instfetch (2);

    if ((inst & 070) == 040) {	/* predecrement */
	for (i = 15; i > 0; i -= 2) {
	    list |= ((mask & regmask) >> i);
	    mask >>= 1;
	}
	for (i = 1; i < 16; i += 2) {
	    list |= ((mask & regmask) << i);
	    mask >>= 1;
	}
	regmask = list;
    }
    Printf ("movem%c\t", (inst & 0100) ? 'l' : 'w');
    if (inst & 02000) {
	printEA (inst);
	Putchar (',');
	pregmask (regmask);
    }
    else {
	pregmask (regmask);
	Putchar (',');
	printEA (inst);
    }
}

private ochk (inst, opcode)
Opcode inst;
char  *opcode;
{
    Printf ("%s\t", opcode);
    printEA (inst);
    Printf (",%c%d", (*opcode == 'l') ? 'a' : 'd', (inst >> 9) & 07);
}

private odiv (inst, opcode)
Opcode inst;
char *opcode;
{
	Printf ("%s\t", opcode);
	printEA (inst);
	Printf (",d%d", (inst >> 9) & 07);
}

private olea (inst, dummy)
Opcode inst;
char *dummy;
{
	Printf ("lea\t");
	printEA (inst);
	Printf (",a%d", (inst >> 9) & 07);
}

private soneop (inst, opcode)
Opcode inst;
char  *opcode;
{
    register int    size = mapsize (inst);

    if (size > 0) {
	Printf ("%s%c\t", opcode, suffix (size));
	printEA (inst);
    }
    else
	Printf (badop, inst);
}

private oquick (inst, opcode)
Opcode inst;
char  *opcode;
{
    register int    size = mapsize (inst);
    register int    data = (int) ((inst >> 9) & 07);

    if (data == 0)
	data = 8;
    if (size > 0) {
	Printf ("%s%c\t#%d,", opcode, suffix (size), data);
	printEA (inst);
    }
    else
	Printf (badop, inst);
}

private omoveq (inst, dummy)
Opcode inst;
char *dummy;
{
    register int    data = (unsigned char) (inst & 0377);

    Printf ("moveq\t#0x%02x,d%d", data, (inst >> 9) & 07);
}

private oconst (inst, opcode)
Opcode inst;
char   *opcode;
{
    Printf ("%s\t#0x%x", opcode, (unsigned int) instfetch (2));
}

private oprint (inst, opcode)
Opcode inst;
char  *opcode;
{
    Printf ("%s", opcode);
}

private omovep (inst, opcode)
Opcode inst;
char  *opcode;
{
    int     areg = inst & 07;
    int     dreg = (inst >> 9) & 07;
    int     disp = instfetch (2);

    Printf ("%s%c\t", opcode, (inst & 0100 ? 'l' : 'w'));
    if (inst & 0200)
	Printf ("d%d,a%d@@(%d)", dreg, areg, disp);
    else
	Printf ("a%d@@(%d),d%d", areg, disp, dreg);
}

private char *getctlreg (op2)
int     op2;
{
	switch (op2 & 0x0FFF) {
	case 0x0000: 
		return ("sfc");
	case 0x0001: 
		return ("dfc");
	case 0x0002: 
		return ("cacr");
	case 0x0800: 
		return ("usp");
	case 0x0801: 
		return ("vbr");
	case 0x0802: 
		return ("caar");
	case 0x0803: 
		return ("msp");
	case 0x0804: 
		return ("isp");
	default: 
		return ("???");
	}
}

private omovec (inst, dummy)
Opcode	inst;
char	*dummy;
{
    char    regtype, *ctlreg;
    int     regnumb;
    int     info = instfetch (2);

    regtype = (info & 0x8000) ? 'a' : 'd';
    regnumb = (info >> 12) & 07;
    ctlreg = getctlreg (info);
    Printf ("movec\t");
    if (inst & 01)
	Printf ("%c%d,%s", regtype, regnumb, ctlreg);
    else
	Printf ("%s,%c%d", ctlreg, regtype, regnumb);
}

private omoves (inst, s)
Opcode	inst;
char	*s;
{
    char    regtype;
    int     regnumb, size;
    int     info = instfetch (2);

    regtype = (info & 0x8000) ? 'a' : 'd';
    regnumb = (info >> 12) & 07;
    size = ((*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
    Printf ("moves%c\t", *s);
    if (info & 0x8000) {
	Printf ("%c%d,", regtype, regnumb);
	printea ((inst >> 3) & 07, inst & 07, size);
    }
    else {
	printea ((inst >> 3) & 07, inst & 07, size);
	Printf (",%c%d", regtype, regnumb);
    }
}

/* 68020 exclusive routines */

private bfoneop (inst, dummy)
Opcode inst;
char *dummy;
{
	unsigned short extwd = instfetch (2);
	static char *bfop_names[4] = {
		"bftst", "bfchg", "bfclr", "bfset"
	};

	Printf ("bf%s\t", bfop_names[((inst >> 9) & 03)]);

	printEA (inst);
	if (extwd & 0x0800) {
		Printf ("{d%d:", (extwd >> 6) & 07);
	} else {
		Printf ("{%d:", (extwd >> 6) & 0x1f);
	}
	if (extwd & 0x0020) {
		Printf ("d%d}", extwd & 07);
	} else {
		Printf ("%d}", extwd & 0x1f);
	}
}

private bftwoop (inst, dummy)
Opcode inst;
char *dummy;
{
	unsigned short extwd = instfetch (2);
	short bfins = 0;
	char *opcode;

	switch ((inst >> 9) & 03) {
	case 0: 
		opcode = "extu";
		break;
	case 1: 
		opcode = "exts";
		break;
	case 2: 
		opcode = "ffo";
		break;
	case 3: 
		opcode = "ins";
		bfins++;
		break;
	}
	Printf ("bf%s\t", opcode);

	if (bfins) {
		Printf ("d%d,", (extwd >> 12) & 07);
	}

	printEA (inst);
	if (extwd & 0x0800) {
		Printf ("{d%d:", (extwd >> 6) & 07);
	} else {
		Printf ("{%d:", (extwd >> 6) & 0x1f);
	}
	if (extwd & 0x0020) {
		Printf ("d%d}", extwd & 07);
	} else {
		Printf ("%d}", extwd & 0x1f);
	}

	if (!bfins) {
		Printf (",d%d", (extwd >> 12) & 07);
	}
}

private ocallm (inst, dummy)
Opcode inst;
char *dummy;
{
	Printf ("callm\t#%d,", instfetch (2));
	printEA (inst);
}

private ocas (inst, s)
Opcode inst;
char *s;
{
	unsigned short extwd = instfetch (2);

	Printf ("cas%s\td%d,d%d,", s, extwd & 07, (extwd >> 6) & 07);
	printEA (inst);
}

private ocas2 (inst, s)
Opcode inst;
char *s;
{
	unsigned short extwd1, extwd2;
	int r1, r2;

	extwd1 = instfetch (2);
	extwd2 = instfetch (2);

	r1 = (extwd1 & 0x8000) ? 'a' : 'd';
	r2 = (extwd2 & 0x8000) ? 'a' : 'd';
	Printf ("cas2%s\td%d:d%d,d%d:d%d,%c%d@@:%c%d@@",
		s, extwd1 & 07, extwd2 & 07,
		(extwd1 >> 6) & 07, (extwd2 >> 6) & 07,
		r1, (extwd1 >> 12) & 07, r2, (extwd2 >> 12) & 07);
}

private ochk2 (inst, s)
Opcode inst;
char *s;
{
	unsigned short extwd = instfetch (2);

	Printf ("%s%s\t", (extwd & 0x0800) ? "chk2" : "cmp2", s);
	printea ((inst >> 3) & 07, inst & 07,
		(*s == 'b') ? 1 : (*s == 'w') ? 2 : 4);
	Printf (",%c%d", (extwd & 0x8000) ? 'a' : 'd', (extwd >> 12) & 07);
}

private opack (inst, opcode)
Opcode inst;
char *opcode;
{
	int rx = (inst & 07), ry = ((inst >> 9) & 07);

	Printf ("%s\t", opcode);
	if (inst & 0x8)
		Printf ("a%d@@-,a%d@@-", rx, ry);
	else
		Printf ("d%d,d%d", rx, ry);
	Printf (",#0x%04x", instfetch (2));
}

private otrapcc (inst, dummy)
Opcode inst;
char *dummy;
{
	register int cond = (inst >> 8) & 017;
	char s;
	int val;

	switch (inst & 07) {
	case 2: 
		val = instfetch (2);
		s = 'w';
		break;
	case 3: 
		val = instfetch (4);
		s = 'l';
		break;
	case 4: 
		s = ' ';
		break;
	}
	Printf ("trap%s%c", cond_names[cond], s);
	if (s == 'w')
		Printf ("\t#0x%04x", val);
	else if (s == 'l')
		Printf ("\t#0x%08x", val);
}

private regpair (inst, opcode)
Opcode inst;
char *opcode;
{
	register short extwd = instfetch (2);
	int reg1, reg2;
	char us, *sz, havepr;

	us = (extwd & 0x0800) ? 's' : 'u';
	reg1 = (extwd >> 12) & 07;
	reg2 = extwd & 07;
	sz = "l";
	if (extwd & 0x0400)
		havepr = 1;
	else if (reg1 != reg2) {
		/* div?ll case */
		sz = "ll";
		havepr = 1;
	} else
		havepr = 0;

	Printf ("%s%c%s\t", opcode, us, sz);
	printea ((inst >> 3) & 07, inst & 07, 4);
	if (havepr) {
		Printf (",d%d:d%d", reg2, reg1);
	} else
		Printf (",d%d", reg1);
}


/* 68881 floating point processor routines */

private char fformat[] = "lsxpwdbp";
/* for formats:	l  s  x   p   w  d  b  p?	*/
private int fsize[] = { 4, 6, 12, 12, 2, 8, 1, 0};
/* fsize[s] == 6 is a kludge to distinguish 'l' and 's' immediates */

private char *fopname[] = {
	/* zero entries are redundant encodings and will be flagged */
	"mov", "int", "sinh", "intrz",
	"sqrt",	0, "lognp1", 0,
	"etoxm1", "tanh", "atan", 0, 
	"asin", "atanh", "sin", "tan",
	"etox", "twotox", "tentox", 0,
	"logn", "log10", "log2", 0,
	"abs", "cosh", "neg", 0,
	"acos", "cos", "getexp", "getman",
	"div", "mod", "add", "mul",
	"sgldiv", "rem", "scale", "sglmul",
	"sub", 0, 0, 0,
	0, 0, 0, 0,
	"", "", "", "", 			/* sincos is special */
	"", "", "", "", 			/* sincos is special */
	"cmp", 0, "tst", 0,
	0, 0, 0, 0,
	/* 0x40 to 0x7f are reserved and checked elsewhere */
};

private fops (inst, dummy)
Opcode inst;
char *dummy;
{
	int cmd = instfetch (2);

	switch ((cmd >> 13) & 07) {
	case 0:	fop_reg_reg (inst, cmd); break;
	case 1: /* reserved encoding */
		Printf ("freserved???\n");
		break;
	case 2:	fop_ea_reg (inst, cmd); break;
	case 3: fmov_ea (inst, cmd); break;
	case 4: fmovec (inst, cmd, 0); break;
	case 5: fmovec (inst, cmd, 1); break;
	case 6: fmovem (inst, cmd, 0); break;
	case 7: fmovem (inst, cmd, 1); break;
	}
}

private fop_reg_reg (inst, cmd)
Opcode inst, cmd;
{
	int op = cmd & 0x7f;

	if (op >= 0x40 || fopname[op] == 0) {
		Printf ("freserved???");
		return;
	}
	if (fopname[op][0] == '\0') {
		/* handle fsincosx op */
		Printf ("fsincosx\tf%d,f%d:f%d", (cmd >> 10) & 07,
			cmd & 07, (cmd >> 7) & 07);
		return;
	}
	if (op == 0x3A) {
		/* handle ftstx op */
		Printf ("ftstx\tf%d", (cmd >> 10) & 07);
		return;
	}
	/* finally all others */
	Printf ("f%sx\tf%d,f%d",
		fopname[op], (cmd >> 10) & 07, (cmd >> 7) & 07);
}

private fop_ea_reg (inst, cmd)
Opcode inst, cmd;
{
	int op = cmd & 0x7f;
	int fmt = (cmd >> 10) & 07;
	short sincos = 0;
	char szch = fformat[fmt];

	if (fmt == 7 && (inst & 0xCF) == 0) {
		/* handle fmovecr op */
		Printf ("fmovecr\t#%d,f%d", op, (cmd >> 7) & 07);
		return;
	}

	if (op >= 0x40 || fopname[op] == 0) {
		Printf ("freserved???");
		return;
	}

	if (fopname[op][0] == '\0') {
		/* handle fsincos op */
		Printf ("fsincos%c\t", szch);
		sincos++;
	} else
		Printf ("f%s%c\t", fopname[op], szch);

	printea ((inst >> 3) & 07, inst & 07, fsize[fmt]);
	if (op != 0x3A) {
		/* all except ftst op have destination */
		Printf (",f%d", (cmd >> 7) & 07);
		if (sincos) {
			/* fsincos has two destination */
			Printf (":f%d", cmd & 07);
		}
	}
}

private fmov_ea (inst, cmd)
Opcode inst, cmd;
{
	int kfactor = cmd & 0x7f;
	int fmt = (cmd >> 10) & 07;

	Printf ("fmov%c\tf%d,", fformat[fmt], (cmd >> 7) & 07);
	printea ((inst >> 3) & 07, inst & 07, fsize[fmt]);

	if (fmt == 3) {
		/* need to sign extend the number */
		if (kfactor > 63)
			kfactor |= ~0x3f;
		Printf ("{%d}", kfactor);
	} else if (fmt == 7) {
		Printf ("{d%d}", (kfactor >> 4) & 07);
	}
}

private fmovec (inst, cmd, dir)
Opcode inst, cmd;
int dir;
{
	register int reglist = (cmd >> 10) & 07;
	char *creg;

	switch (reglist) {
	case 0:
		Printf ("freserved???"); return;
	case 1:
		creg = "fpiar"; break;
	case 2:
		creg = "fpsr"; break;
	case 4:
		creg = "fpcr"; break;
	default:
		Printf ("fmoveml\t");
		if (dir == 0) {
			printEA (inst);
			Putchar(',');
			fcregmask (reglist);
		} else {
			fcregmask (reglist);
			Putchar(',');
			printEA (inst);
		}
		return;
	}

	Printf ("fmovel\t");
	if (dir == 0) {
		printea ((inst >> 3) & 07, inst & 07, 4);
		Printf (",%s", creg);
	} else {
		Printf ("%s,", creg);
		printEA (inst);
	}
}

private fcregmask (mask)
register int mask;
{
	short flag = 0;

	Printf ("#<");
	if (mask & 04) {
		Printf ("fpcr");
		flag++;
	}
	if (mask & 02) {
		if (flag)
			Putchar(',');
		Printf ("fpsr");
		flag++;
	}
	if (mask & 01) {
		if (flag)
			Putchar(',');
		Printf ("fpiar");
	}
	Printf (">");
}

private fmovem (inst, cmd, dir)
Opcode inst, cmd;
int dir;
{
	register int reglist = cmd & 0xff;

	if (cmd & 0x0800) {
		/* register list is in a data register */
		reglist = (reglist >> 4) & 07;
	} else if ((inst & 070) == 040) {/* predecrement */
		register int i, list, mask;

		list = 0; mask = 0x80;
		for (i = 7; i > 0; i -= 2) {
			list |= ((mask & reglist) >> i);
			mask >>= 1;
		}
		for (i = 1; i < 8; i += 2) {
			list |= ((mask & reglist) << i);
			mask >>= 1;
		}
		reglist = list;
	}

	Printf ("fmovemx\t");
	if (dir == 0) {
		printEA (inst);
		Putchar(',');
		if (cmd & 0x0800)
			Printf ("d%d", reglist);
		else
			fregmask (reglist);
	} else {
		if (cmd & 0x0800)
			Printf ("d%d", reglist);
		else
			fregmask (reglist);
		Putchar(',');
		printEA (inst);
	}
}

private fregmask (mask)
register int mask;
{
	register int i;
	register int flag = 0;

	Printf ("#<");
	for (i = 0; i < 8; i++) {
		if (mask & 0x80) {
			if (flag)
				Putchar(',');
			else
				flag++;
			Printf ("f%d", i);
		}
		mask <<= 1;
	}
	Printf (">");
}

private char *fcond_names[32] = {
	"f",    "eq",  "ogt", "oge", "olt", "ole", "ogl",  "or",
	"un",   "ueq", "ugt", "uge", "ult", "ule", "neq",  "t",
	"sf",   "seq", "gt",  "ge",  "lt",  "le",  "gl",   "gle",
	"ngle", "ngl", "nle", "nlt", "nge", "ngt", "sneq", "st",
};

private fbcc (inst, leng)
Opcode inst;
char *leng;
{
	int disp;
	int cond = inst & 0x3F;
	Address save_dot;

	save_dot = ldot;
	if (*leng == 'w') {
		disp = (short)instfetch (2) - 2;
	} else if (*leng == 'l') {
		disp = instfetch (4) - 2;
	}

	if (cond == 0 && *leng == 'w' && disp == -2)
		Printf ("fnop");
	else {
		Printf ("fb%s%c\t0x%x",
			fcond_names[cond], *leng);
		pr_symbol (save_dot + disp);
	}
}

private fdbcc (inst, dummy)
Opcode inst;
char *dummy;
{
	int disp;
	register int cond = instfetch (2);
	Address save_dot;

	save_dot = ldot;
	disp = (short)instfetch (2);
 	Printf ("fdb%s\td%d,",
		fcond_names[cond], inst & 07);
	pr_symbol (save_dot + disp);
}

private fscc (inst, dummy)
Opcode inst;
char *dummy;
{
	register int cond = instfetch (2);

	Printf ("fs%s\t", fcond_names[cond]);
	printea ((inst >> 3) & 07, inst & 07, 1);
}

private ftrapcc (inst, dummy)
Opcode inst;
char *dummy;
{
	register int cond = instfetch (2);
	char s;
	int val;

	switch (inst & 07) {
	case 2: 
		val = instfetch (2);
		s = 'w';
		break;
	case 3: 
		val = instfetch (4);
		s = 'l';
		break;
	case 4: 
		s = ' ';
		break;
	}
	Printf ("ftrap%s%c", fcond_names[cond], s);
	if (s == 'w')
		Printf ("\t#0x%04x", val);
	else if (s == 'l')
		Printf ("\t#0x%08x", val);
}

private Putchar(c)
char c;
{
    if (print_inst)
	putchar(c);
}

private Printf(fmt, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
char *fmt;
{
    if (print_inst)
	printf(fmt, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
}
