/* $Header: /usr/alex/drivers/sb16/RCS/sb16dsp.c,v 1.2 94/05/21 15:59:46 alex Exp $
 *
 * Sound Blaster 16 DSP Functions
 *
 * Copyright (c) 1994 Alex Nash, All Rights Reserved.
 *
 * This software is provided "as is" and without warranty of any kind.
 * The author accepts no responsibility for any damage caused by this
 * software.
 *
 * $Log:	sb16dsp.c,v $
 * Revision 1.2  94/05/21  15:59:46  alex
 * Sound position and level ioctls().  Mixer flags added to SoundInfo.
 * Stronger configuration checking on startup.
 * 
 * Revision 1.1  94/04/07  17:19:21  alex
 * Initial revision
 *
 */

#include	<sys/coherent.h>

#include	<sys/cmn_err.h>
#include 	<sys/errno.h>
#include 	<sys/file.h>
#include 	<sys/sb16.h>
#include 	<sys/sched.h>
#include 	<sys/sound.h>
#include 	<kernel/param.h>
#include 	<kernel/pri.h>
#include 	<kernel/timeout.h>
#include 	<fcntl.h>
#include 	<limits.h>

/*--------------------------------------------------------------------------*/
/*	array dimension macro													*/
/*--------------------------------------------------------------------------*/
#define 	dim(x)		(sizeof(x) / sizeof(x[0]))

/*--------------------------------------------------------------------------*/
/*	sb16dspMode flags														*/
/*--------------------------------------------------------------------------*/
#define		SB16DSP_RECEND			1
#define 	SB16DSP_STOPPED			2
#define 	SB16DSP_INITIALIZED		4
#define 	SB16DSP_PLAYING			8
#define 	SB16DSP_RECORDING		16

/*--------------------------------------------------------------------------*/
/*	DMA buffer size															*/
/*--------------------------------------------------------------------------*/
#define		SB16_DMA_BUFSIZE		(64 * 1024)

/*--------------------------------------------------------------------------*/
/*	maximum number of retries when reading from/writing to DSP				*/
/*--------------------------------------------------------------------------*/
#define 	SB16DSP_RETRIES			100

/*--------------------------------------------------------------------------*/
/*	volatile variables														*/
/*--------------------------------------------------------------------------*/
static __VOLATILE__ int			sb16dspBlockIndex;
static __VOLATILE__ int			sb16dspMode;
static __VOLATILE__ int			sb16dspRecordError;
static __VOLATILE__ unsigned	sb16dspLength;
static __VOLATILE__ unsigned	sb16dspBytes;
static __VOLATILE__ int			sb16dspBufferLock[SB16_DMA_BUFSIZE / 512];

/*--------------------------------------------------------------------------*/
/*	static variables														*/
/*--------------------------------------------------------------------------*/
static int						sb16dspBufferIndex;
static int						sb16dspSleeping;
static int						sb16dspOpen;
static int						sb16dspStereo;
static int						sb16dspRecord;
static int						sb16dspBits;
static int						sb16dspBlocks;
static int						sb16dspBlockSize;
static char						sb16dspDeviceName[]	= "Sound Blaster 16";

static int						sb16dspDma8Page;
static int						sb16dspDma8Base;
static int						sb16dspDma8Count;
static int						sb16dspDma16Page;
static int						sb16dspDma16Base;
static int						sb16dspDma16Count;
static char						*sb16dspDMABuffer;
static struct snd_dma_parms		sb16dspDmaParms;

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_sleep (char *reason)
|
|	reason		Reason the process went to sleep
|
|	Puts the current process to sleep provided that no other processes are
|	currently asleep in this driver.  The process will wakeup after the
|	next interrupt.
|
|	splhi() and splx() must be called outside this routine to avoid race
|	conditions.
|
|	Returns 0 on success.  If an error occurs, -1 is returned and the
|	user process errno is set.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_sleep (char * reason)
#else
static int
sb16dsp_sleep (reason)
char *reason;
#endif
{
	if (sb16dspSleeping)
	{
		set_user_error(EBUSY);	/* other process is already asleep */
		return(-1);
	}

	sb16dspSleeping	= 1;
	x_sleep(&sb16dspSleeping, prihi, slpriSigCatch, reason);
	sb16dspSleeping	= 0;

	if (nondsig())
	{
		set_user_error(EINTR);	/* interrupted by signal */
		return(-1);
	}

	return 0;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_dsp_write (uchar_t *data)
|
|	data		Byte to be written to the DSP
|
|	Writes data to the DSP chip.
|
|	Returns 0 on success, -1 if the DSP is unable to accept the data.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_dsp_write (uchar_t data)
#else
static int
sb16dsp_dsp_write (data)
uchar_t	data;
#endif
{
	int	retries	= SB16DSP_RETRIES;
	int	s		= sphi();

	while ((inb(SB16_DSP_WRBUF_STATUS) & 0x80) && retries)
		--retries;

	outb(SB16_DSP_WRITE, data);	/* stuff data through regardless of error */

	spl(s);

	if (retries == 0)
		return(-1);				/* DSP would not accept data */

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_dsp_read (uchar_t *data, unsigned nBytes)
|
|	data		Buffer to receive the incoming data
|	nBytes		Number of bytes to be read
|
|	Reads data from the DSP chip.
|	Returns 0 on success, -1 if the DSP has no data available.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_dsp_read (uchar_t *data, unsigned nBytes)
#else
static int
sb16dsp_dsp_read (data, nBytes)
uchar_t		*data;
unsigned	nBytes;
#endif
{
	int	retries	= SB16DSP_RETRIES;
	int	s		= sphi();

	while (nBytes--)
	{
		while (!(inb(SB16_DSP_RDBUF_STATUS) & 0x80) && retries)
			retries--;

		*data++	= inb(SB16_DSP_READ);
	}

	spl(s);

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_poll_reset ()
|
|	Called by busyWait2 to check the status of the DSP reset.
|
|	Returns 0 if the reset is not complete, 1 if the reset is complete.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_poll_reset (void)
#else
static int
sb16dsp_poll_reset ()
#endif
{
	if (inb(SB16_DSP_RDBUF_STATUS) & 0x80)
	{
		/* data is available */
		if (inb(SB16_DSP_READ) == 0xAA)
			return(1);
	}

	return(0);	/* reset is not complete */
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_version ()
|
|	Queries the DSP for the version number.
|
|	The major version number is returned in the high byte, the minor in
|	the low byte.  Note that the minor version number should be dislpayed
|	as %02d and not %d.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_version (void)
#else
static int
sb16dsp_version ()
#endif
{
	uchar_t	buffer[2];

	if (sb16dsp_dsp_write(SB16_DSPCMD_VERSION) != 0)
		return(0);

	if (sb16dsp_dsp_read(buffer, 2) != 0)
		return(0);

	return((buffer[0] << 8) | buffer[1]);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_init_reset ()
|
|	Resets the DSP during the driver's load routine.  Use sb16dsp_reset
|	when a user process is active.
|
|	Returns 0 on success, or -1 if an incompatible DSP version or the DSP
|	is not responding.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_init_reset (void)
#else
static int
sb16dsp_init_reset ()
#endif
{
	int	ver;

	outb(SB16_DSP_RESET, 1);
	busyWait2(NULL, USEC_TO_COUNTS(4));
	outb(SB16_DSP_RESET, 0);

	if (busyWait(sb16dsp_poll_reset, HZ / 1000 + 2) == 0)
	{
		cmn_err (CE_WARN, "DSP not responding");
		return(-1);
	}

	ver	= sb16dsp_version();

	if (ver < 0x0400)
	{
		cmn_err (CE_WARN, "Incompatible DSP Version");
		return(-1);
	}
	else
	{
		int	minorVer = ver & 0xFF;

		cmn_err (CE_CONT, "SB16 DSP V%d.", ver >> 8);

		if (minorVer < 0x10)
			cmn_err (CE_CONT, "0%d", minorVer);
		else
			cmn_err (CE_CONT, "%d", minorVer);
	}

	return(0);
}
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_verify_config ()
|
|	Queries the board for DMA/IRQ settings and compares them to the
|	variables sb16Dma8/sb16Dma16/sb16Irq.
|
|	The DMA/IRQ bit patterns are shown below:
|
|		Bit	|  7   6   5   4   3   2   1   0
|		----|---------------------------------
|		DMA |  7   6   5   -   3   -   1   0
|		IRQ |  -   -   -   -  10   7   5   2
|
|	Returns 0 if the driver is configured correctly, or -1 if the kernel
|	was built with invalid values.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_verify_config ()
#else
static int
sb16dsp_verify_config ()
#endif
{
	int	v;
	int v_val;

	outb(SB16_MIXER_ADDRESS, SB16_MXRREG_IRQ_CFG);
	v	= inb(SB16_MIXER_DATA);

	/* Map register bit to designated Irq channel. */
	switch (v&0xf) {
	case 1:
		v_val = 2;
		break;
	case 2:
		v_val = 5;
		break;
	case 4:
		v_val = 7;
		break;
	case 8:
		v_val = 10;
		break;
	default:
		v_val = 0;
	}

	if (v_val == 0 || v_val != sb16Irq) {
		cmn_err (CE_WARN, "SB16 : invalid IRQ v_val=%d, sb16Irq=%d",
		  v_val, sb16Irq);
		return -1;
	}

	outb(SB16_MIXER_ADDRESS, SB16_MXRREG_DMA_CFG);
	v	= inb(SB16_MIXER_DATA);

	/* Map register bit to designated 8-bit DMA channel. */
	switch (v&0xb) {
	case 1:
		v_val = 0;
		break;
	case 2:
		v_val = 1;
		break;
	case 8:
		v_val = 3;
		break;
	default:
		v_val = -1;
	}

	if (v_val == -1 || v_val != sb16Dma8) {
		cmn_err (CE_WARN, "SB16 : invalid 8-bit DMA channel, v=%d, sb16Dma8=%d",
		  v_val, sb16Dma8);
		return -1;
	}

	if (sb16Dma16 == 0) {
		cmn_err (CE_WARN,
		  "SB16 : 16-bit transfers over 8-bit DMA channel not supported");
		return -1;
	}

	/* Map register bit to designated 16-bit DMA channel. */
	switch (v&0xe0) {
	case 0x20:
		v_val = 5;
		break;
	case 0x40:
		v_val = 6;
		break;
	case 0x80:
		v_val = 7;
		break;
	default:
		v_val = -1;
	}

	if (v_val == -1 || v_val != sb16Dma16) {
		cmn_err (CE_WARN,
		  "SB16 : Invalid 16-bit DMA channel, v=%d, sb16Dma16=%d",
		  v_val, sb16Dma16);
		return -1;
	}

	return 0;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_init ()
|
|	Called by the load routine to initialize the DSP chip and internal
|	variables.
|
|	Returns 0 on successful initialization, -1 on error.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_init (void)
#else
static int
sb16dsp_init ()
#endif
{
	int	pad;

	if (sb16Port != 0x220 && sb16Port != 0x240 &&
	  sb16Port != 0x260 && sb16Port != 0x280) {
		cmn_err (CE_WARN, "SB16 : Invalid I/O base address=%x", sb16Port);
		return -1;
	}

	if (sb16dsp_init_reset() != 0)
		return(-1);

	if (sb16dsp_verify_config() != 0)
		return(-1);

#if 0
	if (sb16dsp_init_dma_ports() != 0)
		return(-1);

	sb16dspDMABuffer = (char *)getDmaMem(SB16_DMA_BUFSIZE, 128 * 1024);

	if (sb16dspDMABuffer == NULL) {
		cmn_err (CE_WARN, "SB16 : Unable to allocate DMA buffer (PHYS_MEM)");
		return -1;
	}

	/* workaround 4.2.05 getDmaMem bug */
	pad = ((int) vtop((caddr_t)sb16dspDMABuffer)) % (128 * 1024);
	if (pad != 0)
		sb16dspDMABuffer += (128 * 1024) - pad;
#endif

	sb16dspSleeping	= 0;
	sb16dspOpen		= 0;

	cmn_err (CE_CONT, "  Port %x DMA%d/%d IRQ%d\n",
				sb16Port, sb16Dma8, sb16Dma16, sb16Irq);

	return 0;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_reset_tfn ()
|
|	Wakes up the sleeping user process that awaits the DSP reset.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_reset_tfn (void)
#else
static void
sb16dsp_reset_tfn ()
#endif
{
	if (sb16dspSleeping)
		wakeup(&sb16dspSleeping);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_reset ()
|
|	Resets the DSP while a user context is active.
|
|	Returns 0 if the reset is successful.  On error, -1 is returned and
|	the user's errno is set to EIO.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_reset ()
#else
static int
sb16dsp_reset ()
#endif
{
	static TIM 	tim;
	int			s;
	int			retCode;

	outb(SB16_DSP_RESET, 1);
	busyWait2(NULL, USEC_TO_COUNTS(4));
	outb(SB16_DSP_RESET, 0);

	s	= sphi();
	timeout(&tim, (HZ / 1000) + 1, sb16dsp_reset_tfn, 0);
	retCode	= sb16dsp_sleep("sb16reset");
	timeout(&tim, 0, NULL, 0);
	spl(s);

	if (retCode != 0)
		return(-1);

	if (sb16dsp_poll_reset() != 1)
	{
		set_user_error(EIO);
		return(-1);
	}

	return(0);
}

/*----------------------------------------------------------------------------
|	void
|	sb16dsp_init_transfer (SB16Init *sb16Init)
|
|	sb16Init	Pointer to an SB16Init structure in kernel space
|
|	Programs the DMA controller, page registers, and sampling rate.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_init_transfer (SoundInit * pSoundInit)
#else
static void
sb16dsp_init_transfer (pSoundInit)
SoundInit	*pSoundInit;
#endif
{
	paddr_t		dmaPhys	= vtop(sb16dspDMABuffer);
	unsigned	page	= (unsigned)dmaPhys >> 16;
	int			s;
	int			modePort;
	int			mask;

	s = sphi();

	if (sb16dspBits == 16)
	{
		outb(DMA16_WR_SINGLE, 0x04 | (sb16Dma16 - 4));
		outb(sb16dspDma16Page, page);
		outb(DMA16_FLIP_FLOP, 0);
		outw(sb16dspDma16Base, 0);
		outw(sb16dspDma16Count, SB16_DMA_BUFSIZE / 2 - 1);

		modePort	= DMA16_MODE;
		mask		= sb16Dma16 - 4;
	}
	else
	{
		int	count	= SB16_DMA_BUFSIZE - 1;

		outb(DMA8_WR_SINGLE, 0x04 | sb16Dma8); /* disable DMA channel */
		outb(sb16dspDma8Page, page);
		outb(DMA8_FLIP_FLOP, 0);
		outb(sb16dspDma8Base, 0);
		outb(sb16dspDma8Base, 0);
		outb(sb16dspDma8Count, count & 0xFF);
		outb(sb16dspDma8Count, count >> 8);

		modePort	= DMA8_MODE;
		mask		= sb16Dma8;
	}

	if (pSoundInit->bRecord == 0)
	{
		outb(modePort, 0x58 | mask);
		sb16dsp_dsp_write(0x41);
	}
	else
	{
		outb(modePort, 0x54 | mask);
		sb16dsp_dsp_write(0x42);
	}

	sb16dsp_dsp_write(pSoundInit->sampleRate >> 8);
	sb16dsp_dsp_write(pSoundInit->sampleRate & 0xFF);

	/* enable DMA channel */
	if (sb16dspBits == 16)
		outb(DMA16_WR_SINGLE, mask);
	else
		outb(DMA8_WR_SINGLE, mask);

	spl(s);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_calc_blocksize (int sampleRate)
|
|	sampleRate	Playback/record sampling rate
|
|	Calculates a PCM block size suitable for the supplied sampling rate.
|	The returned PCM block size will provide enough room for no more than
|	100ms worth of data.
|
|	Returns the calculated PCM block size.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_calc_blocksize (int sampleRate)
#else
static int
sb16dsp_calc_blocksize (sampleRate)
int sampleRate;
#endif
{
	static int	blkSizes[] = { 512, 1024, 2048, 4096, 8192, 16384 };
	int			kBps;
	int			desiredBlkSize;
	int			i;

	kBps			= sampleRate * (sb16dspBits / 8) * (sb16dspStereo + 1);
	desiredBlkSize	= kBps / 10;
	i				= dim(blkSizes) - 1;

	while (1)
	{
		if (i == 0 || desiredBlkSize > blkSizes[i])
			return(blkSizes[i]);

		--i;
	}
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_init (char *data)
|
|	data		Pointer to the user supplied SoundInit structure
|
|	Prepares the driver & hardware for playback or recording.
|
|	Returns 0 on success, or an error code on failure.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_init (char * data)
#else
static int
sb16dsp_ioctl_init (data)
char	*data;
#endif
{
	SoundInit	soundInit;

	if ((sb16dspMode & SB16DSP_STOPPED) == 0)
		return(EBUSY);

	if (sb16dsp_reset() != 0)
		return(0);

	if (ukcopy(data, &soundInit, sizeof(soundInit)) != 0)
	{
		int	i;

		bzero(sb16dspDMABuffer, SB16_DMA_BUFSIZE);

		sb16dspRecordError	= 0;
		sb16dspRecord		= !!soundInit.bRecord;
		sb16dspStereo		= soundInit.bStereo & 0x01;
		sb16dspLength		= soundInit.length;
		sb16dspBits			= soundInit.bits;
		sb16dspBufferIndex	= 0;
		sb16dspBlockIndex	= 0;
		sb16dspBytes		= 0;

		sb16dsp_init_transfer(&soundInit);

		/* buffers are locked and then unlocked for recording, the */
		/* opposite for playback 								   */
		for (i = 0; i < dim(sb16dspBufferLock); ++i)
			sb16dspBufferLock[i]	= sb16dspRecord;
	}
	else
		return(EFAULT);

	/* sanity check */
	if (sb16dspBits != 8 && sb16dspBits != 16)
		return(EIO);

	if (soundInit.sampleRate < 5000 || soundInit.sampleRate > 44100)
		return(EIO);

	sb16dspBlockSize	= sb16dsp_calc_blocksize(soundInit.sampleRate);
	sb16dspBlocks		= SB16_DMA_BUFSIZE / sb16dspBlockSize;

	/* driver is now initialized */
	sb16dspMode	= SB16DSP_INITIALIZED | SB16DSP_STOPPED;

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_end ()
|
|	When playing, this function causes the caller to sleep until the PCM
|	data has finished playing.
|
|	Returns 0 on success, ENODEV if recording, or an error code in the
|	user structure if the processes sleep was interrupted.  Note that the
|	caller must issue another end command if an interruption occurs.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_end (void)
#else
static int
sb16dsp_ioctl_end ()
#endif
{
	int	s;

	if (sb16dspMode & SB16DSP_RECORDING)
		return(ENODEV);

	s	= sphi();

	if (!(sb16dspMode & SB16DSP_STOPPED))
	{
		while (sb16dspLength)
		{
			if (sb16dsp_sleep("sb16end") != 0)
				break;
		}

		if (sb16dspLength == 0)
			sb16dspMode	= SB16DSP_STOPPED;
	}

	spl(s);

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_stop ()
|
|	Stops playback/recording immediately.  The DMA controller is programmed
|	to disallow any further DMA requests by the sound card, and the DSP
|	is reset.  If the driver is in the record mode, the DMA controller is
|	queried to determine how many bytes (not accounted for by the interrupt
|	routine) are currently in the buffer.
|
|	Returns 0 on success, the user's error code is set upon failure.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_stop (void)
#else
static int
sb16dsp_ioctl_stop ()
#endif
{
	if (sb16dspBits == 16)
		outb(DMA16_WR_SINGLE, 0x04 | (sb16Dma16 - 4));
	else
		outb(DMA8_WR_SINGLE, 0x04 | sb16Dma8);	/* disable DMA channel */

	if (sb16dspMode & (SB16DSP_PLAYING | SB16DSP_RECORDING))
		sb16dsp_reset();

	if (sb16dspMode & SB16DSP_RECORDING)
	{
		if (sb16dspBits == 16)
		{
			sb16dspLength = inw(sb16dspDma16Base) << 1;

			if (sb16dspStereo)
				sb16dspLength	&= 0xFFFC;
		}
		else
		{
			int	s = sphi();

			outb(DMA8_FLIP_FLOP, 0);
			sb16dspLength = inb(sb16dspDma8Base);
			sb16dspLength = sb16dspLength | (inb(sb16dspDma8Base) << 8);
			spl(s);

			if (sb16dspStereo)
				sb16dspLength	&= 0xFFFE;
		}

		sb16dspMode	= SB16DSP_STOPPED | SB16DSP_RECEND;
	}
	else
		sb16dspMode	= SB16DSP_STOPPED;

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_pause ()
|
|	Pauses the transfer of data to/from the DSP.
|
|	Returns 0 on success, -1 on error.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_pause (void)
#else
static int
sb16dsp_ioctl_pause ()
#endif
{
	return(sb16dsp_dsp_write(SB16_DSPCMD_PAUSE));
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_continue ()
|
|	Continues the transfer of data to/from the DSP that was paused by
|	sb16dsp_ioctl_pause().
|
|	Returns 0 on success, -1 on error.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_continue (void)
#else
static int
sb16dsp_ioctl_continue ()
#endif
{
	return(sb16dsp_dsp_write(SB16_DSPCMD_CONTINUE));
}

/*---------------------------------------------------------------------------- |	int
|	sb16dsp_ioctl_record ()
|
|	Initiates recording.
|
|	Returns 0 on success, -1 on error.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_record (void)
#else
static int
sb16dsp_ioctl_record ()
#endif
{
	/* start DMA */
	int		s;
	int		nSamples;

	s	= sphi();

	if (sb16dspBits == 16)
	{
		nSamples = sb16dspBlockSize / 2 - 1;
		sb16dsp_dsp_write(0xBE);
		sb16dsp_dsp_write((sb16dspStereo << 5) | 0x10);
	}
	else
	{
		nSamples = sb16dspBlockSize - 1;
		sb16dsp_dsp_write(0xCE);
		sb16dsp_dsp_write(sb16dspStereo << 5);
	}

	sb16dsp_dsp_write(nSamples & 0xFF);
	sb16dsp_dsp_write(nSamples >> 8);
	sb16dspMode	= SB16DSP_RECORDING;
	spl(s);

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_query (char *data)
|
|	data		Pointer to the user's SoundInfo structure
|
|	Copies driver specific information into an SoundInfo structure in the
|	user's data segment.
|
|	Returns 0 on success, or an error code on error.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_query (char * data)
#else
static int
sb16dsp_ioctl_query (data)
char *data;
#endif
{
	SoundInfo	soundInfo;

	soundInfo.deviceCode	= SIDC_SOUND_BLASTER_16;
	soundInfo.recordFlags	= SIRF_PAUSE |
							  SIRF_8BIT_MONO | SIRF_8BIT_STEREO |
							  SIRF_16BIT_MONO | SIRF_16BIT_STEREO;
	soundInfo.playFlags		= SIPF_PAUSE |
							  SIPF_8BIT_MONO | SIPF_8BIT_STEREO |
							  SIPF_16BIT_MONO | SIPF_16BIT_STEREO;
	soundInfo.mixerFlags	= SIMXF_CROSS;
	soundInfo.miscFlags		= 0;

	bzero(soundInfo.deviceName, sizeof(soundInfo.deviceName));
	bcopy(sb16dspDeviceName, soundInfo.deviceName, sizeof(sb16dspDeviceName));
	kucopy(&soundInfo, data, sizeof(soundInfo));

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_level8m (int *pLeft, int *pRight, uchar_t *pBuffer)
|
|	pLeft		Pointer to int that will receive the left channel level
|	pRight		Pointer to int that will receive the right channel level
|	pBuffer		Pointer into DMA buffer
|
|	Calculates the left/right channel levels for an 8-bit mono buffer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_level8m (int * pLowL, int * pHighL,
  int * pLowR, int * pHighR, uchar_t * pBuffer)
#else
static void
sb16dsp_level8m (pLowL, pHighL, pLowR, pHighR, pBuffer)
int		*pLowL;
int		*pHighL;
int		*pLowR;
int		*pHighR;
uchar_t	*pBuffer;
#endif
{
	uchar_t	high	= 0;
	uchar_t	low		= UCHAR_MAX;
	int		len		= sb16dspBlockSize;

	while (len--)
	{
		uchar_t v	= *pBuffer++;

		if (v < low)
			low	= v;

		if (v > high)
			high = v;
	}

	*pLowL	= (int)low << 8;
	*pHighL	= (int)high << 8;
	*pLowR	= (int)low << 8;
	*pHighR	= (int)high << 8;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_level8s (int *pLeft, int *pRight, uchar_t *pBuffer)
|
|	pLeft		Pointer to int that will receive the left channel level
|	pRight		Pointer to int that will receive the right channel level
|	pBuffer		Pointer into DMA buffer
|
|	Calculates the left/right channel levels for an 8-bit stereo buffer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_level8s (int * pLowL, int * pHighL,
  int * pLowR, int * pHighR, uchar_t * pBuffer)
#else
static void
sb16dsp_level8s (pLowL, pHighL, pLowR, pHighR, pBuffer)
int		*pLowL;
int		*pHighL;
int		*pLowR;
int		*pHighR;
uchar_t	*pBuffer;
#endif
{
	uchar_t	highL	= 0;
	uchar_t highR	= 0;
	uchar_t	lowL	= UCHAR_MAX;
	uchar_t	lowR	= UCHAR_MAX;
	int		len		= sb16dspBlockSize / 2;

	while (len--)
	{
		uchar_t	l	= *pBuffer++;
		uchar_t	r	= *pBuffer++;

		if (l < lowL)
			lowL	= l;

		if (l > highL)
			highL	= l;

		if (r < lowR)
			lowR	= r;

		if (r > highR)
			highR	= r;
	}

	*pLowL	= (int)lowL << 8;
	*pHighL	= (int)highL << 8;
	*pLowR	= (int)lowR << 8;
	*pHighR	= (int)highR << 8;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_level16m (int *pLeft, int *pRight, short *pBuffer)
|
|	pLeft		Pointer to int that will receive the left channel level
|	pRight		Pointer to int that will receive the right channel level
|	pBuffer		Pointer into DMA buffer
|
|	Calculates the left/right channel levels for an 16-bit mono buffer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_level16m (int * pLowL, int * pHighL,
  int * pLowR, int * pHighR, short * pBuffer)
#else
static void
sb16dsp_level16m (pLowL, pHighL, pLowR, pHighR, pBuffer)
int		*pLowL;
int		*pHighL;
int		*pLowR;
int		*pHighR;
short	*pBuffer;
#endif
{
	short	high	= SHRT_MIN;
	short	low		= SHRT_MAX;
	int		len		= sb16dspBlockSize / 2;

	while (len--)
	{
		short v	= *pBuffer++;

		if (v < low)
			low	= v;

		if (v > high)
			high = v;
	}

	*pLowL	= (int)low + 0x8000;
	*pHighL	= (int)high + 0x8000;
	*pLowR	= (int)low + 0x8000;
	*pHighR	= (int)high + 0x8000;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_level16s (int *pLeft, int *pRight, short *pBuffer)
|
|	pLeft		Pointer to int that will receive the left channel level
|	pRight		Pointer to int that will receive the right channel level
|	pBuffer		Pointer into DMA buffer
|
|	Calculates the left/right channel levels for an 16-bit stereo buffer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_level16s (int * pLowL, int * pHighL,
  int * pLowR, int * pHighR, short * pBuffer)
#else
static void
sb16dsp_level16s (pLowL, pHighL, pLowR, pHighR, pBuffer)
int		*pLowL;
int		*pHighL;
int		*pLowR;
int		*pHighR;
short	*pBuffer;
#endif
{
	short	highL	= 0;
	short	highR	= 0;
	short	lowL	= SHRT_MAX;
	short	lowR	= SHRT_MAX;
	int		len		= sb16dspBlockSize / 4;

	while (len--)
	{
		short l = *pBuffer++;
		short r	= *pBuffer++;

		if (l < lowL)
			lowL	= l;

		if (l > highL)
			highL	= l;

		if (r < lowR)
			lowR	= r;

		if (r > highR)
			highR	= r;
	}

	*pLowL	= (int)lowL + 0x8000;
	*pHighL	= (int)highL + 0x8000;
	*pLowR	= (int)lowR + 0x8000;
	*pHighR	= (int)highR + 0x8000;
}

/*----------------------------------------------------------------------------
|	void
|	sb16dsp_level (SoundLevel *pSoundLevel, char *pBuffer)
|
|	pSoundLevel	Pointer to a SoundLevel structure in kernel space
|	pBuffer		Pointer into DMA buffer
|
|	Sets the fields of the SoundLevel structure based on the supplied
|	buffer.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static void sb16dsp_level (SoundLevel * pSoundLevel, char * pBuffer)
#else
static void
sb16dsp_level (pSoundLevel, pBuffer)
SoundLevel  *pSoundLevel;
char		*pBuffer;
#endif
{
	int	lowL;
	int	highL;
	int	lowR;
	int	highR;

	if (sb16dspStereo)
	{
		if (sb16dspBits == 8)
			sb16dsp_level8s(&lowL, &highL, &lowR, &highR, (uchar_t *)pBuffer);
		else
			sb16dsp_level16s(&lowL, &highL, &lowR, &highR, (short *)pBuffer);
	}
	else
	{
		if (sb16dspBits == 8)
			sb16dsp_level8m(&lowL, &highL, &lowR, &highR, (uchar_t *)pBuffer);
		else
			sb16dsp_level16m(&lowL, &highL, &lowR, &highR, (short *)pBuffer);
	}

	pSoundLevel->left	= highL - lowL;
	pSoundLevel->right	= highR - lowR;

	if (lowL == 0 || highL == 0xFFFF)
		pSoundLevel->bClipLeft	= 1;
	else
		pSoundLevel->bClipLeft	= 0;

	if (lowR == 0 || highR == 0xFFFF)
		pSoundLevel->bClipRight	= 1;
	else
		pSoundLevel->bClipRight	= 0;
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_level (char *data)
|
|	data		Pointer to a SoundLevel structure in user space
|
|	Sets the fields of the user supplied SoundLevel structure based on the
|	data in the DMA buffer.  If the driver is recording, the returned levels
|	are based on the last completed block.  If the driver is in playback
|	mode, the returned levels are based on the block currently playing.
|
|	Returns 0 on success, or ENXIO if the driver is not recording/playing.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_level (char * data)
#else
static int
sb16dsp_ioctl_level (data)
char *data;
#endif
{
	SoundLevel	soundLevel;
	int			blockIx;

	if (sb16dspMode & SB16DSP_STOPPED)
		return(ENXIO);

	if (sb16dspRecord)
	{
		blockIx	= sb16dspBlockIndex - 1;

		if (blockIx == -1)
			blockIx = sb16dspBlocks - 1;
	}
	else
		blockIx	= sb16dspBlockIndex;

	if (sb16dspBytes)
		sb16dsp_level(&soundLevel, sb16dspDMABuffer+blockIx*sb16dspBlockSize);
	else
	{
		soundLevel.left		= 0;
		soundLevel.right	= 0;
	}

	if (copyout(&soundLevel, data, sizeof(soundLevel)) != 0)
		return(EFAULT);

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl_pos (data)
|
|	data		Pointer to a SoundPosition structure in user space
|
|	Fills in the user supplied SoundPosition structure.
|
|	Returns 0 on success, or an error code on failure.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl_pos (char * data)
#else
static int
sb16dsp_ioctl_pos (data)
char *data;
#endif
{
	SoundPosition	soundPos;

	soundPos.bytes	= sb16dspBytes;

	if (copyout(&soundPos, data, sizeof(soundPos)) != 0)
		return(EFAULT);

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_ioctl (int cmd, char *data)
|
|	cmd		IOCTL command
|	data	User data (if any) associated with command
|
|	ioctl() entry point for DSP related functions.
|
|	Returns an error code or 0 on success.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_ioctl (int cmd, char * data)
#else
static int
sb16dsp_ioctl (cmd, data)
int		cmd;
char	*data;
#endif
{
	int	retCode	= ENXIO;

	switch (cmd)
	{
		case SOUND_INIT:
			retCode	= sb16dsp_ioctl_init(data);
			break;

		case SOUND_END:
			retCode	= sb16dsp_ioctl_end();
			break;

		case SOUND_STOP:
			retCode	= sb16dsp_ioctl_stop();
			break;

		case SOUND_RECORD:
			retCode	= sb16dsp_ioctl_record();
			break;

        case SOUND_PAUSE:
			retCode	= sb16dsp_ioctl_pause();
			break;

		case SOUND_CONTINUE:
			retCode	= sb16dsp_ioctl_continue();
			break;

		case SOUND_QUERY:
			retCode	= sb16dsp_ioctl_query(data);
			break;

		case SOUND_LEVEL:
			retCode	= sb16dsp_ioctl_level(data);
			break;

		case SOUND_POS:
			retCode	= sb16dsp_ioctl_pos(data);
			break;

		default:
			break;
	}

	return(retCode);
}

/*----------------------------------------------------------------------------
|	void
|	sb16intr ()
|
|	Interrupt routine handler.  If the interrupt routine is not invoked due
|	to an end of block condition, then a spurious interrupt has occurred
|	and it is discarded.  Otherwise, the block index is incremented (and
|	wrapped to 0 if necessary) and:
|
|	1. If the driver is recording, check to make sure that the new block
|	   has been locked by the foreground process (meaning that it has
|	   processed all data currently available in that block).  If the block
|	   is not locked, then set a flag indicating an error condition.  The
|	   next call to read() will return EIO indicating the presence of
|	   invalidated data.
|
|	2. If the driver is playing, check to see if the last block has been
|	   played.  If so, set sb16dspMode to indicate a stopped condition
|	   and return.  Otherwise, check to see if we have one more block left.
|	   If this is the case, program the final block for single-cycle
|	   transfer mode.
|
|	If a PCM interrupt was generated, wakeup any sleeping processes that
|	are waiting for unlocked buffers or the end of playback.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
void sb16intr (void)
#else
void
sb16intr ()
#endif
{
	int	irqType;

	outb(SB16_MIXER_ADDRESS, SB16_MXRREG_IRQ);
	irqType	= inb(SB16_MIXER_DATA);

	if (irqType & (SB16_IRQ_STATUS_16BIT | SB16_IRQ_STATUS_8BIT))
	{
		if (irqType & SB16_IRQ_STATUS_16BIT)
			inb(SB16_IRQ_RESET16);
		else
			inb(SB16_IRQ_RESET8);

		sb16dspBufferLock[sb16dspBlockIndex]	= 0;

		if (++sb16dspBlockIndex == sb16dspBlocks)
			sb16dspBlockIndex	= 0;

		if (sb16dspRecord)
		{
			if (sb16dspBufferLock[sb16dspBlockIndex] != 1)
				sb16dspRecordError	= 1;

			sb16dspBytes	+= sb16dspBlockSize;
		}
		else
		{
			if (sb16dspLength <= sb16dspBlockSize)
			{
				sb16dspBytes   += sb16dspLength;
				sb16dspLength	= 0;
				sb16dspMode		= SB16DSP_STOPPED;
			}
			else
			{
				sb16dspBytes  += sb16dspBlockSize;
				sb16dspLength -= sb16dspBlockSize;
	
				if (sb16dspLength <= sb16dspBlockSize * 2 &&
					sb16dspLength > sb16dspBlockSize)
				{
					int nSamples;

					if (sb16dspBits == 16)
					{
						nSamples = (sb16dspLength - sb16dspBlockSize) / 2 - 1;
						sb16dsp_dsp_write(0xB0);	/* 16-bit output */
						sb16dsp_dsp_write((sb16dspStereo << 5) | 0x10);
					}
					else
					{
						nSamples = sb16dspLength - sb16dspBlockSize - 1;
						sb16dsp_dsp_write(0xC0);	/* 8-bit output */
						sb16dsp_dsp_write(sb16dspStereo << 5);
					}

					sb16dsp_dsp_write(nSamples & 0xFF);
					sb16dsp_dsp_write(nSamples >> 8);
				}
			}
		}

		if (sb16dspSleeping)
			wakeup(&sb16dspSleeping);
	}
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_open ()
|
|	Called by sb16open() to open the DSP device.
|
|	Returns 0 on success, or EBUSY if the device is currently in use.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_open (void)
#else
static int
sb16dsp_open ()
#endif
{
	if (sb16dspOpen)
		return(EBUSY);

	sb16dspOpen		= 1;
	sb16dspMode		= SB16DSP_STOPPED;
	sb16dspBytes	= 0;

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_close ()
|
|	Closes the DSP device, if recording or playback is taking place it is
|	immediately terminated.
|
|	Returns 0 on success, or an error code on failure.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_close (void)
#else
static int
sb16dsp_close ()
#endif
{
	sb16dsp_ioctl_stop();

	sb16dspOpen	= 0;

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_read_recend (IO *iop)
|
|	iop			I/O request structure
|
|	Reads the remaining data after recording has stopped.
|
|	Returns an error code on failure, 0 on success.
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_read_recend (IO * iop)
#else
static int
sb16dsp_read_recend (iop)
IO	*iop;
#endif
{
	int	lastBlock	= sb16dspLength / sb16dspBlockSize;

	while (iop->io_ioc)
	{
		int	bufferIx;
		int	copySize;

		bufferIx	= sb16dspBufferIndex / sb16dspBlockSize;

		if (bufferIx == lastBlock && sb16dspBufferIndex <= sb16dspLength)
		{
			if (sb16dspBufferIndex == sb16dspLength)
				return(0);

			copySize = sb16dspLength - sb16dspBufferIndex;
		}
		else
			copySize = sb16dspBlockSize-(sb16dspBufferIndex % sb16dspBlockSize);

		if (copySize > iop->io_ioc)
			copySize	= iop->io_ioc;

		iowrite(iop, sb16dspDMABuffer + sb16dspBufferIndex, copySize);
		sb16dspBufferIndex	+= copySize;

		if (sb16dspBufferIndex == SB16_DMA_BUFSIZE)
			sb16dspBufferIndex	= 0;
	}

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_read (IO *iop)
|
|	iop			I/O request structure
|
|	Reads PCM data from the DMA buffer.
|
|	Returns 0 on success.  Possible error codes are:
|		EBUSY 		the device is in playback mode
|		EIO			there is invalid data in the DMA buffer (overflow)
|		EINTR		interruption from sleep
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_read (IO * iop)
#else
static int
sb16dsp_read (iop)
IO	*iop;
#endif
{
	if (!sb16dspRecord)
		return(EBUSY);

	if (sb16dspRecordError)
		return(EIO);

	if ((sb16dspMode & SB16DSP_RECORDING) == 0)
		return(EIO);

	if (sb16dspMode & SB16DSP_RECEND)
		return(sb16dsp_read_recend(iop));

	while (iop->io_ioc)
	{
		int	bufferIx;
		int	s;
		int	copySize;

		bufferIx	= sb16dspBufferIndex / sb16dspBlockSize;

		s	= sphi();

		if (sb16dspBufferLock[bufferIx])
		{
			while (sb16dspBufferLock[bufferIx])
			{
				if (iop->io_flag & (IONDLY | IONONBLOCK))
				{
					spl(s);
					return(0);
				}
	
				if (sb16dsp_sleep("sb16buf") != 0)
				{
					spl(s);
					return(0);
				}
			}
		}

		spl(s);

		copySize = sb16dspBlockSize - (sb16dspBufferIndex % sb16dspBlockSize);

		if (copySize > iop->io_ioc)
			copySize	= iop->io_ioc;

		iowrite(iop, sb16dspDMABuffer + sb16dspBufferIndex, copySize);
		sb16dspBufferIndex	+= copySize;

		if ((sb16dspBufferIndex % sb16dspBlockSize) == 0)
		{
			sb16dspBufferLock[bufferIx]	= 1;

			if (sb16dspBufferIndex == SB16_DMA_BUFSIZE)
				sb16dspBufferIndex	= 0;
		}
	}

	return(0);
}

/*----------------------------------------------------------------------------
|	int
|	sb16dsp_write (IO *iop)
|
|	iop			I/O request structure
|
|	Writes PCM data to the DMA buffer.
|
|	Returns 0 on success.  Possible error codes are:
|		EBUSY		the device is in record mode
|		EINTR		interruption from sleep
----------------------------------------------------------------------------*/
#if __USE_PROTO__
static int sb16dsp_write (IO * iop)
#else
static int
sb16dsp_write (iop)
IO	*iop;
#endif
{
	if (sb16dspRecord)
		return(EBUSY);

	if ((sb16dspMode & (SB16DSP_PLAYING | SB16DSP_INITIALIZED)) == 0)
		return(EIO);

	while (iop->io_ioc)
	{
		int	copySize;
		int	bufferIx;
		int	s;

		bufferIx	= sb16dspBufferIndex / sb16dspBlockSize;

		s	= sphi();

		if (sb16dspLength == 0)
		{
			spl(s);
			return(EIO);
		}

		while (sb16dspBufferLock[bufferIx])
		{
			if (sb16dspLength == 0)
			{
				spl(s);
				return(EIO);
			}

			if (iop->io_flag & (IONDLY | IONONBLOCK))
			{
				spl(s);
				return(0);
			}

			if (sb16dsp_sleep("sb16buf") != 0)
			{
				spl(s);
				return(0);
			}
		}

		spl(s);

		copySize = sb16dspBlockSize - (sb16dspBufferIndex % sb16dspBlockSize);

		if (copySize > iop->io_ioc)
			copySize	= iop->io_ioc;

		ioread(iop, sb16dspDMABuffer + sb16dspBufferIndex, copySize);

		s	= sphi();

		sb16dspBufferIndex	+= copySize;

		if ((sb16dspBufferIndex % sb16dspBlockSize) == 0)
		{
			sb16dspBufferLock[bufferIx]	= 1;
	
			if (sb16dspBufferIndex == SB16_DMA_BUFSIZE)
				sb16dspBufferIndex	= 0;
		}

		if ((sb16dspMode & SB16DSP_PLAYING) == 0)
		{
			/* start DMA */
			int	nSamples;

			if (sb16dspLength > sb16dspBlockSize)
			{
				/* autoinitialize */
				if (sb16dspBits == 16)
				{
					nSamples = sb16dspBlockSize / 2 - 1;
					sb16dsp_dsp_write(0xB6);	/* 16-bit output */
				}
				else
				{
					nSamples = sb16dspBlockSize - 1;
					sb16dsp_dsp_write(0xC6);	/* 8-bit output  */
				}
			}
			else
			{
				/* single cycle */
				if (sb16dspBits == 16)
				{
					nSamples = sb16dspLength / 2 - 1;
					sb16dsp_dsp_write(0xB0);	/* 16-bit output */
				}
				else
				{
					nSamples = sb16dspLength - 1;
					sb16dsp_dsp_write(0xC0);	/* 8-bit output  */
				}
			}

			if (sb16dspBits == 16)
				sb16dsp_dsp_write((sb16dspStereo << 5) | 0x10);
			else
				sb16dsp_dsp_write(sb16dspStereo << 5);

			sb16dsp_dsp_write(nSamples & 0xFF);
			sb16dsp_dsp_write(nSamples >> 8);

			sb16dspMode	= SB16DSP_PLAYING;

			if (sb16dspLength <= sb16dspBlockSize * 2 &&
				sb16dspLength > sb16dspBlockSize)
			{
				if (sb16dspBits == 16)
				{
					nSamples = (sb16dspLength - sb16dspBlockSize) / 2 - 1;
					sb16dsp_dsp_write(0xB0);	/* 16-bit output */
					sb16dsp_dsp_write((sb16dspStereo << 5) | 0x10);
				}
				else
				{
					nSamples = sb16dspLength - sb16dspBlockSize - 1;
					sb16dsp_dsp_write(0xC0);	/* 8-bit output */
					sb16dsp_dsp_write(sb16dspStereo << 5);
				}

				sb16dsp_dsp_write(nSamples & 0xFF);
				sb16dsp_dsp_write(nSamples >> 8);
			}
		}

		spl(s);
	}

	return(0);
}

/* DSP function table. */
struct snd_dsp_ft		sb16_dsp_ftp = {
	sb16dsp_init,
	sb16dsp_open,
	sb16dsp_close,
	sb16dsp_read,
	sb16dsp_write,
	sb16dsp_ioctl,
};

struct snd_dsp_ft		* snd_dsp_ftp = & sb16_dsp_ftp;
