/* RWDOCS */
#ifndef TCP_H
#define TCP_H

/* filename	: tcp.h
 * purpose	: TCP implementation. Follows RFC 793 as closely as possible 
 *
 */

#include "internet.h"
#include "netuser.h"
#include "timer.h"

#define	DEF_WND	2048	/* Default receiver window */
#define	NTCB	19	/* # TCB hash table headers */
#define	DEF_MSS	512	/* Default maximum segment size */
#define	DEF_RTT	5000	/* Initial guess at round trip time (5 sec) */
#define	MSL2	30	/* Guess at two maximum-segment lifetimes */

/* Round trip timing parameters */
#define	AGAIN	8	/* Average RTT gain = 1/8 */
#define	DGAIN	4	/* Mean deviation gain = 1/4 */

/* TCP segment header -- internal representation
 * Note that this structure is NOT the actual header as it appears on the
 * network (in particular, the offset and checksum fields are missing).
 * All that knowledge is in the functions ntohtcp() and htontcp() in tcpsubr.c
 */
struct tcp {
	int32 source;	/* Source port */
	int32 dest;	/* Destination port */
	int32 seq;	/* Sequence number */
	int32 ack;	/* Acknowledgment number */
	int32 flags;	/* Flags, data offset */
#define	URG	0x20	/* URGent flag */
#define	ACK	0x10	/* ACKnowledgment flag */
#define	PSH	0x08	/* PuSH flag */
#define	RST	0x04	/* ReSeT flag */
#define	SYN	0x02	/* SYNchronize flag */
#define	FIN	0x01	/* FINal flag */
	int32 wnd;	/* Receiver flow control window */
	int32 up;	/* Urgent pointer */
	int32 mss;	/* Optional max seg size */
};

/* TCP options */
#define	EOL_KIND	0
#define	NOOP_KIND	1
#define	MSS_KIND	2

#define	TCPLEN		20
#define	MSS_LENGTH	4

/* Resequencing queue entry */
struct reseq {
	struct reseq *next;	/* Linked-list pointer */
	int32 tos;		/* Type of service */
	struct tcp seg;		/* TCP header */
	struct mbuf *bp;	/* data */
	int32 length;		/* data length */
};
#define	NULLRESEQ	(struct reseq *)0

/* TCP connection control block */
struct tcb {
	struct tcb *prev;	/* Linked list pointers for hash table */
	struct tcb *next;

	struct connection conn;

	int32 state;	/* Connection state */
#define	CLOSED		0	/* Must be 0 */
#define	LISTEN		1
#define	SYN_SENT	2
#define	SYN_RECEIVED	3
#define	ESTABLISHED	4
#define	FINWAIT1	5
#define	FINWAIT2	6
#define	CLOSE_WAIT	7
#define	CLOSING		8
#define	LAST_ACK	9
#define	TIME_WAIT	10

	char reason;		/* Reason for closing */
#define	NORMAL		0	/* Normal close */
#define	RESET		1	/* Reset by other end */
#define	TIMEOUT		2	/* Excessive retransmissions */
#define	NETWORK		3	/* Network problem (ICMP message) */

/* If reason == NETWORK, the ICMP type and code values are stored here */
	int32 type;
	int32 code;

	/* Send sequence variables */
	struct {
		int32 una;	/* First unacknowledged sequence number */
		int32 nxt;	/* Next sequence num to be sent for the first time */
		int32 ptr;	/* Working transmission pointer */
		int32 wnd;	/* Other end's offered receive window */
		int32 up;	/* Send urgent pointer */
		int32 wl1;	/* Sequence number used for last window update */
		int32 wl2;	/* Ack number used for last window update */
	} snd;
	int32 iss;		/* Initial send sequence number */
	int32 cwind;		/* Congestion window */
	int32 ssthresh;		/* Slow-start threshold */
	int32 resent;		/* Count of bytes retransmitted */

	/* Receive sequence variables */
	struct {
		int32 nxt;	/* Incoming sequence number expected next */
		int32 wnd;	/* Our offered receive window */
		int32 up;	/* Receive urgent pointer */
	} rcv;
	int32 irs;		/* Initial receive sequence number */
	int32 mss;		/* Maximum segment size */
	int32 rerecv;		/* Count of duplicate bytes received */

	int32 window;		/* Receiver window and send queue limit */

	char backoff;		/* Backoff interval */
	void (*r_upcall)();	/* Call when "significant" amount of data arrives */
	void (*t_upcall)();	/* Call when ok to send more data */
	void (*s_upcall)();	/* Call when connection state changes */
	char flags;		/* Control flags */
#define	FORCE	1		/* We owe the other end an ACK or window update */
#define	CLONE	2		/* Server-type TCB, cloned on incoming SYN */
#define	RETRAN	4		/* A retransmission has occurred */
#define	ACTIVE	8		/* TCB created with an active open */
#define	SYNACK	16		/* Our SYN has been acked */
	int32 tos;		/* Type of service (for IP) */

	struct mbuf *rcvq;	/* Receive queue */
	int32 rcvcnt;

	struct mbuf *sndq;	/* Send queue */
	int32 sndcnt;		/* Number of unacknowledged sequence numbers on
				 * send queue. NB: includes SYN and FIN, which don't
				 * actually appear on sndq!
				 */


	struct reseq *reseq;	/* Out-of-order segment queue */
	struct timer timer;	/* Retransmission timer */
	struct timer rtt_timer;	/* Round trip timer */
	int32 rttseq;		/* Sequence number being timed */
	int32 srtt;		/* Smoothed round trip time, milliseconds */
	int32 mdev;		/* Mean deviation, milliseconds */

	char *user;		/* User parameter (e.g., for mapping to an
				 * application control block
				 */
};

#define	NULLTCB	(struct tcb *)0

/* TCP statistics counters */
struct tcp_stat {
	int32 runt;		/* Smaller than minimum size */
	int32 checksum;		/* TCP header checksum errors */
	int32 conout;		/* Outgoing connection attempts */
	int32 conin;		/* Incoming connection attempts */
	int32 resets;		/* Resets generated */
	int32 bdcsts;		/* Bogus broadcast packets */
};

extern struct tcp_stat tcp_stat;

extern struct tcb *tcbs[];



void rehash_tcb(),tcp_output(),dump_seg();

#ifdef GCC
	/* tcpcmd.c */
void close_self(struct tcb *tcb, int reason);

	/* tcpdump.c */
void tcp_dump(struct mbuf **bpp,int32 source,int32 dest,int check);

	/* tcpsubr.c */
struct tcb * lookup_tcb(struct connection *conn);
struct tcb * create_tcb(struct connection *conn);
void close_self(struct tcb *tcb, int reason);
int32 iss();
int seq_within(int32 x,int32 low,int32 high);
int seq_lt(int32 x,int32 y);
int seq_gt(int32 x,int32 y);
int seq_ge(int32 x,int32 y);
void link_tcb(struct tcb *tcb);
void unlink_tcb(struct tcb *tcb);
void setstate(struct tcb *tcb,int newstate);
struct mbuf *htontcp(struct tcp *tcph,struct mbuf *data,struct pseudo_header *ph);
int ntohtcp(struct tcp *tcph,struct mbuf **bpp);
void sndmsg(struct tcb *tcb, char *msg);

	/* timer.c */
void start_timer(struct timer *t);
void stop_timer(struct timer *t);

	/* tcpin.c */
void tcp_icmp(int32 source,int32 dest,int type,int code,struct mbuf **bpp);
void tcp_input(struct mbuf *bp,int protocol,int32 source,int32 dest,int tos,int length,int rxbroadcast);

	/* tcpout.c */
void tcp_output(struct tcb *tcb);

	/* tcpuser.c */
struct tcb *open_tcp(struct socket *lsocket,struct socket *fsocket,int mode,int window,void (*r_upcall)(),void (*t_upcall)(),void (*s_upcall)(),int tos,int *user);
int send_tcp(struct tcb *tcb,struct mbuf *bp);
int recv_tcp(struct tcb *tcb,struct mbuf **bp,int cnt);
int close_tcp(struct tcb *tcb);
int del_tcp(struct tcb *tcb);
int tprintf(struct tcb *tcb,char *message,char *arg1,char *arg2,char *arg3);
int tcpval(struct tcb *tcb);
int kick_tcp(struct tcb *tcb);
void reset_tcp(struct tcb *tcb);
char *tcp_port(int n);

#else
struct tcb * lookup_tcb();
struct tcb * create_tcb();
void link_tcb();
void unlink_tcb();

void close_self();
void setstate();
struct mbuf *htontcp();
void sndmsg();
void start_timer();
void stop_timer();

void tcp_icmp();
void tcp_input();

#endif /* GCC */

struct mbuf *htontcp();

/* TCP primitives */
struct tcb *open_tcp();
int send_tcp(),recv_tcp(),close_tcp(),del_tcp();
void state_tcp(),tcp_dump();

extern int16 tcp_mss;
extern int16 tcp_window;
extern int32 tcp_irtt;

#endif /* TCP_H */
/* RWDOCE */
