/*
   Copyright (c) 1994 by Harry Pulley.

   May be freely distributed and modified provided that this copyright notice
   is not removed.  I am not responsible for any damages caused in any way,
   shape or form.
*/

#include <sys/coherent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/devices.h>
#include <sys/poll.h>
#include <sys/errno.h>
#include <sys/sbioctl.h>
#include <sys/mmu.h>

/* patchable parameters: */

extern sb_port_base;
extern sb_dma_chan;
extern sb_irq_level;

/* port locations: */

#define DSP_RESET	(sb_port_base+0x06)	/* write-only */
#define DSP_READ	(sb_port_base+0x0A)	/* read-only */
#define DSP_WRITE	(sb_port_base+0x0C)	/* write */
#define DSP_BUF_STATUS	(sb_port_base+0x0C)	/* read - bit7 */
#define DSP_DATA_STATUS	(sb_port_base+0x0e)	/* read-only - bit7 */

/* port commands: */

#define DIRECT_MODE_DAC	0x10
#define DMA_MODE_DAC	0x14
#define DIRECT_MODE_ADC	0x20
#define DMA_MODE_ADC	0x24
#define SET_TIME_CONST	0x40
#define SET_BLOCK_SIZE	0x48
#define HS_DMA_MODE_DAC	0x91
#define HS_DMA_MODE_ADC	0x99
#define TURN_ON_SPKR	0xD1
#define TURN_OFF_SPKR	0xD3
#define HALT_DMA	0xD0
#define CONT_DMA	0xD4
#define GET_DSP_VER	0xE1

#define MAX_RETRIES	128

/* static variables: */

static sb_driver_in_use=0,sb_driver_installed=0;
static char *buffer0=NULL, *buffer1=NULL;
static unsigned long buffer0len=0, buffer1len=0;
static event_t play_poll, record_poll;
static playing=-1,recording=-1;
static TIM sbtp;
static intr;
static struct sb_ioctl cue0,cue1;
static buffer0cued=0,buffer1cued=0;
static justcued=0;
static highspeed=0,time_constant=0;
static short sb_ver;

/* functions: */

static unsigned char dspread();

static void dspwrite();

static unsigned short getdspver()
{
	unsigned short sver;

	sver=0;

	dspwrite(GET_DSP_VER);

	sver=(dspread())<<8;
	sver|=dspread();

	return sver;
}

static void sbplayrec(sbstruc,buffer,command)
struct sb_ioctl sbstruc;
char *buffer;
unsigned char command;
{
	if (dmaon(sb_dma_chan,vtop(buffer),sbstruc.buffer_len,(command==DMA_MODE_DAC)?1:0)==0)
	{
		set_user_error(EFAULT);

		return;
	}

	dmago(sb_dma_chan);

	sbstruc.buffer_len--;

	/* set up SB - send command and buffer length */

	if (!highspeed)
		dspwrite(command);
	else
		dspwrite(SET_BLOCK_SIZE);

	dspwrite((unsigned char)(sbstruc.buffer_len&0xFF));
	dspwrite((unsigned char)((sbstruc.buffer_len>>8)&0xFF));

	if (highspeed)
	{
		switch(command)
		{
		case DMA_MODE_DAC:	dspwrite(HS_DMA_MODE_DAC);
					break;
		case DMA_MODE_ADC:	dspwrite(HS_DMA_MODE_ADC);
					break;
		}
	}
}

static unsigned short getdmacount()
{
	unsigned short scount;

	switch(sb_dma_chan)
	{
	case 1:	scount=inb(0x03);
        	scount|=inb(0x03)<<8;
		break;
	case 3:	scount=inb(0x07);
        	scount|=inb(0x07)<<8;
		break;
	}

	return scount;
}

void sbintr()
{
        unsigned short scount;
	int i;

	/* check for spurious interrupts; if DMA is not finished then we didn't
	   cause the interrupt; for some reason IRQ 7 is hit by other devices */

	i = splhi();
	scount=getdmacount();
	if (buffer0cued && scount >= buffer0len) {
	    splx(i);
	    return;
	}
	if (buffer1cued && scount >= buffer1len) {
	    splx(i);
	    return;
	}
	inb(DSP_DATA_STATUS);

	if (playing!=-1)
	{
		if (playing==0&&buffer1cued)
		{
			buffer1cued=0;

			dmaoff(sb_dma_chan);

			sbplayrec(cue1,buffer1,DMA_MODE_DAC);
			playing=1;
			justcued=1;
		}
		else if (playing==1&&buffer0cued)
		{
			buffer0cued=0;

			dmaoff(sb_dma_chan);

			sbplayrec(cue0,buffer0,DMA_MODE_DAC);
			playing=0;
			justcued=1;
		}
		else
			playing=-1;

		if (play_poll.e_procp)
		{
			pollwake(&play_poll);
		}
	}

	if (recording!=-1)
	{
		if (recording==0&&buffer1cued)
		{
			buffer1cued=0;

			dmaoff(sb_dma_chan);

			sbplayrec(cue1,buffer1,DMA_MODE_ADC);
			recording=1;
			justcued=1;
		}
		else if (recording==1&&buffer0cued)
		{
			buffer0cued=0;

			dmaoff(sb_dma_chan);

			sbplayrec(cue0,buffer0,DMA_MODE_ADC);
			recording=0;
			justcued=1;
		}
		else
			recording=-1;

		if (record_poll.e_procp)
			pollwake(&record_poll);
	}
	splx(i);
}
	
static void reset_dsp()
{
	outb(DSP_RESET,0x01);

	busyWait2(NULL,4);

	outb(DSP_RESET,0x00);
};

static void sbload()
{
	int count;

	reset_dsp();

	/* test for existance of SB at configured setup */

	for (count=0;count<MAX_RETRIES;count++)
	{
		if (inb(DSP_READ)==0xAA)
		{
			break;
		}
	}

	if (count>=MAX_RETRIES)
	{
		printf("SB: hardware at port %X not found.\n",sb_port_base);

		return;
	}

	sb_ver=getdspver();

	/* Try to allocate contiguous physical memory for two 64K banks.  These
	   are for DMA.  Two banks are needed so we can double buffer the play
	   and record buffers.  Otherwise, accesses to disk to load or store
	   additional data will cause noticable, undesirable delays. */

	buffer0=(char *)getDmaMem(((unsigned)131072),((unsigned)65536));

	if (buffer0==NULL)
	{
		printf("SB: unable to allocate buffers.\n");

		return;
	}

	buffer1=buffer0+65536;

	/* install interrupt handler */

        setivec(sb_irq_level,sbintr);

	sb_driver_installed=1;

	printf("SB driver (0x%x,%d,%d,%d.%d,%x,%x) Copyright 1994 Harry Pulley\n",sb_port_base,sb_dma_chan,sb_irq_level,(sb_ver&0xff00)>>8,sb_ver&0xff,vtop(buffer0),vtop(buffer1));
}

static void sbunload()
{
	/* remove interrupt handler */

	clrivec(sb_irq_level);

	sb_driver_installed=0;
}

static sbopen(dev,mode)
dev_t dev;
int mode;
{
	if (minor(dev)!=0)
	{
		set_user_error(ENXIO);
		return ENXIO;
	}

	/* check if load worked OK */

	if (!sb_driver_installed)
	{
		set_user_error(ENXIO);
		return ENXIO;
	}

	/* check if driver already opened */

	if (sb_driver_in_use)
	{
		set_user_error(EBUSY);
		return EBUSY;
	}

	/* mark driver as in use */

	sb_driver_in_use=1;

	return 0;
}

static sbclose(dev)
dev_t dev;
{
	/* error if we haven't opened this */

	if (!sb_driver_in_use)
	{
		set_user_error(EIO);
		return EIO;
	}

	/* check if we are still playing; if so, stop playing */

	if (playing!=-1||recording!=-1)
	{
		dspwrite(HALT_DMA);
		dspwrite(TURN_OFF_SPKR);
	}

	/* clean up state variables */

	sb_driver_in_use=0;
	playing=-1;
	recording=-1;
	buffer0cued=0;
	buffer1cued=0;
	justcued=0;
	highspeed=0;
	time_constant=0;

	return 0;
}

static void dspwrite(data)
unsigned char data;
{
	int count;

	for (count=0;count<MAX_RETRIES&&inb(DSP_BUF_STATUS)&0x80;count++)
		;

	outb(DSP_WRITE,data);
}

static unsigned char dspread()
{
	int count;

	for (count=0;count<MAX_RETRIES&&!(inb(DSP_DATA_STATUS)&0x80);count++)
		;

	return (inb(DSP_READ));
}

static unsigned short getdmastatus()
{
	unsigned short status=0;

	switch(sb_dma_chan)
	{
	case 1:	status=inb(0x08)&2;
	case 3:	status=inb(0x08)&8;
	}

	return status;
}

static void sbioctl(dev, command, struc)
dev_t dev;
int command;
struct sb_ioctl *struc;
{
	struct sb_ioctl sbstruc;

	if (struc!=NULL)
		ukcopy(struc,&sbstruc,sizeof(struct sb_ioctl));

	switch(command)
	{
	case MAP_BUFFER0:	if (sbstruc.buffer_addr%PAGE_SIZE)
				{
					set_user_error(EADDRNOTAVAIL);
					return;
				}

				if (sbstruc.buffer_len<=0||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(EINVAL);
					return;
				}

				if (mapPhysUser(sbstruc.buffer_addr,vtop(buffer0),sbstruc.buffer_len))
					buffer0len=sbstruc.buffer_len;
				else
					set_user_error(ENOMEM);

				return;

	case MAP_BUFFER1:	if (sbstruc.buffer_addr%PAGE_SIZE)
				{
					set_user_error(EADDRNOTAVAIL);
					return;
				}

				if (sbstruc.buffer_len<=0||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(EINVAL);
					return;
				}

				if (mapPhysUser(sbstruc.buffer_addr,vtop(buffer1),sbstruc.buffer_len))
					buffer1len=sbstruc.buffer_len;
				else
					set_user_error(ENOMEM);

				return;

	case SET_TIMECONST:	inb(DSP_DATA_STATUS);

				dspwrite(SET_TIME_CONST);

				dspwrite(sbstruc.parameter);

				time_constant=sbstruc.parameter;

				return;

	/* for high speed transfers (>23KHz) we cannot send commands to the DSP
	   until the interrupt occurs */

	/* we may need to subtract one from the length, for the DMA and the SB */

	case PCUE_BUFFER0:	if (recording!=-1)
				{
					set_user_error(EBUSY);

					return;
				}

				if (playing==1)
				{
					if (buffer0cued)
					{
						set_user_error(ENOBUFS);

						return;
					}

					buffer0cued=1;

					ukcopy(struc,&cue0,sizeof(struct sb_ioctl));

					return;
				}

				if (playing==0)
				{
					set_user_error(EAGAIN);

					return;
				}

	case PLAY_BUFFER0:	playing=-1;
				recording=-1;
				buffer0cued=0;

				if (time_constant>211&&!(sb_ver*0xff00))
					highspeed=1;

				if (sbstruc.buffer_len>buffer0len||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(ENOBUFS);
					return;
				}

				sbplayrec(sbstruc,buffer0,DMA_MODE_DAC);

				playing=0;

				return;

	case PCUE_BUFFER1:	if (recording!=-1)
				{
					set_user_error(EBUSY);

					return;
				}

				if (playing==0)
				{
					if (buffer1cued)
					{
						set_user_error(ENOBUFS);

						return;
					}

					buffer1cued=1;

					ukcopy(struc,&cue1,sizeof(struct sb_ioctl));

					return;
				}

				if (playing==1)
				{
					set_user_error(EAGAIN);

					return;
				}

	case PLAY_BUFFER1:	playing=-1;
				recording=-1;
				buffer1cued=0;

				if (time_constant>211&&(sb_ver&0xff00)!=0x10)
					highspeed=1;

				if (sbstruc.buffer_len>buffer1len||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(ENOBUFS);
					return;
				}

				sbplayrec(sbstruc,buffer1,DMA_MODE_DAC);

				playing=1;

				return;

	case RCUE_BUFFER0:	if (playing!=-1)
				{
					set_user_error(EBUSY);

					return;
				}

				if (recording==1)
				{
					if (buffer0cued)
					{
						set_user_error(ENOBUFS);

						return;
					}

					buffer0cued=1;

					ukcopy(struc,&cue0,sizeof(struct sb_ioctl));

					return;
				}

				if (recording==0)
				{
					set_user_error(EAGAIN);

					return;
				}

	case RECORD_BUFFER0:	playing=-1;
				recording=-1;
				buffer0cued=0;

				if (time_constant>211&&(sb_ver&0xff00)==0x30||time_constant>178&&(sb_ver&0xff00)==0x20)
					highspeed=1;

				if (sbstruc.buffer_len>buffer0len||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(ENOBUFS);
					return;
				}

				sbplayrec(sbstruc,buffer0,DMA_MODE_ADC);

				recording=0;

				return;

	case RCUE_BUFFER1:	if (playing!=-1)
				{
					set_user_error(EBUSY);

					return;
				}

				if (recording==0)
				{
					if (buffer1cued)
					{
						set_user_error(ENOBUFS);

						return;
					}

					buffer1cued=1;

					ukcopy(struc,&cue1,sizeof(struct sb_ioctl));

					return;
				}

				if (recording==1)
				{
					set_user_error(EAGAIN);

					return;
				}

	case RECORD_BUFFER1:	playing=-1;
				recording=-1;
				buffer1cued=0;

				if (time_constant>211||time_constant>178&&(sb_ver&0xff00)==0x20)
					highspeed=1;

				if (sbstruc.buffer_len>buffer1len||sbstruc.buffer_len>BUF_SIZE)
				{
					set_user_error(ENOBUFS);
					return;
				}

				sbplayrec(sbstruc,buffer1,DMA_MODE_ADC);

				recording=1;

				return;

	case SPKR_ON:		dspwrite(TURN_ON_SPKR);

				return;

	case SPKR_OFF:		dspwrite(TURN_OFF_SPKR);

				return;

	case DMA_HALT:		dspwrite(HALT_DMA);

				return;

	case DMA_CONT:		dspwrite(CONT_DMA);

				return;

	case DMA_STATUS:	putusd(sbstruc.status,getdmastatus());

				return;

	case RESET_DSP:		reset_dsp();

				return;

	case DMA_COUNT:		putusd(sbstruc.status,getdmacount());

				return;

	case DSP_VER:		putusd(sbstruc.status,sb_ver);

				return;
	}
}

static sbtimeout(val)
int val;
{
	/* wake up polls */

	if (play_poll.e_procp)
		pollwake(&play_poll);

	if (record_poll.e_procp)
		pollwake(&record_poll);

	/* cancel timeout */

	timeout(&sbtp,0,NULL,0);
}

static sbpoll(dev,events,msec)
dev_t dev;
int events, msec;
{
	register int i;

	/* we only handle input and output polling */

	events&= (POLLIN | POLLOUT);

	/* set timeout for poll */

	if (msec>0)
		timeout(&sbtp,(msec+(msec%10)?10:0)/10,sbtimeout,0);

	/* just do output for now; do input when we get recording done */

	i = sphi();
	if (events&POLLOUT && playing!=-1 && !justcued)
	{
		if (msec)
			pollopen(&play_poll);
	
		if (playing!=-1)
			events&=~POLLOUT;
	}

	if (events&POLLIN && recording!=-1 && !justcued)
	{
		if (msec)
			pollopen(&record_poll);

		if (recording!=-1)
			events&=~POLLIN;;
	}

	justcued=0;
	spl(i);

	return events;
}
/* CON structure: */

CON sbcon=
{
	DFCHR|DFPOL,	/* Flags */
	SBP_MAJOR,	/* Major Index */
	sbopen,		/* open */
	sbclose,	/* close */
	NULL,		/* block */
	NULL,		/* read */
	NULL,		/* write */
	sbioctl,	/* ioctl */
	NULL,		/* powerfail */
	NULL,		/* timeout */
	sbload,		/* load */
	sbunload,	/* unload */
	sbpoll		/* poll */
};
