#
/*
 */

/*
 * TM tape driver
 *   Rewrite by Bruce S. Borden and
 *      S. Tucker Taft
 *      Harvard Science Center
 *
 *
 * This TM-11 driver is modeled after the TU-16
 *  driver written at Piscataway.  It has the following features:
 *
 *	- No rewind is done on close
 * for devices with the NOREW bit on in their minor device number.
 * On a close after a write, 2 tape marks are written, and then
 * a single backspace is done.  On a close after a read, a forward
 * seek is done to the next tape mark, leaving the tape positioned after
 * the tape mark.  Hence, to skip a file, simply open the norewind device for
 * reading and then close it.
 *	- With the raw device, it is possible to write a tape mark.
 * A write with zero bytes for a count will cause a tape mark to
 * be written.  Please note that the buffer address must still
 * be a legal address because "physio" checks it.  Writes of less than 4 bytes
 * on the raw device are automatically filled out to 4 bytes.
 *	- On a read, if a tape mark is encountered, an error is
 * returned for the cooked device (errno = EFBIG), and
 * a zero-length record is returned for the raw device.
 *	- Seeks are allowed on the raw device.  Each block is treated
 * as though it were exactly 512 bytes (including tape marks), so that
 * a seek-by-blocks system call will do the desired operation.
 * During a seek on the raw device, tape marks are counted as one block.
 * On the cooked device, an error will be returned if the seek tries
 * to cross a tape mark(EFBIG).  On the raw device, an error is only returned
 * if the seek tries to cross a double tape mark, leaving
 * the tape positioned after the first of the pair.
 *	- Please note:  If this driver may be used on an 11/70,
 * the symbol "CPU70" must be defined in param.h.
 */

#include "../param.h"
#include "../buf.h"
#include "../conf.h"
#include "../user.h"
#include "../file.h"
#include "../reg.h"

struct {
	int tmer;
	int tmcs;
	int tmbc;
	int tmba;
	int tmdb;
	int tmrd;
};

struct	devtab	tmtab;
struct	buf	rtmbuf,	ctmbuf;

/* NUNIT must be power of 2 (NUNIT-1 used as mask) */
#define NUNIT	8
#define	NOREW	040	/* no rewind on close bit (>= NUNIT) */

struct tminfo {
	char	*t_blkno;	/* block num about to read/write */
	char	*t_rawtm;	/* blkno after last tape mark */
	char	*t_tmblk;	/* block number after last write */
	char	t_openf;	/* 0 = not open, 1 = open, -1 = fatal error */
	char	t_tmwritten;	/* 1 = two eof's already written */
} tminfo[NUNIT];

int	tm_sem	4;	/* tape buffer semaphore */
char	tm_done;	/* i/o done flag */

#define	TMADDR	0172520

#define RETRY	90
#define ETAPEMARK	EFBIG	/* use file big error for tape mark error */

/* commands (and tmcs bits) */
#define	NOCMD	00
#define	GO	01
#define	RCOM	02
#define	WCOM	04
#define	WEOF	06
#define	SFORW	010
#define	SREV	012
#define	WIRG	014
#define	REW	016
#define PWRCLR	010000
#define	DENS	060000		/* 9-channel */
#define	IENABLE	0100
#define	CRDY	0200
#define ERROR	0100000

/* status bits */
#define GAPSD	010000
#define	TUR	1
#define	HARD	0102200	/* ILC, EOT, NXM */
#define	EOF	0040000
#define	SELR	0100	/* select remote */
#define WRLOCK	04
#define	ILLCMD	0100000	/* illegal command error */

#define	SSEEKF	1
#define	SSEEKR	2
#define	SIO	3
#define SCOM	4


tmopen(dev, flag)
{
	register ds;
	register struct tminfo *mp;

	mp = &tminfo[dev & (NUNIT-1)];
	if (dev.d_minor & ~((NUNIT-1)|NOREW) || mp->t_openf) {
		u.u_error = ENXIO;
		return;
	}
	mp->t_openf++;
	mp->t_blkno = 0;
	/* -1 is later interpreted as 65535 */
	mp->t_tmblk = mp->t_rawtm = -1;
	mp->t_tmwritten = 0;
	ds = tcommand(dev, NOCMD);	/* clear controller */
	if ((ds&SELR) == 0 || flag && (ds&WRLOCK)) {
		mp->t_openf = 0;
		prdev("Offline or needs ring", dev);
		u.u_error = ENXIO;
		return;
	}
}

tmclose(dev, flag)
{
	register int rdev;
	register struct buf *bp;
	register struct tminfo *mp;

	rdev = dev;
	mp = &tminfo[rdev & (NUNIT-1)];
	/* because t_rawtm and t_blkno are updated asyncronously */
	/*  must wait for all i/o to complete */
	tcommand(rdev, NOCMD);

	if(flag && !mp->t_tmwritten) {
		if (mp->t_rawtm != mp->t_blkno) tcommand(rdev, WEOF);
		tcommand(rdev, WEOF);
		tcommand(rdev, rdev&NOREW? SREV: REW);
	} else {
		tcommand(rdev, !(rdev&NOREW)? REW:
		    mp->t_rawtm != mp->t_blkno? SFORW: NOCMD);
	}

	mp->t_openf = 0;
	/* de-associate all blocks */
	for (bp = tmtab.b_forw; bp != &tmtab; bp = bp->b_forw)
		if (bp->b_dev == rdev) bp->b_dev.d_minor = -1;
}

tcommand(dev, com)
{
	register struct buf *bp;

	bp = &ctmbuf;
	spl5();
	while(bp->b_flags & B_BUSY) {
		bp->b_flags =| B_WANTED;
		sleep(bp, PRIBIO);
	}
	spl0();
	bp->b_dev = dev;
	bp->b_resid = com;
	bp->b_blkno = 0;
	bp->b_flags = B_BUSY | B_READ;
	tmstrategy(bp);
	iowait(bp);
	if(bp->b_flags & B_WANTED)
		wakeup(bp);
	bp->b_flags = 0;
	return(bp->b_resid);
}

tmstrategy(abp)
struct buf *abp;
{
	register struct buf *bp;
	register struct tminfo *mp;
	register unit;

	bp = abp;
	unit = bp->b_dev&(NUNIT-1);
	mp = &tminfo[unit];

	if (bp == &ctmbuf) goto qup;	/* just queue up the spcl coms */
	if (mp->t_openf < 0) {
		/* fatal error */
		bp->b_flags =| B_ERROR;
		iodone(bp);
		return;
	}

	if (mp->t_tmblk <= bp->b_blkno) {
		if (mp->t_tmblk < bp->b_blkno) {
			bp->b_error = ETAPEMARK;
			bp->b_flags =| B_ERROR;
			iodone(bp);
			return;
		}
		if (bp->b_flags&B_READ) {
			/* this may be read-before-write */
#ifdef CPU70
			if (bp->b_flags&B_PHYS) bp->b_resid = bp->b_wcount;
			else
#endif CPU70
				clrbuf(bp);
			iodone(bp);
			return;
		}
	} else if(mp->t_tmblk!=-1 && (bp->b_flags&B_READ) && !mp->t_tmwritten){
		/* he is about to backspace after writing */
		/* write out two eof's now for him */
		tcommand(unit, WEOF);
		tcommand(unit, WEOF);
		tcommand(unit, SREV);
		tcommand(unit, SREV);
		mp->t_tmwritten++;
	}
	if ((bp->b_flags&B_READ)==0) {
		mp->t_tmblk = bp->b_blkno + 1;
		mp->t_tmwritten = 0;
	}
#ifdef CPU70
	if (bp->b_flags&B_PHYS)
		mapalloc(bp);
#endif CPU70

qup:
	bp->av_forw = 0;
	spl5();
	if (tmtab.d_actf==0)
		tmtab.d_actf = bp;
	else
		tmtab.d_actl->av_forw = bp;
	tmtab.d_actl = bp;
	if (tmtab.d_active==0)
		tmstart();
	spl0();
	sema_p(&tm_sem, PRIBIO);
	return;
}

tmstart()
{
	register struct buf *bp;
	register struct tminfo *mp;
	register com;

	tmtab.d_active = 0;
    loop:
	if ((bp = tmtab.d_actf) == 0)
		return;
	if (tm_done) {
		/* i/o done, chain to next */
		tmtab.d_actf = bp->av_forw;
		tmtab.d_errcnt = 0;
		tm_done = 0;
		iodone(bp);
		sema_v(&tm_sem);	/* release semaphore */
		goto loop;
	}
	mp = &tminfo[com = bp->b_dev & (NUNIT-1)];
	com = (com<<8) | ((bp->b_xmem & 03) << 4) | DENS;
	if(bp == &ctmbuf) {
		if(bp->b_resid == NOCMD) {
			/* don't want IENABLE bit or else */
			/* interrupt will occur immediately */
			TMADDR->tmcs = com;
			bp->b_resid = TMADDR->tmer;
			tm_done++;
			goto loop;
		}
		tmtab.d_active = SCOM;
		TMADDR->tmbc = 1;	/* i.e. infinity, or until eof */
		TMADDR->tmcs = bp->b_resid | com | IENABLE | GO;
		return;
	}
	if (mp->t_openf < 0 || (TMADDR->tmcs & CRDY)==0) {
		bp->b_flags =| B_ERROR;
		tm_done++;
		goto loop;
	}
	if (mp->t_blkno != bp->b_blkno) {
		if (mp->t_blkno < bp->b_blkno) {
			tmtab.d_active = SSEEKF;
			TMADDR->tmbc = mp->t_blkno - bp->b_blkno;
			mp->t_blkno =- TMADDR->tmbc;
			com =| SFORW|IENABLE|GO;
		} else {
			tmtab.d_active = SSEEKR;
			/* cannot do rewind even if b_blkno == 0 */
			/* because may not be in first file of tape */
			TMADDR->tmbc = bp->b_blkno - mp->t_blkno;
			/* extra backspace if error retry */
			if (tmtab.d_errcnt && (tmtab.d_errcnt&07) == 0 &&
			    bp->b_blkno != 0)
				TMADDR->tmbc--;
			mp->t_blkno =+ TMADDR->tmbc;
			com =| SREV|IENABLE|GO;
		}
		TMADDR->tmcs = com;
		return;
	}
	tmtab.d_active = SIO;
	TMADDR->tmbc = bp->b_wcount << 1;
	TMADDR->tmba = bp->b_addr;		/* core address */
	mp->t_blkno++;
	if (bp->b_flags&B_READ) {
		com =| IENABLE|RCOM|GO;	/* read */
	} else if (bp->b_wcount == -1) {
		com =| IENABLE|WEOF|GO;	/* write eof */
		mp->t_rawtm = mp->t_blkno;
	} else if (tmtab.d_errcnt) {
		com =| IENABLE|WIRG|GO;	/* write with 3 inch gap */
	} else {
		com =| IENABLE|WCOM|GO;	/* write */
		mp->t_rawtm = -1;
	}
	TMADDR->tmcs = com;
}

tmintr()
{
	register struct buf *bp;
	register unit;
	register struct tminfo *mp;

	if ((bp = tmtab.d_actf)==0)
		return;
	unit = bp->b_dev & (NUNIT-1);
	mp = &tminfo[unit];
	/* ignore errors on SCOM */
	if (tmtab.d_active == SCOM) {
		tm_done++;
		return(tmstart());
	}
	bp->b_resid = TMADDR->tmbc>>1;	/* word count wanted */
	if (TMADDR->tmcs < 0) {

	    /* if eof or error during seek, adjust t_blkno */
	    if (tmtab.d_active == SSEEKR)
	    	mp->t_blkno =- TMADDR->tmbc;
	    else if (tmtab.d_active == SSEEKF)
	    	mp->t_blkno =+ TMADDR->tmbc;

	    if (!(TMADDR->tmer&EOF)) {

		/* error */

		while(TMADDR->tmrd & GAPSD) ; /* wait for gap shutdown */
		if (TMADDR->tmer&HARD) {
			mp->t_openf = -1;
			prdev("Hard error", bp->b_dev);
		} else if (++tmtab.d_errcnt < RETRY) {
			return(tmstart());
		}

		deverror(bp, TMADDR->tmer, TMADDR->tmcs);
		bp->b_flags =| B_ERROR;
		tm_done++;
		return(tmstart());
	    } else {

		/* eof */

		if (bp != &rtmbuf || mp->t_rawtm+1 == mp->t_blkno &&
		    tmtab.d_active != SSEEKR) {
			/* cooked eof, or two in a row for raw */
			/* must set error or cooked read will not stop */
			bp->b_error = ETAPEMARK;
			bp->b_flags =| B_ERROR;
			/* backspace over eof */
			tmtab.d_active = SCOM;
			TMADDR->tmbc = -1;
			TMADDR->tmcs = (unit<<8)|SREV|DENS|IENABLE|GO;
			mp->t_blkno--;
			return;
		}
		/* single eof on raw returns as zero length record */
		bp->b_resid = bp->b_wcount;
		mp->t_rawtm = mp->t_blkno;
	    }
	}
	if (tmtab.d_active == SIO) tm_done++;
	return(tmstart());
}

tmread(dev)
{
	physio(tmstrategy, &rtmbuf, dev, B_READ);
	tmadjust();
}

tmwrite(dev)
{
	/* PHYSIO complains on count of zero, so */
	/* count of zero is set to 2, indicating write-tape-mark */
	/* count of 2 is set to 4 to avoid confusion and ridiculously */
	/*  small blocks */
	if (u.u_count < 4) u.u_count =+ 2;
	physio(tmstrategy, &rtmbuf, dev, B_WRITE);
	tmadjust();
}

tmadjust()
{
	/* fudge file offset to make all blocks appear to be 512 bytes */
	register struct file *fp;
	struct { long lng; };

	fp = getf(u.u_ar0[R0]);
	fp->f_offset->lng =+ 512 + u.u_count.integ - u.u_arg[1];
}

sema_p(sem, pri)
int *sem;
{
	register sps;

	sps = PS->integ;
	spl6();
	while (--(*sem) < 0) sleep(sem, pri);
	PS->integ = sps;
}

sema_v(sem)
int *sem;
{
	register sps;

	sps = PS->integ;
	spl6();
	if (++(*sem) <= 0) {
		wakeup(sem);
		*sem = 1;
	}
	PS->integ = sps;
}
