/*
 *	opr -- off line print via daemon 
 *
 *	modified by Kenj McDonell at University of Melbourne
 *	April 13, 1978.
 *
 *	Futher modified by Joseph Longo for compatibility with
 *	the new daemon.
 *	December, 1979.
 */

#include <stdio.h>
#include <signal.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include "local.h"
#include "spool.h"

#define NSIZE		(sizeof(SPOOLDIR)+DIRSIZ+2)	/* file path-name max. size */

char	tfname[NSIZE];
char	cfname[NSIZE];
char	lfname[NSIZE];
char	dfname[NSIZE];

#define	STRCPM(a, b)	strncpy(a, b, sizeof(a)-1)
#define	LASTCH(a)	a[sizeof(a)-1]

char	*strcpy(), *strncpy(), *strncat(), *strcat();
char	*getlogin();
FILE	*popen();
struct	passwd *getpwuid();
char	*ttyname();

/*
 *	files used by the spooling subsystem
 *
 *	tfzz.aaXXXXXX	temporary file of commands to the spooling daemon
 *	cfzz.a?XXXXXX	a copy of a file to be spooled
 *	lfzz.a?XXXXXX	a link to a file to be spooled
 *	dfzz.aaXXXXXX	the actual commands file passed to the daemon
 *		"XXXXXX" is the process id for the current
 *			invocation of opr
 *		"?" is 'a' for the first file argument, 'b' for the 
 *			second, 'c' for the third, etc.
 *			(the preceding 'a' is incremented when '?' > 'z')
 *		"zz" is the name of the printer requested (if any)
 */
struct nnam {
	char	*nptr;
	char	nlet;
} nnam[] = {
	{ tfname, 't' },
	{ cfname, 'c' },
	{ lfname, 'l' },
	{ dfname, 'd' },
	{ (char *)0, '\0' }
};

int	devflg	= 0;	/*  device modified flag  */
int	nact;		/*  number of files spooled for output  */
int	tff;		/*  file descriptor for "tfaXXXXXX"  */
char	person[20];	/*  login user name  */
char	workdir[500];	/*  `pwd` of invoker */
int	pid;		/*  process id  */
int	inchar;		/*  index to the character '?' in the file name strings  */
int	maxrec	= MAXCPY;	/* limit on the number of blocks to copied across */
int	debug;		/*  debug flag  */
long	t;		/* the time the spol job is submitted */
int	copies;		/*  number of copies  */
int	bcopies;	/*  number of banner pages per copy  */
int	bcflag;		/*  number of copies specified explicitly */
int	copyflg;	/*  copy file into spool directory flag  */
int	remflg;		/*  remove file after spooling flag  */
int	wrapflg;	/* wrap long lines on real printers */
int	perflag;	/* don't skip over the perforations (real printers) */
int	margin;		/* margin on real printers */
int	uncrunch;	/* don't crunch lines at top of page */
char	banner[10];	/*  banner heading  */
char	mesg[31];	/*  all done message  */
char	xbuf[30];	/*  extra buffer for general use */
char	device[CLSIZE] ;		/*  the device to be spooled to  */
int	maxchild;			/* just for 'merge()' */
struct	pr	prt[TABSIZE];
struct	pr	*lastpr = prt;
char	ifile[] = IFILE;
 
 
main(argc, argv)
int argc;
char *argv[];
{
	register char *arg;
	register char	*p;	/*  pointer to string following argument flag  */
	int out();
	char	*ctime();
	long	*time();

	register	f;
	struct stat	fstatb;		/*  buffer for "stat" call  */
	register i;
	PR pr;

	/*  initialize control variables and flags  */
	copies = 1;
	bcopies = 1;
	loginid();
	mesg[0] = '\0';
 
	/*  scan arguments looking for flags  */
	while (argc>1 && (arg = argv[1])[0]=='-') {
		p = arg+2;
		switch (arg[1]) {

		case 'b':
			/*  user defined banner  */
			if (*p == 0 && argc > 2) {
				p = argv[2];
				argv++;
				argc--;
			}
			STRCPM(banner, p);
			LASTCH(banner) = '\0';
			break;
 
		case 'c':
			/*  copy file, rather than linking to it  */
			copyflg = 1;
			break;

		case 'C':
			/* don't crunch lines at top of page */
			uncrunch = 1;
			break;

		case 'd':
			if (*p == 0 && argc > 2) {
				p = argv[2];
				argv++;
				argc--;
			}

			/* if the d option has already been used, forget it */

			if(devflg)
				break;

			STRCPM(device, p);
			LASTCH(device) = '\0';
			devflg = 1;
			break;

		case 'i':
			/* indent from left edge of page (margin) */
			margin++;
			break;

		case 'm':
			/*  write message after spooling  */
			if (*p == 0 && argc > 2) {
				p = argv[2];
				argv++;
				argc--;
			}
			STRCPM(mesg, p);
			LASTCH(mesg) = '\0';
			break;
 
		case 'n':
			/*  produce more than one copy  */
			if ((copies = atoi(p)) < 1) {
				printf ("Copies must be > 0\n");
				exit(1);
			}
			break;

		case 'p':
			/* don't skip over the perforations */
			perflag++;
			break;
 
		case 'r':
			/*  remove the file (if possible) after spooling  */
			remflg++;
			break;

		case 'w':
			/* wrap long lines */
			wrapflg++;
			break;

		case 'x':
			/*  define number of banner pages per copy  */
			bcopies = atoi(p);
			bcflag++;
			break;
 
		case 'D':
			/*  set debug flag  */
			debug = 1;
			break;
 
		default:
			/*  invalid flag  */
			printf ("Bad option: %s\n",arg);
			exit(1);
		}
		argc--;
		argv++;
	}
	if (remflg)
		getpwd();

	merge();
	i = f = 0;
	for (printers)
		if (thispr(pr)) {
			i++;
			if (pr->comm != '-')
				f++;
		}
	if (i == 0) {
		printf("Opr: no '%s' printers available\n", device);
		exit(1);
	}
	if (f == 0)
		printf("Opr: warning: no '%s' printers enabled\n", device);

	/*  build file names, with process id appended  */
	buildfn();
 
	/*  hangup, interrupt and quit signals are trapped by "out"  */
	if (signal(SIGHUP, SIG_IGN) != SIG_IGN)
		signal(SIGHUP, out);
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, out);
	if (signal(SIGQUIT, SIG_IGN) != SIG_IGN)
		signal(SIGQUIT, out);
	if (signal(SIGTERM, SIG_IGN) != SIG_IGN)
		signal(SIGTERM, out);
 
	/*  create "tfaXXXXXX"  */
	tff = nfile(tfname);

 
/*  debug  */
if (debug) {
	printf ("pid: %l\nperson: %s\nbanner: %s\n", pid, person, banner);
	printf ("mesg: %s\ncopies: %l\nremflg: %l\n", mesg, copies, remflg);
	printf ("copyflg: %l\n", copyflg);
	printf ("Wdir: %s\n", workdir);
	printf("bcopies: %l\nttyno: %s\n", bcopies, ttyname(2));
	printf( "spooled device: %s\n",device);
}
 
	/*
	 *  commands to the daemon
	 *
	 *	D		set debug flag
	 *	Lxxxx		login id = "xxxx"
	 *	Nnn		"nn" copies of each file
	 *	Bxxxx		banner heading = "xxxx"
	 *	Xxxxx		"xxxx" banner pages per copy
	 *	Mxxxx		send message "xxxx" when job done
	 *	Omw		options: m = margin, w = wrap
	 *	Rxxxx		spooled file's real name is "xxxx"
	 *	Sxxxx		"xxxx" is the time/date the job was submitted
	 *	Txxxx		job submitted from tty named "xxxx"
	 *	Fxxxx		print file "xxxx"
	 *	Uxxxx		unlink file "xxxx"
	 *	Wxxxx		working dir is "xxxx"
	 *
	 *		output the global commands (D, L, T, N, O, B, X, M, T)
	 */
	if (debug) card('D',"");
	if (remflg > 0)
		card('W', workdir);
	card('L', person);
	if ((p = ttyname(2)) == NULL)
		p = "";
	card('T', p);
	i = 0;
	if (margin) xbuf[i++] = 'm';
	if (wrapflg) xbuf[i++] = 'w';
	if (perflag) xbuf[i++] = 'p';
	if (uncrunch) xbuf[i++] = 'C';
	xbuf[i] = 0;
	if (i > 0)
		card('O', xbuf);
	if (copies > 1) {
		sprintf(xbuf, "%d", copies);
		card ('N', xbuf);
	}
	if (bcflag) {		/* If has overridden the default */
		sprintf(xbuf, "%d", bcopies);
		card ('X', xbuf);
	}
	if (banner[0] != '\0') card('B', banner);
	if (mesg[0] != '\0') card('M', mesg);
	time(&t);
	p = ctime(&t);
	STRCPM(xbuf, p);
	xbuf[strlen(xbuf)-1] = '\0';
	card ('S', xbuf);
 
	if(argc == 1)
		/*  no input files specified as arguments .. use standard input  */
		if (!copy(0))
			out();
	/*  pick up file names and process them one at a time  */
	while(--argc) {
		arg = *++argv;
 
		if (access(arg, 4) < 0) {
			printf("Opr: Can't read %s\n", arg);
			continue;
		}
		/*  skip empty files  */
		if (stat(arg, &fstatb) >= 0 && fstatb.st_size > 0) {

			if ((fstatb.st_mode&S_IFMT) != S_IFREG || unprintable(arg)) {
				printf("Opr: %s - unprintable\n", arg);
				continue;
			}
 
			/*  output real file name  */
			card('R', arg);
			if ( !copyflg && link(arg, lfname) >= 0) {
				/*  print via link, then unlink  */
				card('F', lfname);
				card('U', lfname);
				/*  ensure next link has a unique name  */
				incr(lfname);
				nact++;
			}
			else if (!copyflg && arg[0] == '/') {
				card('F', arg);
				nact++;
			}
			else {
				f = open(arg, 0);
				if(f < 0) {
					printf("Cannot open %s\n", arg);
					continue;
				}
				copy(f);
				close(f);
			}
		}
		/*  if "-r" specified, remove the original file  */
		if (remflg > 0 && access(arg, 2) >= 0)
			card('U', arg);
	}

	if(nact) {
		/*  
		 *  some files have been queued ....
		 *  rename tfaXXXXXX to dfaXXXXXX
		 *  then "fire up" the daemon
		 */
		decr(tfname);
		f = link(tfname, dfname);
		if(f < 0) {
			printf("Cannot rename %s\n", dfname);
			incr(tfname);
			out();
		}
		/*  inform user of spool job number  */
		printf ("Spool job no. %u\n", pid);
		unlink(tfname);
		execle(DAEMON, "LPD", 0, 0);
		/*  woops  */
		printf("Cannot execute %s\n", DAEMON);
		exit(1);
	}
	/*  exit here if no files were successfully queued  */
	out();
}

copy(f)
int f;
{
	/*  copy a file  */
	int ff, i, nr, nc, flg;
	static char buf[512];

	card('F', cfname);
	card('U', cfname);
	ff = nfile(cfname);
	nc = 0;
	nr = 0;
	flg = 0;
	while((i = read(f, buf, 512)) > 0) {
		write(ff, buf, i);
		flg = 1;
		nc += i;
		if(nc >= 512) {
			/*  another block  */
			nc -= 512;
			nr++;
			if(nr > maxrec) {
				/*  too many blocks in the copy file  */
				printf("Copy file is too large .. truncated after %l bytes\n", nr*512+nc);
				break;
			}
		}
	}
	close(ff);
	nact++;
	return(flg);
}

card(c, s)
int c;
char s[];
{
	static char buf[512];
	register len;

	buf[0] = c;
	buf[1] = '\0';
	strcat(buf, s);
	len = strlen(buf);
	buf[len] = '\n';
	write(tff, buf, len+1);
}

loginid()
{
	/*  extract the user's login id, and return it via 'person'  */
	register char *bp;
 
	if ((bp = getlogin()) == NULL && (bp = getpwuid(getuid())->pw_name) == NULL)
		bp = "Who ???";
	STRCPM(person, bp);
	LASTCH(person) = '\0';
}

getpwd()
{
	FILE *fd;
	register i;

	fd = popen("/bin/pwd", "r");
	if ((i = fread(workdir, sizeof(char), sizeof(workdir), fd)) <= 0
		|| i >= sizeof(workdir) || workdir[0] != '/') {
			remflg = -100;		/* prohibit '-r' flag */
			printf("Warning: opr can't get current working dir\n");
	}
	fclose(fd);
	workdir[i - 2] = '\0';
}

buildfn()
{
	/*  build file names, with process id appended  */
	register struct nnam *np;

	/*  get process id  */
	pid = getpid();

	for (np = nnam; np->nptr ; np++ )
		sprintf(np->nptr, "%s/%cf%s.aa%05d",
			SPOOLDIR, np->nlet, device, pid);

	inchar = strlen(nnam[0].nptr) - 6;
}

nfile(name)
char *name;
{
	register f;
	/*  create a new file  */

	umask(0);
	f = creat(name, 0666);
	if(f < 0) {
		printf("Cannot create %s\n", name);
		out();
	}
	/*
	 *  increment the 6th last character of the file name
	 *  -- this is required if multiple files are spooled in a single
	 *  execution of opr.
	 *  The resultant files names are of the form
	 *	.....aXXXXXX, .....bXXXXXX, .....cXXXXXX, .....dXXXXXX, etc
	 */
	incr(name);
	return(f);
}
 
 
out()
{
	/*  
	 *  come here following :-
	 *	a) no files successfully queued
	 *	b) a fatal error
	 *	c) a hangup, interrupt, or quit signal.
	 *
	 *  remove all the files
	 */
	register struct nnam *np;

	signal(SIGHUP, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);

	for (np = nnam; np->nptr; np++)
		while (!basename(np->nptr)) {
			decr(np->nptr);
			unlink(np->nptr);
		}
	exit(1);
}

/*
 * produce the next file name based on 'name'
 */
incr(name)
register char *name;
{
	if (name[inchar] >= 'z') {
		name[inchar] = 'a';
		name[inchar-1]++;
		return;
	}
	name[inchar]++;
}

/*
 * go backwards one name
 */
decr(name)
register char *name;
{
	if (name[inchar] <= 'a') {
		name[inchar] = 'z';
		name[inchar-1]--;
		return;
	}
	name[inchar]--;
}

/*
 * return true if this is the 'base' name
 */
basename(name)
register char *name;
{
	return(name[inchar]=='a' && name[inchar-1]=='a');
}

/*
 * return true if the file spec'd by 'arg' is unprintable
 *
 * (for now, just test the first word for one of the known magic numbers)
 */
unprintable(s)
char *s;
{
#include <a.out.h>
#include <ar.h>
	register f, l;
	int word;
	register *mp;
	static int magic[] = {
		A_MAGIC1,
		A_MAGIC2,
		A_MAGIC3,
		A_MAGIC4,
		A_MAGIC5,
		ARMAG,

		0
	};

	if ((f = open(s, 0)) < 0)
		return(0);
	l = read(f, &word, sizeof(word));
	close(f);
	if (l != sizeof(word))
		return(0);
	for (mp = magic; *mp; mp++)
		if (word == *mp)
			return(1);
	return(0);
}

