#
/*****************************************************************
 **								**
 **		   U C L A  Data Secure Unix			**
 **								**
 **			Copyright 1977				**
 **								**
 **    Mark Kampe, Charles Kline, Gerald Popek, Evelyn Walton	**
 **								**
 *****************************************************************/
#define true 0177777
#define false 000000
#define then  /* */

#include "../../shared/constants.h"
#include "../../shared/shared.h"
#include "../../shared/fscom.h"
#include "../seg.h"
#include "../caps.h"
#include "../conf.h"
#include "../param.h"

struct captab captab[_NUM_CLIST_ENTRIES];
struct fsmsg fsmsg;
/* name:
	getcap

function:
	to obtain a capability for the specified thing

algorithm:
	search the capability table
		if an adequate capability is found
			bump its reference count
			return its index
	find an empty slot (or failing that, an inactive one)
	get a capability for the desired thing
	if I could,
		return index of that slot
	else	return a -1

parameters:
	id0	major designation
	id1	minor designation
	type	object type
	access	desired access

returns:
	a capability index
	or a -1

globals:
	captab

calls:
	panic
	gdevcap		to get device capabilities
	glscap		to get long segment capabilities
	gsscap		to get short segment capabilities
	gpcap		to get a process capability
	setcap		to fill in a capability entry

called by:
	bfetch
	device open routines

history:
	designed and coded by Mark Kampe, 11/76
*/

getcap( id0 , id1 , type , access )
 int id0, id1, type, access;
{	register int capx;
	register int empty;
	register int inactive;
	int segid;
	static int roundrobbin;	/* used to lru cababilities */

	empty = -1;  inactive = -1;
	for( capx = 0; capx < _NUM_CLIST_ENTRIES; capx++)
	{	if (	(captab[capx].c_id[0] == id0) &&
			(captab[capx].c_id[1] == id1) &&
			(captab[capx].c_type == type))
		{	if ((captab[capx].c_access&access) == access) then
			{	captab[capx].c_refcount++;
				return( capx );
			}
			else
			{	access =| captab[capx].c_access;
				goto needgrant;
			}
		}
		if (captab[capx].c_type == NULL)
		then	empty = capx;
		else	if (captab[capx].c_refcount == 0)
			then	if (inactive < roundrobbin)
				then	inactive = capx;
	}
	/* empty is index of last unused entry */
	/* inactive is index of last inactive entry */
	if (empty == -1) then
	{	/* i could revoke the inactive capability, but the new grant
		   will do that automatically */
		capx = inactive;
		if (inactive < roundrobbin) /* no more inactives past here */
			roundrobbin = 0;
		else	roundrobbin = inactive + 1;
	}
	else	capx = empty;
	if (capx == -1)
		then	panic("captab overflow");

needgrant:	/* ask the file process for (augmented) access to the object */
	switch (type)
	{ case _DEVICE:		if ( gdevcap( capx , id0 , id1 , access ) )
				then	break;
				else	return( -1 );

	  case _LARGE_SEGMENT:	if ( segid = glsegcap(capx, id0, id1, access) )
				then
				{	if (id1 == 0)
					then	id1 = segid;
					break;
				}
				else	return( -1 );

	  case _SMALL_SEGMENT:	if ( gssegcap( capx , id0 , id1 , access ) )
				then	break;
				else	return( -1 );

	  case _PROCESS:	if ( gproccap( capx , id0 , id1 , access ) )
				then	break;
				else	return( -1 );

	  default:		return( -1 );
	}
	setcap( capx , id0 , id1 , type , access , captab[capx].c_refcount+1 );
	return( capx );
}
/* name:
	revokecap

function:
	to revoke a capability

algorithm:
	make sure that it is not in use
	format a revoke request to the file process
	ship it off

parameters:
	capx	capability index to be revoked

returns:
	true	done successfully
	false	capability not revoked

globals:
	captab
	fsmsg

calls:
	gcap

called by:
	getcap

history:
	Designed and Coded by Mark Kampe
*/
revokecap( capx )
 int capx;
{	register int cap;

	cap = capx;
	if (captab[ cap ].c_refcount)
		then return( false );

	fsmsg.fs_opcode = FS_REVOKE;
	fsmsg.fs_type = captab[ cap ].c_type;
	fsmsg.fs_id[0] = captab[ cap ].c_id[0];
	fsmsg.fs_id[1] = captab[ cap ].c_id[1];
	fsmsg.fs_cap0 = cap;
	fsmsg.fs_cap1 = -1;
	fsmsg.fs_pid = _proc_index;
	fsmsg.fs_access = _NULL_ACCESS;

	return( gcap() );
}

/* name:
	capinit

function:
	to initialized the capability table

algorithm:
	initialize the capability entry for:
		system input device
		system output device
		communication with the file process
		al initial segments

parameters:

returns:

globals:
	captab
	_ttyi_cap
	_ttyo_cap
	_fp_cap
	_segcap

calls:

called by:
	main

history:
	Designed and coded by Mark Kampe, 11/76
*/

capinit()
{	register int capx;
	int frame;
	register struct captab *cp;
	register struct frametab *fp;

	setcap( _fp_cap, 0, _fp_index, _PROCESS, _READ_WRITE_ACCESS, 1 );

	for(frame = 0; frame < 32; frame++)
	{	capx = _segcap[frame];
		if (capx < 0)
			then	continue;

		fp = &frametab[frame];
		cp = &captab[capx];
		if (cp->c_refcount)
		then
		{	cp->c_refcount++;
			cp->c_access =| (fp->f_modes&_FULL_ACCESS);
		}
		else	setcap( capx , fp->f_id[0] , fp->f_id[1] ,
				_LARGE_SEGMENT , fp->f_modes&_FULL_ACCESS , 1 );
	}
}
/* name:
	gdevcap
	gsscap
	glscap

function:
	to get a capability for a device
	to get a capability for a small segment
	to get a capability for a large segment

algorithm:
	format the message to the file process
	call gcap

parameters:
	capability index
	major designation
	minor designation
	access desired

returns:
	true	if it succeeded
	false	if it failed

globals:
	fsmsg	(buffer for messages to the file system process)

calls:
	gcap

called by:
	getcap

history:
	designed and coded by Mark Kampe, 11/76
*/
gdevcap( capx , id0 , id1 , access )
 int capx, id0, id1, access;
{
	fsmsg.fs_opcode = FS_GRANT;
	fsmsg.fs_type = _DEVICE;
	fsmsg.fs_cap0 = capx;
	fsmsg.fs_cap1 = -1;
	fsmsg.fs_pid = _proc_index;
	fsmsg.fs_access = access;
	fsmsg.fs_id[0] = id0;
	fsmsg.fs_id[1] = id1;

	return( gcap() );
}

gssegcap( capx , id0 , id1 , access )
 int capx, id0, id1, access;
{	fsmsg.fs_opcode = FS_GRANT;
	fsmsg.fs_type = _SMALL_SEGMENT;
	fsmsg.fs_cap0 = capx;
	fsmsg.fs_cap1 = id0;
	fsmsg.fs_pid = _proc_index;
	fsmsg.fs_access = access;
	fsmsg.fs_id[0] = 0;
	fsmsg.fs_id[1] = id1;

	return ( gcap() );
}

glsegcap( capx , id0 , id1 , access )
 int capx, id0, id1, access;
{	fsmsg.fs_opcode = FS_GRANT;
	fsmsg.fs_type = _LARGE_SEGMENT;
	fsmsg.fs_cap0 = capx;
	fsmsg.fs_cap1 = -1;
	fsmsg.fs_pid = _proc_index;
	fsmsg.fs_access = access;
	fsmsg.fs_id[0] = id0;
	fsmsg.fs_id[1] = id1;

	return( gcap() );
}

gproccap( capx , id0 , id1 , access )
 int capx, id0, id1, access;
{	return( false );
}
/* name:
	gcap

function:
	to send a request to the file system process and await a reply

algorithm:
	transmit the message
	await the reply
	if it worked, return true,
	else return false

parameters:

returns:
	true or false

globals:
	fsmsg	contains a fully constructed message

calls:
	k_send
	k_receive
	panic
	idle	
	bit_on, reset_bit

called by:
	gdevcap
	glsegcap
	gssegcap
	gproccap

history:
	Designed and coded by Mark Kampe, 11/76
*/
gcap()
{	register int retval;
	int msgarray[_MSG_TEXT_LENGTH];

	while( (retval = k_send(_fp_cap, _fp_index, &fsmsg, sizeof fsmsg))
		== 0)	idle();
	if (retval == -1)
		then	panic("FSPROC send");

	while( bit_on( comm_blk->c_nbits , _fp_index ) == false )
		idle();
	reset_bit( comm_blk->c_nbits, _fp_index );

	if (k_receive( _fp_cap , _fp_index , msgarray ) <= 0)
	then	panic("FSPROC receive");
	else	return( msgarray[0] );
}
/* name:
	setcap

function:
	to fill in a capability table entry

algorithm:
	fill in the fields from the parameters

parameters:
	capability index
	major designation
	minor designation
	object type
	desired access
	initial reference count

returns:

globals:
	captab

calls:

called by:
	lots of people

history:
	Designed and coded by Mark Kampe
*/

setcap( capx , id0 , id1 , type , access , refcount )
 int capx, id0, id1, type, access, refcount;
{	register struct captab *cp;

	cp = &captab[capx];
	cp->c_id[0] = id0;
	cp->c_id[1] = id1;
	cp->c_type = type;
	cp->c_access = access;
	cp->c_refcount = refcount;
}
