/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)PrintState.c	1.7 85/08/10
*/

/*
**	Print out State file from contents of sorted node list.
*/

#include	<stdio.h>

#include	"global.h"

#define	FLAG_DATA
#include	"state.h"

#include	"node.h"


static FILE *	PrStFd;
static bool	PrStHeader;
static bool	PrStVerbose;

static void	PrStNode();
void		PrStFlags();

#define	INDENT	(NODE_NAME_SIZE + 4)
#define	Fprintf	(void)fprintf



void
PrintState(fd, match, verbose)
	FILE *			fd;
	register States		match;
	bool			verbose;
{
	register Entry **	epp;
	register int		i;

	if ( match == (States)0 )
		return;

	PrStFd = fd;
	PrStHeader = (bool)(match==S_FOUND);
	PrStVerbose = verbose;

	if ( NodeList == (Entry **)0 )
		MakeList(NodeCount, &NodeList, NodeHash);

	if ( Home != (Entry *)0 && (Home->e_states & match) )
		PrStNode(Home);

	for ( epp = NodeList, i = NodeCount ; --i > 0 ; epp++ )
		if ( (*epp)->e_states & match )
			PrStNode(*epp);
}



static void
PrStNode(ep)
	Entry *		ep;
{
	register Node *	np;
	register Link *	lp;
	Time_t		date;
	extern Time_t	Time;

	if ( PrStHeader )
	{
		PrStHeader = false;

		if ( PrStVerbose )
		{
			register Domain *	dp;

			Fprintf(PrStFd, "ACSnet map for node \"%s\" (%d nodes):-\n", Home->e_name, NodeCount);

			if ( (dp = DomHier.d_head) != (Domain *)0 )
			{
				Fprintf(PrStFd, "\nLocal domain hierarchy: %s", dp->d_entry->e_name);

				while ( (dp = dp->d_next) != (Domain *)0 )
					Fprintf(PrStFd, ".%s", dp->d_entry->e_name);

				Fprintf(PrStFd, "\n\n");
			}
		}
	}

	Fprintf(PrStFd, "%-*s", INDENT, ep->e_name);

	np = ep->e_node;

	if ( PrStVerbose )
	{
		register Domain *	dp;
		register States		s;

		if ( np->n_hierarchy != NULLSTR )
			Fprintf(PrStFd, "{%s} ", np->n_hierarchy);

		if ( (dp = np->n_domains.d_head) != (Domain *)0 )
		{
			register bool	first = true;

			do
			{
				if ( first )
				{
					first = false;
					putc('[', PrStFd);
				}
				else
					putc('|', PrStFd);

				if ( dp->d_states & S_OTHDOM )
					putc('(', PrStFd);

				fputs(dp->d_entry->e_name, PrStFd);

				if ( dp->d_states & S_OTHDOM )
					putc(')', PrStFd);
			}
			while
				( (dp = dp->d_next) != (Domain *)0 );

			putc(']', PrStFd);
		}

		if ( s = (ep->e_states & NODE_FLAGS) )
			PrStFlags(PrStFd, s);

		if ( np->n_comment != NULLSTR )
			Fprintf(PrStFd, "\n%*s\"%s\"", INDENT, "", np->n_comment);

		if ( date = np->n_state )
		{
			if ( date > Time )	/* Catch bad date changes */
				date = Time;
			Fprintf(PrStFd, "\n%*slast state message %.12s", INDENT, "", ctime(&date)+4);
		}

#		if	NODE_STATS == 1
		if ( date = np->n_date )
		{
			if ( date > Time )	/* Catch bad date changes */
				date = Time;
			Fprintf(PrStFd, "\n%*slast active %.12s", INDENT, "", ctime(&date)+4);
		}
		if ( np->n_recvd )
			Fprintf(PrStFd, "\n%*sreceived %lu", INDENT, "", np->n_recvd);
		if ( np->n_sent )
		{
			if ( np->n_recvd )
				Fprintf(PrStFd, ", ");
			else
				Fprintf(PrStFd, "\n%*s", INDENT, "");

			Fprintf(PrStFd, "sent %lu", np->n_sent);
		}
		if ( np->n_passto )
			Fprintf(PrStFd, "\n%*s%lu transferred to", INDENT, "", np->n_passto);
		if ( np->n_passfrom )
		{
			if ( np->n_passto )
				Fprintf(PrStFd, ", ");
			else
				Fprintf(PrStFd, "\n%*s", INDENT, "");

			Fprintf(PrStFd, "%lu transferred from", np->n_passfrom);
		}
#		endif	NODE_STATS

		if ( np->n_handlers != (Hndl *)0 )
		{
			if ( np->n_handlers->h_spooler )
				Fprintf(PrStFd, "\n%*sspooler:\"%s\"", INDENT, "", np->n_handlers->h_spooler);
			if ( np->n_handlers->h_connector )
				Fprintf(PrStFd, "\n%*sconnector:\"%s\"", INDENT, "", np->n_handlers->h_connector);
			if ( np->n_handlers->h_caller )
				Fprintf(PrStFd, "\n%*scaller:\"%s\"", INDENT, "", np->n_handlers->h_caller);
		}
	}

	putc('\n', PrStFd);

	for ( lp = np->n_l_first ; lp != (Link *)0 ; lp = lp->l_next )
	{
		Fprintf(PrStFd, " -> %*s", NODE_NAME_SIZE, lp->l_entry->e_name);

		if ( PrStVerbose )
		{
			register Data *	dp = lp->l_data;
			register ulong	l;
			register States	s;

			if ( s = (dp->d_states & LINK_FLAGS) )
				PrStFlags(PrStFd, s);
			if ( l = dp->d_speed )
				Fprintf(PrStFd, " %lu rated bytes/sec.", l);
			if ( l = dp->d_cost )
				Fprintf(PrStFd, " %lu cost/Mb", l);

#			if	LINK_STATS == 1
			if
			(
				(l = dp->d_time) > 0
				&&
				(l = (dp->d_bytes + l - 1) / l) > 20
			)
				Fprintf(PrStFd, " %lu bytes/sec.", l);
#			endif	LINK_STATS
		}

		putc('\n', PrStFd);
	}
}



void
PrStFlags(fd, states)
	FILE *		fd;
	register States	states;
{
	register int	i;
	register int	bit;

	Fprintf(fd, " (");

	for ( i = 0, bit = 1 ; ; bit <<= 1, i++ )
		if ( states & bit )
		{
			Fprintf(fd, FlagNames[i]);

			if ( states &= ~bit )
				putc(',', fd);
			else
				break;
		}

	putc(')', fd);
}
