/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)receiver.c	1.34 85/08/31";

/*
**	Handle messages from node-node daemon or sender.
**
**	Messages are passed to a local handler if destination is reached,
**	otherwise they are spooled for further transmission.
*/

#define	FILE_CONTROL
#define	STAT_CALL
#define	RECOVER
#define	LOCKING

#include	"global.h"

#include	"address.h"
#include	"command.h"
#include	"debug.h"
#include	"header.h"
#include	"messagename.h"
#include	"spool.h"
#include	"state.h"

#include	<ndir.h>
#include	<signal.h>


/*
**	Parameters set from arguments.
*/

long	ComLength;		/* Length of message parts described in CommandsFile */
char *	CommandsFile;		/* Name of file containing commands for message parts */
char *	DmnDir;			/* Directory for router to operate from */
int	DmnPid;			/* Pid of NNdaemon */
ulong	ElapsedTime;		/* Journey time */
char *	HomeNode;		/* Name of this node */
char *	LinkNode;		/* Message has arrived from this node */
bool	Local;			/* Local invokation, return errors */
bool	MesgDup;		/* This message may be a duplicate */
char *	Message;		/* Current message being processed by receive() */
bool	NoCall;			/* Don't auto-call on this message */
char *	Name;			/* Program invoked name */
bool	Router;			/* Invoked as ROUTER */
int	Traceflag;		/* Global tracing control */

/*
**	Miscellaneous
*/

VarArgs	Args;			/* Global argument list for Exec...() */
int	Count;			/* Count used by spool1() */
bool	Explicit;		/* True if current message is using explicit routing */
NodeLink LinkD;			/* Info. from DoRoute() */
int	MesgFd;			/* File descriptor for message being delivered */
bool	NoOpt;			/* True if no message ordering required */
int	Pid;			/* Process id */
bool	StateMessage;		/* This message should have high priority in queue */
NodeLink SourceD;		/* Info. from FindAddress() */
char *	SpoolDir = SPOOLDIR();
char	Template[NNCFNZ+1];	/* Last component of node-node command file name */
Time_t	Time;			/* Current time */
ulong	Ttd;			/* Time-to-die for current message */


void	Return(), addresserr(), deliver(), finish(), freeArgs(),
	receive(), router(), spool1(), handlebad();



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	if ( strcmp(*argv, ROUTER) == STREQUAL )
		Router = true;

	DODEBUG(EchoArgs(argc, argv));

	(void)sprintf(Template, "%c%-*.*s", SMALL_ID, sizeof Template-2, sizeof Template-2, "router");

	Pid = getpid();

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				if ( Router )
				switch ( c )
				{
				case 'd':
					DmnDir = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 'p':
					DmnPid = atol(++*argv);
					break;

				default:
					Error("unrecognised flag '%c'", c);
					return 1;
				}
				else
				switch ( c )
				{
				case 'C':
					NoCall = true;
					continue;

				case 'D':
					MesgDup = true;
					continue;

				case 'L':
					Local = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					CommandsFile = ++*argv;
					goto break2;

				case 'd':
					ComLength = atol(++*argv);
					break;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 't':
					ElapsedTime = atol(++*argv);
					break;

				default:
					Error("unrecognised flag '%c'", c);
					return 1;
				}

				while ( (c = **argv) <= '9' && c >= '0' || c == '-' )	/* KLUDGE! */
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			if ( Router )
			{
				Error("Unexpected file name");
				return 1;
			}

			Time = time((long *)0);

			receive(*argv);

			if ( CommandsFile != NULLSTR )
			{
				(void)unlink(CommandsFile);
				CommandsFile = NULLSTR;
			}
		}
	}

	if ( Router )
	{
		if ( DmnDir == NULLSTR || LinkNode == NULLSTR || DmnPid == 0 )
		{
			Error("Need directory, link, and daemon's pid");
			return 1;
		}

		router();
	}
	else
	if ( Message == NULLSTR )
		Error("no spool file name");

	return 0;
}




/*
**	If "error" is non-zero, there has been an unrecoverable
**	system error and the daemon will hang pending operator intervention.
*/

void
finish(error)
	int	error;
{
	if ( Message != NULLSTR )
	{
		register int i;

		for (i = 0; i < 20; i++)
			UnLock(i);

		if ( Local )
			(void)unlink(Message);
	}

	if ( CommandsFile != NULLSTR )
		(void)unlink(CommandsFile);

	if ( DmnPid )
		(void)kill(DmnPid, SIGERROR);

	(void)exit(error);
}


/*
**	Scan receiving directory for messages, and route them.
*/

void
router()
{
	register DIR *		dirp;
	register struct direct *direp;

	while ( (dirp = opendir(DmnDir)) == NULL )
		Syserror("Can't read \"%s\"", DmnDir);

	if ( chdir(DmnDir) == SYSERROR )
		Syserror("Can't chdir \"%s\"", DmnDir);

	for ( ;; )
	{
		register int	count = 0;

		while ( (direp = readdir(dirp)) != NULL )
		{
			Parts		parts;
			struct stat	statb;

			if ( strlen(direp->d_name) != PNAMELEN )
				continue;

			count++;

			Time = time((long *)0);

			(void)stat(direp->d_name, &statb);

			(void)DecodeName(direp->d_name, &parts);

			ElapsedTime = parts.p_time + Time - statb.st_mtime;
			MesgDup = (parts.p_flag1 & MESG_DUP) ? true : false;

			receive(direp->d_name);
		}

		if ( count == 0 )
		{
			if ( kill(DmnPid, SIG0) == SYSERROR )
				break;

			(void)sleep(20);
		}

		rewinddir(dirp);
	}

	closedir(dirp);
}


/*
**	Process message.
*/

void
receive(message)
	char *		message;
{
	register char *	cp;
	HdrReason	reason;
	bool		expired;
	char *		handler;

	Message = message;

	Trace3(1, "Message \"%s\" received via \"%s\"", Message, LinkNode);

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();

	while ( (MesgFd = open(Message, O_RDWR)) == SYSERROR )
		Syserror("Can't open \"%s\"", Message);

	if ( (reason = ReadHeader(MesgFd)) != hr_ok )
	{
		/*
		**	Bad message header.
		**
		**	Almost no recovery possible, except to pass
		**	it to the badhandler for post-mortem.
		*/

		(void)close(MesgFd);

		handlebad
		(
			concat
			(
				"-eHeader ",
				HeaderReason(reason),
				" error",
				NULLSTR
			)
		);

		(void)unlink(Message);
		return;
	}

	if ( UpdateHeader(ElapsedTime, &Ttd) )
		expired = false;
	else
		expired = true;

	if ( HdrDest[0] == ATYP_EXPLICIT )
		Explicit = true;
	else
		Explicit = false;

	if ( MesgDup )
		HdrEnv = concat(HdrEnv, MakeEnv(ENV_DUP, HomeNode, NULLSTR), NULLSTR);

	/*
	**	If this node is in destination, pass message to handler.
	*/

	if ( !Local )
		Recover(ert_return);

	if ( (cp = GetEnv(ENV_STATE)) != NULLSTR )
	{
		free(cp);
		StateMessage = true;
	}
	else
		StateMessage = false;

	if ( HdrDest[0] == '\0' )
	{
		if ( Local )
			goto hdr_handler;
		else
			Return(newstr("Bad destination address format"));
	}
	else
	if ( DestReached(deliver) )
	{
hdr_handler:
		handler = concat(INSLIB(), HdrHandler, NULLSTR);
		deliver(handler, NULLSTR);
		free(handler);
	}

	(void)close(MesgFd);

	Trace2(1, "New destination \"%s\"", HdrDest);

	/*
	**	If the message has not expired,
	**	and there are more nodes in destination,
	**	spool message for each outbound link.
	*/

	if ( !expired && HdrDest[0] != '\0' )
	{
		extern char *	RouteBase;

		if ( (cp = GetEnv(ENV_NO_AUTOCALL)) != NULLSTR )
		{
			free(cp);
			NoCall = true;
		}
		else
			NoCall = false;

		if ( (cp = GetEnv(ENV_NOOPT)) != NULLSTR )
		{
			free(cp);
			NoOpt = true;
		}
		else
			NoOpt = false;

		if ( StateMessage && RouteBase != NULLSTR )
			(void)ReadRoute();	/* Re-load routing table */

		Count = 0;
		(void)DoRoute(HdrSource, LinkNode, &LinkD, spool1, addresserr);
	}

	Recover(ert_finish);

#	ifdef	DEBUG
	if ( Traceflag )
	{
		char *	tracemesg = UniqueName
				    (
					concat(TRACEDIR(), Template, NULLSTR),
					DataLength+ComLength,
					Time
				    );

		(void)link(Message, tracemesg);
		free(tracemesg);
	}
#	endif	DEBUG

	(void)unlink(Message);
}



void
deliver(handler, address)
	char *	handler;
	char *	address;
{
	char *	errs;

	Trace3(1, "deliver(handler \"%s\", address \"%s\")", handler, address);

	if ( StateMessage && Local )
		return;	/* Don't bother */

	if ( access(handler, 1) == SYSERROR )
	{
		/*
		**	Requested handler doesn't exist here.
		**
		**	Return message to source.
		*/

		Return(concat("Handler \"", handler, "\" unavailable", NULLSTR));
		return;
	}

	while ( WriteHeader(MesgFd, DataLength, HdrLength) == SYSERROR )
		Syserror("Can't write header in \"%s\"", Message);

	freeArgs();

	NEXTARG(&Args) = newstr(handler);
	if ( HdrDest[0] == ATYP_BROADCAST )
		NEXTARG(&Args) = newstr("-B");
	if ( Traceflag )
		NEXTARG(&Args) = newstr(Traceflag==1?"-T1":"-T2");
	if ( address != NULLSTR && address[0] != '\0' )
		NEXTARG(&Args) = concat("-a", address, NULLSTR);
	if ( CommandsFile != NULLSTR )
		NEXTARG(&Args) = concat("-c", CommandsFile, NULLSTR);
	NEXTARG(&Args) = Malloc(LENGTH_SIZE+2);
	(void)sprintf(LASTARG(&Args), "-d%ld", DataLength);
	NEXTARG(&Args) = concat("-e", HdrEnv, NULLSTR);
	NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
	NEXTARG(&Args) = concat("-l", LinkNode, NULLSTR);
	NEXTARG(&Args) = concat("-s", HdrSource, NULLSTR);
	NEXTARG(&Args) = concat("-t", HdrTt, NULLSTR);
	NEXTARG(&Args) = newstr(Message);

	if ( (errs = Execute(&Args)) != NULLSTR )
	{
		/*
		**	Handler failed.
		**
		**	Return message to source.
		*/

		Return(errs);
	}

	DODEBUG
	(
		if ( access(Message, 0) == SYSERROR )
			Syserror
			(
				"handler \"%s\" removed \"%s\"!",
				handler,
				Message
			)
	);
}



/*
**	Called once for each link with appropriate address set in 'HdrDest'.
**
**	'LinkD' contains the link details.
**
**	'Message' contains the message.
*/

void
spool1(source)
	char *		source;
{
	register int	fd;
	register char *	mesgfile;
	register Time_t	T;

	/*
	**	If this node is inderdicted, ignore it.
	*/

	if ( StateMessage && (LinkD.nl_flags & S_FOREIGN) )
		return;

	if ( (mesgfile = GetEnv(ENV_NOTNODE)) != NULLSTR )
	{
		if ( strccmp(mesgfile, LinkD.nl_name) == STREQUAL )
		{
			free(mesgfile);
			return;
		}

		free(mesgfile);
	}

	/*
	**	Messages from "local" nodes are restricted to
	**	the primary domain of the sender.
	*/

	if
	(
		!StateMessage
		&&
		(HomeFlags & S_LOCAL)
		&&
		HomeAddress(HdrSource, (char **)0, EXCL_HIER)
		&&
		!NodeinPrim(LinkD.nl_index)
	)
	{
		Return(concat("Can't send messages from \"local\" node \"", HdrSource, "\" off site", NULLSTR));
		return;
	}

	/*
	**	Messages from foreign nodes to foreign nodes
	**	are disallowed if we are a terminal node.
	*/

	if
	(
		!StateMessage
		&&
		(HomeFlags & S_MSGTERMINAL)
		&&
		!NodeinPrim(LinkD.nl_index)
		&&
		(
			!FindAddress(HdrSource, &SourceD)
			||
			!NodeinPrim(SourceD.nl_index)
		)
	)
	{
		Return(concat("Can't route foreign messages through \"terminal\" node \"", HdrSource, "\"", NULLSTR));
		return;
	}

	/*
	**	Make uniqueness a little easier
	*/

	T = Time - atol(HdrTt);
	if ( StateMessage )
		T /= 2;
	T += (Count+=2);

	Trace2(1, "Message passed to link \"%s\"", LinkD.nl_name);

	/*
	**	Make a link to Message
	*/

	mesgfile = UniqueName
		   (
			concat(WORKDIR(), Template, NULLSTR),
			DataLength+ComLength,
			T
		   );

	while ( link(Message, mesgfile) == SYSERROR )
		Syserror("Can't link \"%s\" to \"%s\"", Message, mesgfile);

	freeArgs();

	if ( LinkD.nl_spooler != NULLSTR )
	{
		char *	errs;
		char	ttime[TIME_SIZE+1];

		/*
		**	If (through routing changes) this message is about
		**	to return to a node that it has already visited,
		**	send it back to its source.
		*/

		if
		(
			strccmp(HdrDest, LinkD.nl_name) != STREQUAL
			&&
			(fd = InRoute(LinkD.nl_name)) > 0
			&&
			(!Explicit || fd > 3)
		)
		{
			(void)unlink(mesgfile);

			Return
			(
				concat
				(
					"Routing loop detected to node \"",
					LinkD.nl_name,
					"\"",
					NULLSTR
				)
			);

			return;
		}

		/*
		**	This node requires a non-standard spooler to be invoked.
		*/

		NEXTARG(&Args) = newstr(LinkD.nl_spooler);
		NEXTARG(&Args) = concat("-a", HdrDest, NULLSTR);
		if ( CommandsFile != NULLSTR )
			NEXTARG(&Args) = concat("-c", CommandsFile, NULLSTR);
		NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
		NEXTARG(&Args) = concat("-l", LinkD.nl_name, NULLSTR);
		NEXTARG(&Args) = concat("-s", source, NULLSTR);
		(void)sprintf(ttime, "%lu", ElapsedTime);
		NEXTARG(&Args) = concat("-t", ttime, NULLSTR);
		if ( Traceflag )
			NEXTARG(&Args) = newstr(Traceflag==1?"-T1":"-T2");
		NEXTARG(&Args) = mesgfile;

		if ( (errs = Execute(&Args)) != NULLSTR )
		{
			/*
			**	Spooler failed.
			**
			**	Return message to source.
			*/

			(void)unlink(mesgfile);	/* Because Args will be freed in "Return()" */

			Return(concat("Gateway handler failed :-\n", errs, NULLSTR));

			free(errs);
		}
		else
		{
			Trace4
			(
				1,
				"message passed to \"%s\" for \"%s\" via \"%s\"",
				LinkD.nl_spooler,
				HdrDest,
				LinkD.nl_name
			);

			UpStats(LinkD.nl_name);
		}
	}
	else
	{
		char *	headerfile;
		ComHead	comhead;
		char *	commands;
		char *	commandtemp;

		/*
		**	Make a new file for header
		*/

		headerfile = UniqueName
			     (
				concat(WORK2DIR(), Template, NULLSTR),
				DataLength+ComLength,
				T + 1
			     );

		while ( (fd = creat(headerfile, 0600)) == SYSERROR )
			Syserror("Can't create \"%s\"", headerfile);

		HdrSource = source;

		while ( WriteHeader(fd, (long)0, 0) == SYSERROR )
			Syserror("Can't write header in \"%s\"", headerfile);

		(void)close(fd);

		/*
		**	Make a command file for this message in
		**	appropriate command directory for node.
		*/

		commandtemp = UniqueName
			      (
				concat(WORK2DIR(), Template, NULLSTR),
				DataLength+ComLength,
				T
			      );

		/*
		**	4[5] commands: (see "command.h")
		**		send message; send header;
		**		unlink message; unlink header;
		**		[set timeout for message].
		*/
		
		FreeCom(&comhead, init_ch);

		AddCom(&comhead, mesgfile, (long)0, DataLength);
		AddCom(&comhead, headerfile, (long)0, (long)HdrLength);
		AddCom(&comhead, mesgfile, (long)0, (long)0);
		AddCom(&comhead, headerfile, (long)0, (long)0);

		if ( Ttd > 0 )
			AddCom(&comhead, NULLSTR, (long)Ttd, (long)0);

		while ( (fd = creat(commandtemp, 0600)) == SYSERROR )
			Syserror("Can't create \"%s\"", commandtemp);

		if ( CommandsFile != NULLSTR )
			CopyFile(CommandsFile, fd, commandtemp);

		(void)WriteCom(&comhead, fd, commandtemp);

		(void)close(fd);

		FreeCom(&comhead, free_ch);

		/*
		**	If (through routing changes) this message is about
		**	to return to a node that it has already visited,
		**	spool it in the holding directory.
		*/

		if
		(
			strccmp(HdrDest, LinkD.nl_name) != STREQUAL
			&&
			(fd = InRoute(LinkD.nl_name)) > 0
			&&
			(!Explicit || fd > 3)
		)
		{
			if ( fd == 1 && ReRoutable() )
			{
				commands = UniqueName
					   (
						concat(REROUTEDIR(), Template, NULLSTR),
						DataLength+ComLength,
						T
					   );

				while ( link(commandtemp, commands) == SYSERROR )
					Syserror
					(
						"Can't move \"%s\" to \"%s\"",
						commandtemp,
						commands
					);
				
				UpStats(LinkD.nl_name);
			}
			else
			{
				Return
				(
					concat
					(
						"Routing loop detected to node \"",
						LinkD.nl_name,
						"\"",
						NULLSTR
					)
				);

				commands = Malloc(1);
				(void)unlink(headerfile);
				(void)unlink(mesgfile);
			}

			(void)unlink(commandtemp);
		}
		else
		{
			char *	calldir = NULLSTR;

			/*
			**	Now move commands into daemon's de-spooling directory.
			*/

			commands = UniqueName
				   (
					concat(SpoolDir, LinkD.nl_name, "/", Template, NULLSTR),
					NoOpt
					? (long)MAX_MESG_DATA
					: StateMessage
					  ? (long)0
					  : (DataLength+ComLength+HdrLength),
					T
				   );

			while ( link(commandtemp, commands) == SYSERROR )
				Syserror("Can't move \"%s\" to \"%s\"", commandtemp, commands);

			(void)unlink(commandtemp);

			/*
			**	If link is "call-on-demand", start daemon.
			*/

			if
			(
				!NoCall
				&&
				(LinkD.nl_flags & S_CALL)
				&&
				!DaemonActive(calldir = concat(SpoolDir, LinkD.nl_name, NULLSTR), true)
			)
			{
				char *	callfile;

				callfile = concat(calldir, "/", CALLFILE, NULLSTR);

				if ( (fd = access(callfile, 04)) == SYSERROR && LinkD.nl_caller == NULLSTR )
				{
					Warn
					(
						"Can't access \"%s\" for call-on-demand link \"%s\"",
						callfile,
						LinkD.nl_name
					);

					free(callfile);
				}
				else
				{
					char *	errs;

					if ( LinkD.nl_caller != NULLSTR )
					{
						NEXTARG(&Args) = newstr(LinkD.nl_caller);
						if ( fd == SYSERROR )
						{
							free(callfile);
							callfile = NULLSTR;
						}
					}
					else
						NEXTARG(&Args) = newstr(NNCALL);
					if ( Traceflag )
					{
						NEXTARG(&Args) = newstr("-T0");
						LASTARG(&Args)[2] += Traceflag;
					}
					else
						NEXTARG(&Args) = newstr("-&");
					NEXTARG(&Args) = concat("-h", LinkD.nl_name, NULLSTR);
					NEXTARG(&Args) = callfile;

					if ( (errs = Execute(&Args)) != NULLSTR )
					{
						Warn(errs);
						free(errs);
					}
				}
			}
			
			if ( calldir != NULLSTR )
				free(calldir);

			UpStats(LinkD.nl_name);
		}

		Trace3
		(
			1,
			"message spooled in \"%s\" for \"%s\"",
			commands,
			HdrDest
		);

		free(commands);
		free(commandtemp);
		free(headerfile);
		free(mesgfile);
	}
}



/*
**	Pass message to BadHandler
*/

void
handlebad(arg)
	char *	arg;	/* free()'d */
{
	char *	badmesg;
	char *	errs;

	Trace1(1, "Message passed to badhandler");

	/*
	**	Move mesg to BADDIR
	*/

	badmesg = UniqueName
		  (
			concat(BADDIR(), Template, NULLSTR),
			DataLength+ComLength,
			Time
		  );

	while ( link(Message, badmesg) == SYSERROR )
		Syserror("Can't move \"%s\" to \"%s\", (error: \"%s\")", Message, badmesg, arg);

	freeArgs();
	NEXTARG(&Args) = newstr(BADHANDLER);
	NEXTARG(&Args) = arg;
	NEXTARG(&Args) = concat("-l", LinkNode, NULLSTR);
	NEXTARG(&Args) = concat("-i", Name, NULLSTR);
	NEXTARG(&Args) = concat("-h", HomeNode, NULLSTR);
	if ( Traceflag )
		NEXTARG(&Args) = newstr(Traceflag==1?"-T1":"-T2");

	if ( CommandsFile != NULLSTR )
	{
		char *	badcom;

		/*
		**	Move CommandsFile to BADDIR
		*/

		badcom = UniqueName
			 (
				concat(BADDIR(), Template, NULLSTR),
				DataLength+ComLength,
				Time
			 );

		while ( link(CommandsFile, badcom) == SYSERROR )
			Syserror("Can't move \"%s\" to \"%s\"", CommandsFile, badcom);

		NEXTARG(&Args) = concat("-d", CommandsFile, NULLSTR);
	}

	NEXTARG(&Args) = badmesg;

	if ( (errs = Execute(&Args)) != NULLSTR )
	{
		/*
		**	A total disaster!
		*/

		Error(errs);

		finish(1);
	}
}



/*
**	Unknown node encountered in destination/source address:
**	return message to source if possible.
**
**	"LinkD.nl_name" contains an error string set by DoRoute().
*/

void
addresserr(address, source)
	char *	address;
	bool	source;
{
	char *	reason;

	if ( source )
		reason = "Source address \"";
	else
		reason = "Destination \"";

	Return
	(
		concat
		(
			reason,
			address,
			"\" ",
			LinkD.nl_name,
			" at \"",
			HomeNode,
			"\"",
			NULLSTR
		)
	);
}



/*
**	Return the message to its source with explanation of error
*/

bool	returning;

void
Return(error)
	char *		error;	/* free()'d */
{
	char *		cp;
	char *		errors;
	Hdr_Fields	save_header;
	Time_t		ttd;

	Trace1(1, "Message returned to sender");

	errors = StripErrString(error);

	if ( strlen(errors) < 4 )
	{
		free(errors);

		if ( strlen(error) < 4 )
		{
			free(error);
			error = newstr("Handler failed, reason unknown");
		}

		errors = newstr(error);
	}

	/*
	**	If this is a local message, don't bother.
	*/

	if
	(
		Local
		||
		HomeAddress(HdrSource, (char **)0, EXCL_HIER)
	)
	{
		if ( Local )
			Error("Could not deliver message: %s", errors);
		else
			handlebad(concat("-eCould not deliver message: ", error, NULLSTR));

		free(errors);
		free(error);
		return;
	}

	/*
	**	Check no recursion, or returning unwanted.
	*/

	cp = NULLSTR;

	if
	(
		returning
		||
		(cp = GetEnv(ENV_RETURNED)) != NULLSTR
		||
		(cp = GetEnv(ENV_NORET)) != NULLSTR
	)
	{
#		if	LOG_RETURNED == 1
		handlebad(concat("-eReturned message discarded: ", error, NULLSTR));
#		endif	LOG_RETURNED == 1
		if ( cp != NULLSTR )
			free(cp);
		free(errors);
		free(error);
		return;
	}

#	if	LOG_RETURNED == 1
	/*
	**	Log the error.
	*/

	handlebad(concat("-eReturned message log: ", error, NULLSTR));
#	endif	LOG_RETURNED == 1

	free(error);

	returning = true;

	/*
	**	Save old header
	*/

	save_header = HdrFields;

	/*
	**	Set new header
	*/

	HdrEnv = MakeEnv
		 (
			ENV_RETURNED, NULLSTR,
			ENV_ERR1, errors,
			ENV_DESTINATION, HdrDest,
			ENV_ROUTE, HdrRoute,
			NULLSTR
		 );

	free(errors);

	HdrDest = HdrSource;
	HdrSource = HomeNode;

	HdrTtd = HdrTt = HdrRoute = "";

	(void)UpdateHeader((Time_t)0, &ttd);

	/*
	**	Spool it
	*/

	if ( !FindAddress(HdrDest, &LinkD) )
	{
		if ( FindAddress(LinkNode, &LinkD) )
		{
			char	sep[2];

			handlebad
			(
				concat
				(
					"-eDomain configuration problem?\nUnknown source \"",
					HdrDest,
					"\"\nMessage for \"",
					HdrDest,
					"\" returned to source via \"",
					LinkNode,
					"\"",
					NULLSTR
				)
			);

			if ( HdrDest[0] == ATYP_EXPLICIT )
				HdrDest++;

			sep[0] = ATYP_EXPLICIT;
			sep[1] = '\0';

			HdrDest = concat(sep, LinkNode, sep, HdrDest, NULLSTR);

			spool1(HdrSource);

			free(HdrDest);
		}
		else
			handlebad(concat("-eUnknown source \"", HdrDest, "\"", NULLSTR));
	}
	else
		spool1(HdrSource);

	/*
	**	Restore header
	*/

	free(HdrEnv);

	HdrFields = save_header;

	returning = false;
}



/*
**	Clear out Args (all must have been malloc'ed).
*/

void
freeArgs()
{
	register char *	cp;

	if ( NARGS(&Args) > MAXVARARGS )
		Fatal1("Too many VarArgs");
	
	while ( NARGS(&Args)-- )
		if ( (cp = ARG(&Args, NARGS(&Args))) != NULLSTR )
			free(cp);
	
	NARGS(&Args) = 0;
}
