/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)Rstate.c	1.15 84/10/17
*/

/*
**	Read a state file and structure it in memory.
*/

#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"state.h"

#include	"statefile.h"

#undef	Extern
#define	Extern
#include	"node.h"

#include	<setjmp.h>


static Crc_t	StCRC;

extern jmp_buf	NameErrJmp;
extern char *	HomeNode;

static Token	Rtoken();
static void	RsetFlags();



void
Rstate(fd, foreign, sourceaddr, sourcedate, ignoreCRC)
	FILE *		fd;
	bool		foreign;
	char *		sourceaddr;
	Time_t		sourcedate;
	bool		ignoreCRC;
{
	Entry *		nep;
	Entry *		oep;
	register Node *	np;
	register Data *	dp;
	register Node *	op;
	Token		accept;
	Token		expect;
	Entry *		source;
	int		tokens;
	char *		errs1;
	char *		errs2;
	Address *	address;
	char		temp[TOKEN_SIZE+1];

	Trace4(1, "Rstate %d %s %s", fileno(fd), foreign?"true":"false", sourceaddr==NULLSTR?"":sourceaddr);

	if ( setjmp(NameErrJmp) )
	{
		Error
		(
			"Bad %s nodename \"%s\"",
			foreign?"source":"home",
			foreign?sourceaddr:HomeNode
		);

		return;
	}

	if ( Home == (Entry *)0 )
	{
		Home = Enter(HomeNode, NodeHash);

		if ( !(Home->e_states & S_FOUND) )
		{
			Home->e_states |= S_FOUND;
			NodeCount++;
			ClearDomains(Home, true);
		}
	}

	if ( foreign )
	{
		register Link **lpp;

		address = SplitAddress(sourceaddr);

		if ( (source = Enter(address->ad_node, NodeHash)) == Home )
			return;

		if ( sourcedate > 0 )
		{
			if ( source->e_node->n_state >= sourcedate )
			{
				Report2("foreign statefile for \"%s\" date earlier than last", source->e_name);

				AnteState = true;
				return;
			}

			source->e_node->n_state = sourcedate;

#			if	NODE_STATS == 1
			if ( sourcedate > source->e_node->n_date )
				source->e_node->n_date = sourcedate;
#			endif	NODE_STATS == 1
		}

		ClearDomains(source, true);

		for ( lpp = &source->e_node->n_l_first ; *lpp != (Link *)0 ; )
			if ( (*lpp)->l_entry != Home )
				Unlink(source, lpp);
			else
				lpp = &(*lpp)->l_next;

		SetHierarchy(NULLSTR, &ForeignHier);

		if ( source->e_node->n_comment != NULLSTR )
		{
			free(source->e_node->n_comment);
			source->e_node->n_comment = NULLSTR;
		}
	}
	else
		SetHierarchy(NULLSTR, &DomHier);

	if ( setjmp(NameErrJmp) )
	{
		Error("Bad node name in Rstate: \"%s\"", temp);
		return;
	}

	StCRC = 0;
	tokens = 0;
	accept = t_any;
	expect = t_any;

	for ( np = (Node *)0 ; ; tokens++ )
	{
		switch ( Rtoken(fd, temp, accept, &expect) )
		{
		case t_node:
			nep = Enter(temp, NodeHash);
			np = nep->e_node;
			oep = (Entry *)0;
			dp = (Data *)0;

			Trace2(2, "Node \"%s\"", nep->e_name);

			if ( foreign )
			{
				if
				(
					nep == Home
					||
					(
						nep != source
						&&
						np->n_state	/* Had a statefile direct */
					)
				)
					accept = t_node;	/* Skip line */
				else
					accept = t_foreign;	/* Reject rude tokens */
			}
			else
				accept = t_any;

			if ( !(nep->e_states & S_FOUND) )
			{
				NodeCount++;
				nep->e_states |= S_FOUND;
			}

			continue;

		case t_link:
			dp = (Data *)0;

			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected link";
				errs2 = temp;
				break;
			}

			oep = Enter(temp, NodeHash);
			op = oep->e_node;

			Trace2(2, "Link \"%s\"", oep->e_name);

			if ( foreign && oep == Home )
				continue;

			if ( op == np )
			{
				errs1 = "Link to self for";
				errs2 = oep->e_name;
				break;
			}

			dp = MakeLink(nep, oep);

			if ( foreign && nep == source )
			{
				if ( !(oep->e_states & S_FOUND) )
				{
					NodeCount++;
					oep->e_states |= S_FOUND;
				}

				(void)MakeLink(oep, nep);

				dp->d_states &= ~FOREIGN_FLAGS;
				dp->d_cost = 0;
				dp->d_speed = 0;
			}

			continue;

		case t_alias:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected alias";
				errs2 = temp;
				break;
			}
			nep = Enter(temp, AliasHash);
			expect = t_aliasv;
			continue;

		case t_aliasv:
			if ( expect != t_aliasv )
			{
				errs1 = "Unexpected alias value";
				errs2 = temp;
				break;
			}
			expect = t_any;
			if ( nep->e_states & S_FOUND )
			{
				errs1 = "Duplicate alias value";
				errs2 = temp;
				break;
			}
			else
			{
				nep->e_value = newstr(temp);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			continue;

		case t_xalias:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected export alias";
				errs2 = temp;
				break;
			}
			nep = Enter(temp, AliasHash);
			if ( foreign )
			{
				if
				(
					(nep->e_states & S_FOUND) 
					||
					Lookup(temp, NodeHash) != (Entry *)0
					||
					Lookup(temp, DomainHash) != (Entry *)0
				)
					continue;	/* Reject if already exists */
				nep->e_value = newstr(sourceaddr);
				AliasCount++;
				nep->e_states |= S_FOUND;
			}
			else
			{
				if
				(
					!(nep->e_states & S_FOUND)
					||
					strcmp(nep->e_value, Home->e_name) != STREQUAL
				)
				{
					errs1 = "Bad export alias name";
					errs2 = temp;
					break;
				}
				XAlias = nep;
			}
			continue;
			
		case t_domain:
			if ( np == (Node *)0 )
			{
				errs1 = "Unexpected domain";
				errs2 = temp;
				break;
			}
			oep = AddDomain(temp, &np->n_domains, true);
			if ( nep == Home )
				oep->e_states |= S_DOMAINS;
			continue;

		case t_domhier:
			if ( np != (Node *)0 )
			{
				errs1 = "Unexpected domain hierarchy";
				errs2 = temp;
				break;
			}
			(void)AddDomain(temp, foreign?&ForeignHier:&DomHier, false);
			continue;

		case t_state:
			if ( np != (Node *)0 )
			{
				Time_t	t;

				if ( (t = atol(temp)) > np->n_state )
					np->n_state = t;
			}
			continue;

		case t_date:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
			{
				Time_t	t;

				if ( (t = atol(temp)) > np->n_date )
					np->n_date = t;
			}
#			endif	NODE_STATS
			continue;

		case t_recvd:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_recvd += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_sent:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_sent += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_passto:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_passto += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_passfrom:
#			if	NODE_STATS == 1
			if ( np != (Node *)0 )
				np->n_passfrom += (ulong)atol(temp);
#			endif	NODE_STATS
			continue;

		case t_time:
#			if	LINK_STATS == 1
			if ( dp != (Data *)0 )
				dp->d_time += (ulong)atol(temp);
#			endif	LINK_STATS
			continue;

		case t_bytes:
#			if	LINK_STATS == 1
			if ( dp != (Data *)0 )
				dp->d_bytes += (ulong)atol(temp);
#			endif	LINK_STATS
			continue;

		case t_speed:
			if ( foreign && nep != source )
				continue;
			if ( dp != (Data *)0 )
				dp->d_speed = (ulong)atol(temp);
			continue;

		case t_cost:
			if ( foreign && nep != source )
				continue;
			if ( dp != (Data *)0 )
				dp->d_cost = (ulong)atol(temp);
			continue;

		case t_nflags:
			if ( np != (Node *)0 )
				RsetFlags(&nep->e_states, temp, foreign);
			continue;

		case t_lflags:
			if ( dp != (Data *)0 )
				RsetFlags(&dp->d_states, temp, foreign);
			continue;

		case t_spooler:
			if ( np == (Node *)0 )
				continue;
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_spooler != NULLSTR )
				free(np->n_handlers->h_spooler);
			np->n_handlers->h_spooler = newstr(temp);
			continue;

		case t_connector:
			if ( np == (Node *)0 )
				continue;
			if ( np->n_handlers == (Hndl *)0 )
				np->n_handlers = Talloc(Hndl);
			else
			if ( np->n_handlers->h_connector != NULLSTR )
				free(np->n_handlers->h_connector);
			np->n_handlers->h_connector = newstr(temp);
			continue;

		case t_comment:
			if ( np == (Node *)0 )
				continue;
#			if	ALL_COMMENTS != 1
			if ( nep != Home )
				continue;
#			endif	ALL_COMMENTS != 1
			if ( np->n_comment != NULLSTR )
				free(np->n_comment);
			np->n_comment = newstr(temp);
			continue;

		case t_handler:
			if ( oep == (Entry *)0 || !(oep->e_states & S_DOMAINS) )
			{
				errs1 = "Unexpected domain handler";
				errs2 = temp;
				break;
			}
			if ( oep->e_handler != NULLSTR )
				free(oep->e_handler);
			oep->e_handler = newstr(temp);
			continue;

		case t_eof:
			goto out;

		default:
			Fatal1("Rstats: unknown token");
		}

		if ( foreign )
			Error("%s \"%s\"", errs1, errs2);
		else
			Warn("%s \"%s\" ignored", errs1, errs2);
	}

out:
	if
	(
		!ignoreCRC
		&&
		tokens > 0
		&&
		(
			getc(fd) != C_EOF
			||
			getc(fd) != LOCRC(StCRC)
			||
			getc(fd) != HICRC(StCRC)
			||
			getc(fd) != '\n'
		)
	)
	{
		Error("bad %sstatefile CRC", foreign?"foreign ":"");
		return;
	}

	if ( foreign )
	{
		if
		(
			source->e_node->n_domains.d_head != (Domain *)0
			&&
			ForeignHier.d_head == (Domain *)0
			&&
			address->ad_domains >= 2
		)
		{
			register int	i;

			while ( address->ad_domains > 0 )
				AddDomain
				(
					address->ad_strings[address->ad_domains--],
					&source->e_node->n_domains,
					true
				);

			SetHierarchy(strchr(sourceaddr, DOMAIN_SEP)+1, &ForeignHier);
		}

		FreeAddress(address);
		CheckHierarchy(source, &ForeignHier);
		CheckSourceDomains(source);
	}
}



/*
**	Read a separator followed by a token (name or number)
**	and return token type.
*/

static Token
Rtoken(fd, string, accept, expect)
	FILE *		fd;
	char *		string;
	Token		accept;
	Token *		expect;
{
	register char *	s;
	register int	n;
	register Token	token;
	register Token	new_token;
	register int	repeat_c;
	register Crc_t	last_crc;

	for ( n = 0, s = string, token = t_null, repeat_c = 0 ; ; )
	{
		register int	c;

		c = getc(fd);
		*s = c & 0177;

		StCRC = acrc((last_crc = StCRC), s, 1);

		switch ( c )
		{
		case C_ALIAS:		new_token = t_alias;	goto found;
		case C_ALIASV:		new_token = t_aliasv;	goto found;
		case C_BYTES:		new_token = t_bytes;	goto found;
		case C_COMMENT:		new_token = t_comment;	goto found;
		case C_CONNECTOR:	new_token = t_connector;goto found;
		case C_COST:		new_token = t_cost;	goto found;
		case C_DATE:		new_token = t_date;	goto found;
		case C_DOMAIN:		new_token = t_domain;	goto found;
		case C_DOMHIER:		new_token = t_domhier;	goto found;
		case C_HANDLER:		new_token = t_handler;	goto found;
		case C_LFLAGS:		new_token = t_lflags;	goto found;
		case C_LINK:		new_token = t_link;	goto found;
		case C_NFLAGS:		new_token = t_nflags;	goto found;
		case C_NODE:		new_token = t_node;	goto found;
		case C_PASSFROM:	new_token = t_passfrom;	goto found;
		case C_PASSTO:		new_token = t_passto;	goto found;
		case C_RECVD:		new_token = t_recvd;	goto found;
		case C_SENT:		new_token = t_sent;	goto found;
		case C_SPEED:		new_token = t_speed;	goto found;
		case C_SPOOLER:		new_token = t_spooler;	goto found;
		case C_STATE:		new_token = t_state;	goto found;
		case C_TIME:		new_token = t_time;	goto found;
		case C_XALIAS:		new_token = t_xalias;	goto found;
		case C_SPARE1:		new_token = t_ignore;	goto found;
		case C_SPARE2:		new_token = t_ignore;	goto found;
		case C_SPARE3:		new_token = t_ignore;	goto found;
		case C_SPARE7:		new_token = t_ignore;	goto found;
		case C_SPARE8:		new_token = t_ignore;	goto found;
		case C_SPARE9:		new_token = t_ignore;	goto found;

		case C_EOF:		(void)ungetc(*s, fd);
					StCRC = last_crc;
		case EOF:		if ( n == 0 || token == t_null )
						return t_eof;
					Trace1(1, "Rtoken at EOF!");
					goto out;
found:
			if ( n == 0 || token == t_null )
			{
				if ( repeat_c )
				{
					if ( *s != repeat_c )
						break;
					repeat_c = 0;
					token = t_null;
					continue;
				}

				if ( accept == t_foreign )
				{
					switch ( new_token )
					{
					case t_alias:
					case t_aliasv:
					case t_bytes:
					case t_connector:
					case t_date:
					case t_handler:
					case t_passfrom:
					case t_passto:
					case t_recvd:
					case t_sent:
					case t_spooler:
					case t_state:
					case t_time:
						token = t_null;
						continue;
					}
				}
				else
				if ( accept != t_any && new_token != accept )
				{
					token = t_null;
					continue;
				}

				switch ( token = new_token )
				{
				case t_ignore:
					token = t_null;
					continue;

				case t_comment:
				case t_connector:
				case t_handler:
				case t_spooler:
					repeat_c = *s;
				}

				s = string;
				n = 0;
				continue;
			}
			else
			{
				if ( repeat_c )
				{
					if ( *s != repeat_c )
						break;
				}
				else
				{
					(void)ungetc(c, fd);
					StCRC = last_crc;
				}

out:
				*s = '\0';
				Trace3(3, "Rtoken %d \"%s\"", (int)token, string);
				if ( (new_token = *expect) != t_any && token != new_token )
				{
					*expect = t_any;
					Warn("Read unexpected token %d value \"%s\" (expected %d)", token, string, new_token);
				}
				return token;
			}
		}

		if ( ++n <= TOKEN_SIZE )
			s++;
		else
			n = TOKEN_SIZE;
	}
}



/*
**	Decode state flags from string.
*/

static void
RsetFlags(sp, cp, foreign)
	register States *sp;
	register char *	cp;
	bool		foreign;
{
	register int	c;
	register States	accept;

	if ( foreign )
	{
		accept = FOREIGN_FLAGS;

		if ( *sp & S_LAN )
			accept &= ~S_DOWN;
	}
	else
		accept = EXTERNAL_FLAGS;

	*sp &= ~accept;
	accept |= S_DOWN;

	while ( c = *cp++ )
		*sp |= (1 << (c-'A')) & accept;
}
