/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
**
**	SCCSID @(#)children.c	1.6 84/10/12
*/

/*
**	Routines that handle child processes
*/

#define	STDIO

#include	"global.h"
#include	"command.h"
#include	"debug.h"
#include	"spool.h"

#define	BM_DATA
#include	"bad.h"
#include	"daemon.h"
#include	"Stream.h"
#include	"driver.h"
#include	"AQ.h"

#include	<setjmp.h>


extern jmp_buf	QTimeouts;


#define	MINWAIT		3	/* Minimum seconds between "wait"s */
#define	MAXWAIT		60	/* Maximum seconds between "wait"s */
#define	MAXPROCWAIT	(30*60)	/* Maximum wait time for a process */
#define	STATEWAIT	30	/* No need to be interested in the state-changer */

static Time_t	NSstarted;
static int	NSpid;
static bool	NewWait;
static Time_t	Started[NSTREAMS];

void	WaitError(), StatusError();



/*
**	Terminate reception, pass message to handler.
*/

void
EndMessage(chan)
	AQarg		chan;
{
	register Str_p	strp = &inStreams[(int)chan];
	char *		args[7];
	char		elapsedtime[12];

	Trace3(1, "EndMessage for channel %d state %d", (int)chan, strp->str_state);

	(void)fflush(stderr);

	switch ( strp->str_fd = fork() )
	{
	case SYSERROR:
		Report1("EndMessage cannot fork");
		qAction(EndMessage, chan);	/* Try again later */
		return;
	
	case 0:
		closeall();

		(void)sprintf(elapsedtime, "-t%lu", LastTime - strp->str_time);

		args[0] = MessageHandler;
		args[1] = elapsedtime;
		args[2] = concat("-l", LinkDir, NULLSTR);
		args[3] = concat("-h", HomeNode, NULLSTR);
#		ifdef	DEBUGPROGS
		if ( Traceflag )
		{
			args[4] = Traceflag==1?"-T1":"-T2";
			args[5] = strp->str_fname;
			args[6] = NULLSTR;
		}
		else
		{
#		endif	DEBUGPROGS
		args[4] = strp->str_fname;
		args[5] = NULLSTR;
#		ifdef	DEBUGPROGS
		}
#		endif	DEBUGPROGS

		for ( ;; )
		{
			execve(args[0], args, (char **)0);
			Syserror("cannot exec handler \"%s\"", args[0]);
		}
	}
	
	Started[(int)chan] = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+MINWAIT));
	}
	else
		NewWait = true;

	Trace2(1, "EndMessage started process %d", strp->str_fd);
}



/*
**	Wait for handler to terminate;
**	if found, send an EOM_ACCEPT.
*/

void
WaitHandler(delay)
	AQarg		delay;
{
	register Str_p	strp;
	register int	chan;
	register int	procs;
	auto int	pid;	/* Protect from "setjmp()" */
	int		status;

	Trace2(1, "WaitHandler %ld", (Time_t)delay - LastTime);

	if ( (Time_t)delay >= LastTime )
	{
		if ( NewWait )
		{
			NewWait = false;

			delay = (AQarg)(LastTime+MINWAIT);
		}

		qAction(WaitHandler, delay);
		return;
	}

	NewWait = false;

	for
	(
		strp = &inStreams[Fstream], chan = Fstream, procs = 0 ;
		chan < Nstreams ;
		chan++, strp++
	)
		if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
			procs++;
	
	if ( NSpid != 0 )
		procs++;
	else
	if ( procs == 0 )
	{
		Waiting = false;
		return;
	}

	do
	{
		pid = 0;

		if ( !setjmp(QTimeouts) )
		{
			ALARM_ON(2);

			pid = wait(&status);
			
			ALARM_OFF();

			if ( pid == SYSERROR )
			{
				char *	fmt;

				if ( (pid = NSpid) > 0 )
				{
					fmt = "newstatehandler (pid %d) disappeared!";
					NSpid = 0;
				}
				else
				{
					fmt = "handler (pid %d) for channel %d disappeared!";

					for
					(
						strp = &inStreams[Fstream], chan = Fstream ;
						chan < Nstreams ;
						chan++, strp++
					)
						if ( strp->str_state == STR_ENDING && (pid = strp->str_fd) > 0 )
						{
							strp->str_fd = 0;
							qAction(EndMessage, (AQarg)chan);
							break;
						}
				}

				Error(fmt, pid, chan);
				Waiting = false;
				return;
			}
		}
		else
		if ( pid == 0 )
		{
			/*
			**	Timed out, try later
			*/

			for
			(
				strp = &inStreams[Fstream], chan = Fstream, procs = MAXWAIT ;
				chan < Nstreams ;
				chan++, strp++
			)
				if ( strp->str_state == STR_ENDING && strp->str_fd > 0 )
				{
					if ( (pid = LastTime-Started[chan]) > MAXPROCWAIT )
					{
						Started[chan] = LastTime;
						WaitError(MessageHandler, pid, strp->str_fd, chan);
					}

					if ( pid < procs )
						procs = pid;
				}
			
			if ( NSpid != 0 )
			{
				if ( (pid = LastTime-NSstarted) > MAXPROCWAIT )
				{
					NSstarted = LastTime;
					WaitError(NewstateHandler, pid, NSpid, 0);
				}

				if ( pid < procs )
					procs = pid;
			}

			if ( procs < MINWAIT || (!Transmitting && !Receiving) )
				procs = MINWAIT;

			qAction(WaitHandler, (AQarg)(LastTime+procs));
			return;
		}

		Trace2(1, "WaitHandler found process %d", pid);

		if ( pid == NSpid )
		{
			NSpid = 0;

			if ( status )
				StatusError(NEWSTATEHANDLER, "", status);

			procs--;
			continue;
		}
		
		for
		(
			strp = &inStreams[Fstream], chan = Fstream ;
			chan < Nstreams ;
			chan++, strp++
		)
			if ( strp->str_state == STR_ENDING && pid == strp->str_fd )
			{
				strp->str_fd = 0;

				if ( status )
				{
					qAction(EndMessage, (AQarg)chan);

					StatusError(MessageHandler, strp->str_fname, status);

					procs--;
					break;
				}

				SndEOMA(chan);

				strp->str_messages++;
				NNstate.allmessages++;
				strp->str_bytes += strp->str_posn;

				Update(up_force);
				
				procs--;
				break;
			}
	}
		while ( procs > 0 );

	Waiting = false;
}



void
StatusError(prog, file, status)
	char *	prog;
	char *	file;
	int	status;
{
	Error
	(
		 "\"%s %s\" bad exit status:%d signal:%d%s"
		,prog
		,file
		,(status>>8)&0xff
		,status&0x7f
		,status&0x80?" (core dumped)":""
	);
}



void
WaitError(prog, wait, pid, chan)
	char *	prog;
	int	wait;
	int	pid;
	int	chan;
{
	Error
	(
		"overlong wait (%d secs.) for process %d, \"%s\" on channel %d",
		wait,
		pid,
		prog,
		chan
	);
}


/*
**	Run state update program
*/

void
RunState(newstate)
	AQarg		newstate;
{
	register int	i;
	char *		args[7];

	while ( NSpid != 0 )
		WaitHandler((AQarg)0);
	
	(void)fflush(stderr);

	switch ( NSpid = fork() )
	{
	case SYSERROR:
		NSpid = 0;
		Report1("RunState cannot fork");
		qAction(RunState, newstate);
		return;

	case 0:
		closeall();

		i = 0;
		args[i++] = NEWSTATEHANDLER;
		args[i++] = (int)newstate==LINK_DOWN?"-D":"-U";
		args[i++] = concat("-h", HomeNode, NULLSTR);
		if ( BatchMode )
			args[i++] = "-N";
#		ifdef	DEBUGPROGS
		if ( Traceflag )
			args[i++] = Traceflag==1?"-T1":"-T2";
#		endif	DEBUGPROGS
		args[i++] = LinkDir;
		args[i++] = NULLSTR;

		for ( ;; )
		{
			(void)execve(args[0], args, (char **)0);
			Syserror("can't execve \"%s\"", args[0]);
		}
	}

	NSstarted = LastTime;

	if ( !Waiting )
	{
		Waiting = true;
		qAction(WaitHandler, (AQarg)(LastTime+STATEWAIT));
	}
	else
		NewWait = true;

	Trace2(1, "RunState started process %d", NSpid);
}



/*
**	Move offending command file to bad directory,
**	and set bad message handler onto it.
**	Then reset the channel and queue NewMessage.
*/

void
BadMessage(chan, reason)
	AQarg		chan;
	BMReason	reason;
{
	register Str_p	strp = &outStreams[(int)chan];
	char *		mesg;

	Trace3(1, "BadMessage for channel %d state %d", (int)chan, strp->str_state);

	mesg = BMExplanations[(int)reason];

	Report3("Bad message \"%s\" - %s", strp->str_id, mesg);

	if ( reason != bm_nodata && access(strp->str_id, 0) == 0 )
	{
		char *	badname = concat(BADDIR(), strp->str_id, NULLSTR);
		char *	args[7];

		while
		(
			access(badname, 0) != 1
			&&
			link(strp->str_id, badname) == SYSERROR
		)
			Syserror("cannot link \"%s\" to \"%s\"", strp->str_id, badname);

		Report3("bad message \"%s\" moved to \"%s\"", strp->str_id, badname);

		/*
		**	'NewMessage' will unlink bad command file.
		*/

		(void)fflush(stderr);

		switch ( fork() )
		{
		case SYSERROR:
			Report1("BadMessage cannot fork");
			qAction(BadMessage, chan);		/* Try again */
			free(badname);
			return;

		case 0:
			closeall();

			args[0] = BadHandler;
			args[1] = concat("-d", badname, NULLSTR);
			args[2] = concat("-l", LinkDir, NULLSTR);
			args[3] = concat("-i", Name, NULLSTR);
			args[4] = concat("-h", HomeNode, NULLSTR);
			args[5] = concat("-e", mesg, NULLSTR);
			args[6] = NULLSTR;

			for ( ;; )
			{
				execve(BadHandler, args, (char **)0);
				Syserror("cannot exec handler \"%s\"", BadHandler);
			}
		}

		free(badname);
	}
	else
		Report2("bad message \"%s\" non-existent", strp->str_id);

	strp->str_state = STR_ERROR;
}
