/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "89/01/24 @(#)fileserver.c	1.19";

/*
**	Process request from remote site for a file from our system
**
**	SETUID -> root
*/

#define	MAXFILES	28		/* depends on MAXVARARGS */

#define	FILE_CONTROL
#define	LOCKING
#define	RECOVER
#define	STAT_CALL
#define	STDIO

#include	"global.h"

#include	"Passwd.h"
#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"spool.h"
#include	"ndir.h"
#ifdef	FILESERVERLOG
#include	"stats.h"
#endif

#include	<errno.h>

#if	AUSAS == 0
#include	<grp.h>
#endif

/*
**	Parameters set from arguments.
*/

bool	Broadcast;		/* Message has been everywhere */
char *	CommandsFile;		/* Commands of files to be pre-pended to Message */
char *	HomeNode;		/* Name of this node */
char *	LinkNode;		/* Message arrived from this node */
char *	Message;		/* Message */
char *	Name;			/* Program invoked name */
char *	SourceNode;		/* Message originated at this node */
int	Traceflag;		/* Global tracing control */

/*
**	Parameters read from header of file
*/

char *	Options;		/* Service related options */
char *	RemUser;		/* Name of user requesting service */
char *	Service;		/* Service requested */

#ifdef	PUBLICFILES
/*
**	The services offered
*/

int	sendfiles(), sendlist();

typedef struct
{
	char *	name;
	int	(*server)();
	bool	param;
}
	ServiceType;

ServiceType	Serves[] =
{
	{	"SendFile",	sendfiles,	false	},
	{	"List",		sendlist,	false	},
	{	"ListVerbose",	sendlist,	true	},
	{	NULLSTR					}
};
#endif	PUBLICFILES

/*
**	Miscellaneous
*/

char *	DataName;		/* Used by ExpandArgs() */
char *	FileList[MAXFILES];	/* List of files requested */
long	FileSizes;		/* Total size of all files requested */
char *	Home_Address;		/* Address of this node */
int	Pid;			/* Used by UniqueName() in Exec...() */
char *	PublicDir;		/* Place where files are stored */
int	RetVal;			/* Value to return to receiver */
char *	SenderName;		/* Used by ExpandArgs() */
Time_t	StartTime;		/* Used by ExpandArgs() */
Time_t	Time;			/* Used by UniqueName() in Exec...() */
char *	UserName;		/* Used by ExpandArgs() */

#ifdef	PUBLICFILES
char *	PublicFiles = PUBLICFILES;	/* where we find about public files */
#endif

#ifdef	FILESERVERLOG
char *	FileServerLog	= FILESERVERLOG;
char *	Errors[]	= {"ERROR", NULLSTR, NULLSTR};
#define	Errorstr	Errors[1]
void			DoError();
#else	FILESERVERLOG
#define	DoError		Error
#endif	FILESERVERLOG

#define	MINCOL		16
#define	MAXCOL		32
#define	LINELEN		80

#define	Fprintf		(void)fprintf
#define	free(A)

void	finish(), DoRequest(), MailBack(), Return(), GetFSHdr();
void	flist(), vlist(), blist(), writestats();
char	*FileStr(), *CheckPerm();

#if	AUSAS == 0
struct group *	getgrnam();
struct passwd *	getpwnam();
#endif



int
main(argc, argv)
	register int	argc;
	register char *	argv[];
{
	register FILE *	fd;
	char *		cp;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	Pid = getpid();
	Time = time((long *)0);
	DataName = SenderName = UserName = "fileserver";

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'B':
					Broadcast = true;
					continue;

				case 'T':
					if ( (Traceflag = atol(++*argv)) == 0 )
						Traceflag = 1;
					break;

				case 'c':
					CommandsFile = ++*argv;
					goto break2;

				case 'd':
					DataLength = atol(++*argv);
					break;

				case 'e':
					HdrEnv = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'l':
					LinkNode = ++*argv;
					goto break2;

				case 's':
					SourceNode = ++*argv;
					goto break2;

				default:
					Trace2(1, "unrecognised flag '%c'", c);
					goto break2;
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		if ( Message != NULLSTR )
			DoError("only one message allowed");
		else
			Message = *argv;
	}

	Home_Address = concat(HomeNode, Hierarchy(), NULLSTR);

	if ( (cp = GetEnv(ENV_ACK)) != NULLSTR )
	{
		free(cp);
		DoError("file request acknowledged from \"%s\"", SourceNode);
	}

	while ( (fd = fopen(Message, "r")) == NULL )
		Syserror("Can't open \"%s\"", Message);

	Recover(ert_finish);

	if ( (cp = GetEnv(ENV_RETURNED)) != NULLSTR )
	{
		free(cp);
		MailBack(GetEnv(ENV_ERR1), fd);
	}
	else
	{
		DoRequest(fd);

#		ifdef	FILESERVERLOG
		if ( RetVal )
			writestats(Errors);
#		endif	FILESERVERLOG
	}

	exit(RetVal);
}



/*
**	Called from the error routines to cleanup.
*/

void
finish(error)
	int	error;
{
#	ifdef	FILESERVERLOG
	writestats(Errors);
#	endif	FILESERVERLOG
	(void)exit(error);
}




/*
**	Check validity of file request, and send files back.
*/

void
DoRequest(ifd)
	FILE *			ifd;
{
	register char *		cp;
	register char * 	dp;
	register char **	fp;
#	ifdef	PUBLICFILES
	register ServiceType *	sp;
#	endif	PUBLICFILES
	FILE *			fd;
	struct stat		statb;

	GetFSHdr(ifd);

#	ifndef	PUBLICFILES

	DoError("There are no remotely accessible files at %s.  Sorry.", Home_Address);

#	else	PUBLICFILES

	if ( stat(PublicFiles, &statb) == SYSERROR )
	{
		DoError("No public files are currently available.");
	}

	switch ( statb.st_mode & S_IFMT )
	{
	default:
		DoError("%s is a weird file.  Nothing available", PublicFiles);

	case S_IFDIR:
		PublicDir = PublicFiles;
		break;

	case S_IFREG:
		/*
		** This can't fail - file exists, and we are root ..
		*/
		if ( (fd = fopen(PublicFiles, "r")) == NULL )
		{
			DoError("Public files are private!");
		}

		PublicDir = CheckPerm(fd);

		(void)fclose(fd);
		break;
	}

#	if	AUSAS == 0
#	ifdef	SERVERGROUP
	{
		register struct group * gp;

		if ( (gp = getgrnam(SERVERGROUP)) == (struct group *)0 )
			DoError("Group %s does not exist!", SERVERGROUP);

		(void)setgid(gp->gr_gid);
	}
#	else	SERVERGROUP
	(void)setgid(ACSNETGID);
#	endif	SERVERGROUP
#	endif	AUSAS == 0

#	ifdef	SERVERUSER
	{
		Passwd	pw;

		if ( !GetUid(&pw, SERVERUSER) )
			DoError("User %s does not exist!", SERVERUSER);

#		if	AUSAS == 1
		(void)setgid(pw.P_gid);
#		endif

		(void)setuid(pw.P_uid);
	}
#	else	SERVERUSER
	(void)setuid(ACSNETUID);
#	endif	SERVERUSER

	if ( chdir(PublicDir) == SYSERROR )
		DoError("Public files are inaccessible");

	Recover(ert_return);

	fp = FileList;

	while ( fp < &FileList[MAXFILES-1] )
	{
		if ( (cp = FileStr(ifd)) == NULLSTR )
			break;

		if ( *cp == '/' )
		{
			DoError("%s of %s refused: absolute path illegal", Service, cp);
			RetVal = 1;
			continue;
		}

		if ( strncmp(cp, "../", 3) == STREQUAL || strcmp(cp, "..") == STREQUAL )
		{
			DoError("Values of B will give rise to Dom <%s>", cp);
			RetVal = 1;
			continue;
		}

		for ( dp = cp + 1 ; (dp = strchr(dp, '.')) != NULLSTR ; dp++ )
		{
			if ( strncmp(dp-1, "/../", 4) == STREQUAL )
				break;
			if ( strcmp(dp-1, "/..") == STREQUAL )
				break;
		}

		if ( dp != NULLSTR )
		{
			DoError("Illegal \"..\" in filename <%s>", cp);
			RetVal = 1;
			continue;
		}

		if ( access(cp, 4) == SYSERROR || stat(cp, &statb) == SYSERROR )
		{
			DoError("%s of %s refused: unavailable", Service, cp);
			RetVal = 1;
			continue;
		}

		if ( (statb.st_mode & S_IFMT) == S_IFDIR && strcmp(Service, "SendFile") == STREQUAL )
		{
			DoError("%s is a directory: not returned", cp);
			RetVal = 1;
			continue;
		}

#		ifdef	FILESERVERLOG
		FileSizes += statb.st_size;
#		endif	FILESERVERLOG

		*fp++ = cp;
	}

	if ( fp == FileList )
	{
		if ( RetVal == 0 )
			DoError("No files requested.");
		RetVal = 1;
		return;
	}

	*fp++ = NULLSTR;

	if ( cp != NULLSTR )
	{
		Warn("Too many files requested, excess requests ignored");
		RetVal = 1;
	}

	for ( sp = Serves ; (cp = sp->name) != NULLSTR ; sp++ )
		if ( strcmp(Service, cp) == STREQUAL )
		{
			(void)(*sp->server)(sp->param);
#			ifdef	FILESERVERLOG
			writestats(FileList);
#			endif	FILESERVERLOG
			return;
		}

	DoError("Service \"%s\" is not implemented at %s", Service, Home_Address);
	RetVal = 1;
#	endif	PUBLICFILES
}



#ifdef	PUBLICFILES
int
sendfiles(param)
	bool		param;
{
	register char **list = FileList;
	char *		errs;
	VarArgs		va;

	Trace3(1, "sendfiles(%s) \"%s\" ...", param?"true":"false", *list);

	FIRSTARG(&va) = SEND;
	NEXTARG(&va) = concat("-q", RemUser, "@", SourceNode, NULLSTR);
	if ( Options != NULLSTR && strchr(Options, 'C') != NULLSTR )
		NEXTARG(&va) = "-C";
	if ( Traceflag != 0 )
		NEXTARG(&va) = NumericArg('T', (long)Traceflag);

	while ( *list != NULLSTR )
	{
		NEXTARG(&va) = *list;
		list++;
	}

	if ( (errs = Execute(&va)) != NULLSTR )
	{
		DoError("Sendfile said '%s'", errs);
		RetVal = 1;
	}

	return 0;
}



int
sendlist(verbose)
	register bool	verbose;
{
	register char **list = FileList;
	register FILE *	sfd;
	register char *	cp;
	char *		errs;
	VarArgs		va;

	Trace3(1, "sendlist(%s) \"%s\" ...", verbose?"true":"false", *list);

	FIRSTARG(&va) = SEND;
	NEXTARG(&va) = concat("-q", RemUser, "@", SourceNode, NULLSTR);
	if ( Options != NULLSTR && strchr(Options, 'C') != NULLSTR )
		NEXTARG(&va) = "-C";
	if ( Traceflag != 0 )
		NEXTARG(&va) = NumericArg('T', (long)Traceflag);
	NEXTARG(&va) = "-nPublicFileList";

	sfd = ExecPipe(&va);

	Fprintf(sfd, "Results of %slist request at %s on %s\n",
		(verbose ? "verbose " : ""), Home_Address, ctime(&Time));

	Fprintf(sfd, "%-9s  %-42s  %10s  %s\n\n",
		"Type", "Name", "Size", "Date");

#	ifdef	FILESERVERLOG
	FileSizes = 2*LINELEN;
#	endif	FILESERVERLOG

	while ( (cp = *list++) != NULLSTR )
		vlist(cp, sfd, verbose, 0);

	if ( (errs = ExPipeClose(sfd)) != NULLSTR )
	{
		DoError("SendFile said '%s'", errs);
		RetVal = 1;
	}

	return 0;
}



void
vlist(file, fd, recurs, level)
	register char *		file;
	register FILE *		fd;
	bool			recurs;
	int			level;
{
	register DIR *		dd;
	register struct direct *dp;
	register char *		cp;
	register int		i;
	char			datebuf[13];
	struct stat		statb;

	if ( stat(file, &statb) == SYSERROR )
	{
		Fprintf(fd, "%s: nonexistent\n", file);
		return;
	}

	/*
	** Note: we expressly do not tell owner, link count, permissions ..
	*/

	switch ( statb.st_mode & S_IFMT )
	{
	case S_IFREG:
		cp = "file";
		break;

	case S_IFDIR:
		cp = "directory";
		break;

	default:
		cp = "special";
		break;
	}

	Fprintf
	(
		fd,
		"%-9s  %-42s  %10ld  %s\n",
		cp, file, statb.st_size,
		DateString(statb.st_mtime, datebuf)
	);

#	ifdef	FILESERVERLOG
	FileSizes += LINELEN;
#	endif	FILESERVERLOG

	if ( (statb.st_mode & S_IFMT) != S_IFDIR )
		return;

	if ( (dd = opendir(file)) == NULL )
	{
		Fprintf(fd, "\nContents unavailable\n\n");
		return;
	}

	for ( i = 0 ; (dp = readdir(dd)) != NULL ; )
	{
		if ( dp->d_name[0] == '.' )
			continue;

		if ( recurs && level < 5 )
		{
			vlist((cp = concat(file, "/", dp->d_name, NULLSTR)), fd, recurs, level+1);
			free(cp);
		}
		else
		{
			if ( i == 0 )
				Fprintf(fd, "\nContents:\n");
			flist(dp->d_name, fd);
		}

		i++;
	}

	if ( i )
	{
		flist(NULLSTR, fd);
		(void)putc('\n', fd);
	}

	closedir(dd);
}



void
flist(file, fd)
	register char *	file;
	register FILE *	fd;
{
	static int	linelen = 0;
	register int	len;

	if ( file == NULLSTR )
	{
		if ( linelen > 0 )
		{
			(void)putc('\n', fd);
#			ifdef	FILESERVERLOG
			FileSizes += LINELEN;
#			endif	FILESERVERLOG
		}
		linelen = 0;
		return;
	}

	if ( (len = strlen(file)) < MINCOL )
		len = MINCOL;
	else
	if ( len < MAXCOL )
		len = MAXCOL;
	else
		linelen = LINELEN;

	if ( (linelen += len) > LINELEN )
	{
		(void)putc('\n', fd);
		linelen = len;
#		ifdef	FILESERVERLOG
		FileSizes += LINELEN;
#		endif	FILESERVERLOG
	}
	else
	if ( linelen == LINELEN )
		len = 1;

	Fprintf(fd, "%-*s", len, file);
}



char *
CheckPerm(fd)
	register FILE *	fd;
{
	register char *	cp;
	register char *	hp;
	char		buf[BUFSIZ];

	while ( fgets(buf, sizeof buf, fd) != NULLSTR )
	{
		if ( (cp = strchr(buf, '\n')) != NULLSTR )
			*cp = '\0';

		if ( cp = strpbrk(buf, " \t") )
		{
			*cp++ = '\0';
			cp += strspn(cp, " \t");
		}

		if ( hp = strchr(buf, '@') )
			*hp++ = '\0';
		else
			hp = buf;

		if ( *hp != '*' && strccmp(hp, SourceNode) != STREQUAL )
			continue;

		if ( hp != buf && buf[0] != '*' && strcmp(buf, RemUser) != STREQUAL )
			continue;

		if ( cp == NULLSTR )
			break;

		Trace2(1, "CheckPerm resultis \"%s\"", cp);

		return newstr(cp);
	}

	DoError("You have no access permissions at %s", Home_Address);

	/*NOTREACHED*/
}
#endif	PUBLICFILES



void
GetFSHdr(ifd)
	FILE	* ifd;
{
	RemUser = FileStr(ifd);
	Service = FileStr(ifd);
	Options = FileStr(ifd);

	if ( Service == NULLSTR || RemUser == NULLSTR || getc(ifd) != '\0' )
		DoError("File request header bad");

	Trace4(1, "GetFSHdr: RemUser=\"%s\", Service=\"%s\", Options=\"%s\"", RemUser, Service, Options);
}



char *
FileStr(fd)
	register FILE * fd;
{
	register	c;
	register char * cp;
	char		string[128];

	for ( cp = string ; cp < &string[sizeof string] ; cp++ )
	{
		if ( (c = getc(fd)) == EOF )
			return NULLSTR;

		*cp = c;

		if ( c == '\0' )
		{
			if ( cp == string )
				return NULLSTR;

			return newstr(string);
		}
	}

	return NULLSTR;
}



void
MailBack(mesg, fd)
	char *	mesg;
	FILE *	fd;
{
	VarArgs	va;
	char *	cp;

#	ifdef	SERVERUSER
	SenderName = SERVERUSER;
#	else
	SenderName = "FileServer";
#	endif
	GetFSHdr(fd);

	FIRSTARG(&va) = BINMAIL;

#	ifdef	BINMAILARGS
	if ( HomeAddress(SourceNode, (char **)0, INCL_HIER) )
		ExpandArgs(&va, BINMAILARGS, NULLSTR);
#	endif

	NEXTARG(&va) = RemUser;

	SetUser(ACSNETUID, ACSNETGID);

	fd = ExecPipe(&va);

	Fprintf(fd, "From %s %sFrom: %s@%s\nTo: %s@%s\n",
	    SenderName, ctime(&Time), SenderName, SourceNode, RemUser, Home_Address);

	Fprintf(fd, "Subject: Your request for files from %s\n\n", SourceNode);

	Fprintf(fd, "Some or all of the files you requested from %s\n", SourceNode);

	Fprintf(fd, "are not available.  ");

	if ( mesg != NULLSTR )
	{
		if ( (cp = StripErrString(mesg)) != NULLSTR )
		{
			Fprintf(fd, "The reason given was:\n\n\t%s\n", cp);
			free(cp);
		}
		free(mesg);
	}
	else
		Fprintf(fd, "No reason was given.\n");
	
	(void)ExPipeClose(fd);
}



#ifdef	FILESERVERLOG
/*
**	Write out stats record in the same format as that used in STATSFILE.
*/

void
writestats(alist)
	char **			alist;
{
	register char **	list = alist;
	register FILE *		fd;
	static bool		statswritten = false;

	Trace3(1, "writestats(%s)%s", list[0], statswritten?" {written}":"");

	if ( statswritten )
		return;

	if
	(
		access(FileServerLog, 0) == SYSERROR
		||
		(fd = fopen(FileServerLog, "a")) == NULL
	)
	{
		Trace3(1, "Can't open %s, errno %d", FileServerLog, errno);
		return;
	}

#	if	AUTO_LOCKING != 1
	if ( Lock(FileServerLog, fileno(fd), for_writing) == SYSERROR )
	{
		(void)fclose(fd);
		return;
	}
#	endif	AUTO_LOCKING != 1

	(void)fseek(fd, 0L, 2);

	Fprintf
	(
		fd,
		"%c%c%ld%c%ld%c%s@%s%c%s%c",
		ST_F_S_LOG, ST_RE_SEP,
		Time, ST_RE_SEP,
		FileSizes, ST_RE_SEP,
		RemUser, SourceNode, ST_RE_SEP,
		Service, ST_RE_SEP
	);

	while ( *list != NULLSTR )
	{
		if ( list != alist )
			(void)putc(',', fd);
		Fprintf(fd, *list++);
	}

	(void)putc(ST_SEP, fd);

#	if	AUTO_LOCKING != 1
	(void)fflush(fd);
	UnLock(fileno(fd));
#	endif	AUTO_LOCKING != 1

	(void)fclose(fd);

	statswritten = true;
}



/*
**	Capture error for log
*/

#if VARARGS
void
DoError(va_alist)
	va_dcl
{
	static char	errbuf[256];
	register char	*fmt;
	va_list		va;

	va_start(va);
	fmt = va_arg(va, char *);

	errbuf[0] = ' ';

	(void)vsprintf(&errbuf[1], fmt, va);

	Errorstr = errbuf;	/* Just remember last one */

	Error(errbuf);
	va_end(va);
}
#else
/*VARARGS1*/
void
DoError(f, a1, a2)
	char *	f;
	char *	a1;
	char *	a2;
{
	static char	errbuf[256];

	errbuf[0] = ' ';

	(void)sprintf(&errbuf[1], f, a1, a2);

	Errorstr = errbuf;	/* Just remember last one */

	Error(errbuf);
}
#endif
#endif	FILESERVERLOG
