/*
**	Copyright (c) 1984 Piers Lauder, University of Sydney
**
**	Warning: Distribution of this software without written
**		 permission is prohibited.
*/

static char	sccsid[]	= "@(#)state.c	1.34 88/09/05";

/*
**	Manage state and routing files.
*/

char *	Usage	= "[-[A][B][C][D][E][I][M][O][P][Q][R][S][T[n]][V][W][X][Y][Z]] \\\n\
	[-c[<commandsfile>]] [-f<sourcenode>] [-h<homename>] [-r<routefile>] \\\n\
	[-s<statefile>] [-t<statsfile>] [node|alias|domain ...]";

#define	FILE_CONTROL
#define	LOCKING
#define	RECOVER
#define	STDIO

#include	"global.h"
#include	"address.h"
#include	"debug.h"
#include	"header.h"
#include	"state.h"
#include	"statefile.h"
#include	"sun4state.h"
#include	"stats.h"


/*
**	Parameters set from arguments.
*/

bool	Commands;			/* Read commandsfile after statefile */
bool	CmdsFromStdin;			/* Read commands from stdin */
bool	DoStats;			/* Update statistics in statefile */
Export_t ExportType = t_export;		/* State info type */
char *	HomeNode;			/* Name of Home node */
bool	IgnoreCRC;			/* Ignore CRC in statefile */
char *	Name;				/* Program invoked name */
bool	NoCheck;			/* Don't check the statefile */
bool	Print_Alias;			/* Print aliases */
bool	Print_Domains;			/* Print domains */
bool	Print_Route;			/* Print routing tables */
bool	PrintMap;			/* Print network map */
Time_t	RemoveOlder;			/* Remove nodes older than this */
bool	ShowRoute;			/* Show route home for selected nodes */
Time_t	SourceDate;			/* Date of message at source */
char *	SourceNode;			/* Source for imported statefile */
bool	StFromStdin;			/* Read statefile from stdin */
bool	StToStdout;			/* Write statefile to stdout */
int	Traceflag;			/* Trace level */
bool	UpdateRoute;			/* Update routing tables */
bool	UpdateState;			/* Update state file */
bool	Verbose;			/* For lots of verbiage on print-outs */
bool	Warnings;			/* For warnings rather than termination on errors */
bool	ZeroStats;			/* Clear out all statistics */

/*
**	Miscellaneous definitions.
*/

bool	AnteState;			/* True if statefile on ``stdin'' is out-of-date */
bool	ChangeState;			/* True if statefile should be propagated */
char	Id;				/* ID of current statistics record */
bool	NeedState;			/* TRUE if statefile should be read */
int	Pid;				/* Process id */
FILE *	StateFd;			/* File descriptor for statefile */
Time_t	Time;				/* Time of invocation */
bool	MoveStats;			/* Update statistics if DoStats */

void	Mesg(), state(), usage();

bool	dostat();

extern char	_sobuf[];
extern bool	PrintInDomain;

#define	CHECKSTATE	((StToStdout||UpdateRoute||UpdateState||ZeroStats||Warnings)&&!NoCheck)
#define	Fprintf		(void)fprintf



int
main(argc, argv)
	int		argc;
	register char *	argv[];
{
	bool		flags		= false;

	if ( (Name = strrchr(*argv, '/')) != NULLSTR )
		Name++;
	else
		Name = *argv;

	(void)umask(022);
	Pid = getpid();
	Time = SourceDate = time((long *)0);

	while ( --argc > 0 )
	{
		if ( **++argv == '-' )
		{
			register int	c;

			flags = true;

			while ( c = *++*argv )
			{
				switch ( c )
				{
				case 'A':
					Print_Alias = true;
					continue;

				case 'B':
					PrintInDomain = true;
					continue;

				case 'C':
					CmdsFromStdin = true;
					NeedState = true;
					continue;

				case 'D':
					Print_Domains = true;
					continue;

				case 'E':
					IgnoreCRC = true;
					continue;

				case 'I':
					StFromStdin = true;
					NeedState = true;
					continue;

				case 'M':
					PrintMap = true;
					continue;

				case 'N':
					NoCheck = true;
					continue;

				case 'O':
					StToStdout = true;
					NeedState = true;
					continue;

				case 'P':
					Print_Route = true;
					continue;

				case 'Q':
					ExportType = t_allexport;
					StToStdout = true;
					NeedState = true;
					continue;

				case 'R':
					UpdateRoute = true;
					NeedState = true;
					continue;

				case 'S':
					UpdateState = true;
					NeedState = true;
					continue;

				case 'T':
					if ( (Traceflag = atoi(++*argv)) <= 0 )
						Traceflag = 1;
					break;

				case 'V':
					Verbose = true;
					PrintMap = true;
					continue;

				case 'W':
					Warnings = true;
					continue;

				case 'X':
					MoveStats = true;
					DoStats = true;
					continue;

				case 'Y':
					ShowRoute = true;
					PrintMap = true;
					continue;

				case 'Z':
					ZeroStats = true;
					NeedState = true;
					continue;

				case 'c':
					if ( *++*argv != '\0' )
						SetCommands(*argv);
					Commands = true;
					NeedState = true;
					goto break2;

				case 'd':
					SourceDate = atol(++*argv);
					goto break2;

				case 'f':
					SourceNode = ++*argv;
					goto break2;

				case 'h':
					HomeNode = ++*argv;
					goto break2;

				case 'o':
					RemoveOlder = Time - atol(++*argv) * 60*60*24;
					if ( RemoveOlder == Time )
						RemoveOlder = Time - 60*60*24*365;	/* Default -- 1 year */
					NeedState = true;
					goto break2;

				case 'r':
					SetRoute(++*argv);
					goto break2;

				case 's':
					SetState(++*argv);
					goto break2;

				case 't':
					NewStats(++*argv);
					DoStats = true;
					goto break2;

				default:
					usage("unrecognised flag '%c'", c);
					exit(1);
				}

				while ( (c = **argv) <= '9' && c >= '0' )
					++*argv;
				--*argv;
			}

break2:			;
		}
		else
		{
			PrintOnly(*argv);
			PrintMap = true;
		}
	}

	if ( flags == false )
		PrintMap = true;

	if ( HomeNode == NULLSTR )
		HomeNode = NodeName();

	setbuf(stdout, _sobuf);

	state();

	exit(0);
}



/*
**	Read in state file and check it, or read route file,
**	print and/or update statefile, and/or write routefile.
*/

void
state()
{
	if ( DoStats )
	{
		RdStats(&Id, dostat, MoveStats);
		InitRoute();
	}

	if ( PrintMap || NeedState )
	{
		if ( Warnings || StToStdout)
			Recover(ert_return);

		StateFd = ReadState(UpdateState?for_writing:for_reading, IgnoreCRC);

		DODEBUG(TraceNode(HomeNode));

		if ( StFromStdin && SourceNode != NULLSTR )
		{
			register int	c;

			(void)ungetc(c = getc(stdin), stdin);	/* Forward compatibility with SUN IV */

			if ( c & TOKEN_C )
				R4state(stdin, true, SourceNode, SourceDate, IgnoreCRC);
			else
				Rstate(stdin, true, SourceNode, SourceDate, IgnoreCRC);

			if ( AnteState )
			{
				while ( getc(stdin) != EOF );	/* Flush input */
				CmdsFromStdin = false;
				UpdateState = false;
				UpdateRoute = false;

				UnLock(fileno(StateFd));
				(void)fclose(StateFd);
			}
		}

		DODEBUG(TraceNode(HomeNode));
		
		if ( Commands || CmdsFromStdin )
		{
			if ( CmdsFromStdin )
				Rcommands(stdin, Warnings);
			
			if ( Commands )
				RdStCommands(Warnings);
		}

		Recover(ert_finish);

		if ( CHECKSTATE )
		do
		{
			while ( CheckState(ZeroStats, RemoveOlder) )
				;
		}
		while
			( CheckConnectivity() );

		DODEBUG(TraceNode(HomeNode));
	}

	if ( UpdateState )
	{
		WriteState(StateFd, t_local);
		UnLock(fileno(StateFd));
		(void)fclose(StateFd);
	}

	DODEBUG(TraceNode(HomeNode));

	if ( UpdateRoute )
	{
		WriteRoute();

		if ( ChangeState && !StToStdout )
		{
			VarArgs	va;
			char *	errs;

			FIRSTARG(&va) = REQUESTER;
			NEXTARG(&va) = "-BLN";

			if ( Traceflag )
			{
				Trace1(1, "ChangeState true");
			}
			else
			if ( (errs = Execute(&va)) != NULLSTR )
			{
				Warn(errs);
				free(errs);
			}
		}
	}

	DODEBUG(TraceNode(HomeNode));

	if ( StToStdout )
		WriteState(stdout, ExportType);
	
	if ( PrintMap )
		PrintState(stdout, SetPrint(), Verbose, ShowRoute);
	
	if ( Print_Route )
		PrintRoute(stdout);

	if ( Print_Domains )
		PrintDomains(stdout);

	if ( Print_Alias )
		PrintAlias(stdout);
}



/*
**	Cleanup for error routines
*/

void
finish(error)
	int	error;
{
	if ( StateFd != NULL )
		UnLock(fileno(StateFd));
	(void)exit(error);
}



/*
**	Explain usage
*/

/*VARARGS1*/
void
usage(s, a1)
	char *	s;
	char *	a1;
{
	Mesg("bad arguments", s, a1);
	Fprintf(stderr, "\nUsage is \"%s %s\"\n", Name, Usage);
}



#if	NODE_STATS == 1 || LINK_STATS == 1

/*
**	Process statistics field from record
*/

static bool	AtHomeVal;
static NodeLink	Nl;
static int	count;
static char *	dest;
static char **	from;
static bool	nodedone;
static char *	link;
static char *	route;
static char *	routelink;
static long	size;
static char *	source;
static Time_t	tim;
static char **	to;
static ulong	tt;

bool		dostat1(), dostat2(), dostat3();
static bool	AtHome();

#endif	NODE_STATS == 1 || LINK_STATS == 1

bool
dostat(cp, index)
	char *	cp;
	int	index;
{
	Trace3(3, "dostat \"%s\" %d", cp, index);

#	if	NODE_STATS == 1 || LINK_STATS == 1
	if ( index == 0 )
	{
		switch ( Id )
		{
		case ST_RETMESG:
			to = &source;
			from = &dest;
			break;

		case ST_MESSAGE:
		case ST_INMESG:
		case ST_OUTMESG:
			from = &source;
			to = &dest;
			break;

		default:
			return false;
		}

		link = NULLSTR;
	}

	switch ( (StMesg_t)index )
	{
	case sm_time:
		if ( (tim = (Time_t)atol(cp)) > Time )
			tim = Time;	/* Catch bad date changes */
		break;

	case sm_size:
		size = atol(cp);
		break;

	case sm_dest:
		dest = cp;
		break;

	case sm_source:
		source = cp;
		break;

	case sm_route:
		route = cp;
		break;

	case sm_tt:
		tt = (ulong)atol(cp);
		break;

	case sm_link:
		if ( *cp != '\0' )
			link = cp;
		break;
	}

	if ( (StMesg_t)index == sm_last )
	{
		if ( !(AtHomeVal = AtHome(*from)) && !FindNode(*from, pt_msg, &Nl) )
			from = &link;

		Trace7
		(
			2,
			"dostat tim %lu siz %ld dest %s src %s route %s tt %lu",
			tim, size, dest, source, route, tt
		);

		count = 0;
		tim -= tt;
		nodedone = false;

		if ( routelink != NULLSTR )
		{
			free(routelink);
			routelink = NULLSTR;
		}

		switch ( Id )
		{
		case ST_MESSAGE:
			(void)ExRoute(route, dostat2);
			if ( link == NULLSTR )
				link = routelink;
			(void)ExDests(*to, dostat1);
			break;

		case ST_INMESG:
		case ST_RETMESG:
			(void)ExRoute(route, dostat2);
			if ( link == NULLSTR )
				link = routelink;
			(void)StatsFrom(tim, size, *from, AtHome(*to));
			break;

		case ST_OUTMESG:
			(void)ExDests(*to, dostat3);
			break;
		}
	}

	return true;
#	endif	NODE_STATS == 1 || LINK_STATS == 1
}



#if	NODE_STATS == 1 || LINK_STATS == 1

bool
dostat1(dest)
	char *	dest;
{
	if ( AtHome(dest) )
	{
		if ( nodedone )
			return true;
		nodedone = true;
	}

	if ( count++ == 0 )
		return Stats(st_from, tim, size, *from, dest);
	else
		return Stats(st_to, tim, size, *from, dest);
}



bool
dostat2(tt, dest1, dest2)
	ulong	tt;
	char *	dest1;
	char *	dest2;
{
	if ( link == NULLSTR )
	{
		if ( routelink != NULLSTR )
			free(routelink);
		routelink = newstr(dest1);
	}

#	if	LINK_STATS == 1
	if ( FindNode(dest, pt_msg, &Nl) )
		(void)Stats(st_link, tt, size, dest1, dest2);
#	endif	LINK_STATS == 1

	return true;
}



bool
dostat3(dest)
	char *	dest;
{
	if ( strccmp(dest, link) == STREQUAL || !FindNode(dest, pt_msg, &Nl) )
	{
		if ( nodedone )
			return true;
		dest = link;
		nodedone = true;
	}

	return StatsTo(tim, size, dest, AtHomeVal);
}



/*
**	Return 'true' if 'HomeNode' is in destination list.
*/

static bool
AtHome(s)
	register char *	s;
{
	register char *	a;
	register int	c;
	char *		handler;	/* Not used */

	if ( s == NULLSTR )
		return false;

	switch ( c = *s++ )
	{
	case '\0':
		return false;

	case ATYP_EXPLICIT:
		if ( (s = strrchr(s, c)) == NULLSTR )
			return false;

		return AtHome(s+1);

	case ATYP_GROUP:
	case ATYP_MULTICAST:
		for ( ;; )
		{
			if ( (a = strchr(s, c)) != NULLSTR )
				*a = '\0';
	
			if ( AtHome(s) )
			{
				if ( a != NULLSTR )
					*a = c;
				return true;
			}

			if ( a != NULLSTR )
			{
				*a++ = c;
				s = a;
			}
			else
				return false;
		}

	case ATYP_BROADCAST:
		if
		(
			s[0] == '\0'
			||
			(
				s[0] == DOMAIN_SEP
				&&
				(
					(
						s[1] == ATYP_BROADCAST
						&&
						s[2] == '\0'
					)
					||
					HomeDomain(&s[1], &handler, EXCL_HIER)
				)
			)
		)
			return true;

		return false;
	}
	
	if ( HomeAddress(--s, &handler, INCL_HIER) )
		return true;

	return false;
}

#endif	NODE_STATS == 1 || LINK_STATS == 1
